/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.wiki.engine.jspwiki.internal;

import com.ecyrd.jspwiki.QueryItem;
import com.ecyrd.jspwiki.WikiEngine;
import com.ecyrd.jspwiki.WikiPage;
import com.ecyrd.jspwiki.providers.ProviderException;
import com.ecyrd.jspwiki.providers.WikiPageProvider;

import com.liferay.portal.kernel.exception.SystemException;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.StringBundler;
import com.liferay.wiki.exception.NoSuchPageException;
import com.liferay.wiki.service.WikiPageLocalServiceUtil;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Properties;

/**
 * @author Jorge Ferrer
 */
public class LiferayPageProvider implements WikiPageProvider {

	public static WikiPage toJSPWikiPage(
		com.liferay.wiki.model.WikiPage page, WikiEngine engine) {

		WikiPage jspWikiPage = new WikiPage(engine, page.getTitle());

		jspWikiPage.setAuthor(page.getUserName());
		jspWikiPage.setVersion((int)(page.getVersion() * 10));
		jspWikiPage.setLastModified(page.getCreateDate());

		return jspWikiPage;
	}

	@Override
	public void deletePage(String name) {
		if (_log.isDebugEnabled()) {
			_log.debug("Invoking deletePage(" + name + ")");
		}
	}

	@Override
	public void deleteVersion(String title, int version) {
		if (_log.isDebugEnabled()) {
			_log.debug(
				StringBundler.concat(
					"Invoking deleteVersion(", title, ", ",
					String.valueOf(version), ")"));
		}
	}

	@Override
	public Collection<WikiPage> findPages(QueryItem[] query) {
		if (_log.isDebugEnabled()) {
			_log.debug("Invoking findPages(" + Arrays.toString(query) + ")");
		}

		return Collections.emptyList();
	}

	@Override
	public Collection<WikiPage> getAllChangedSince(Date date) {
		if (_log.isDebugEnabled()) {
			_log.debug("Invoking getAllChangedSince(" + date + ")");
		}

		try {
			return getAllPages();
		}
		catch (ProviderException pe) {
			_log.error("Could not get changed pages", pe);

			return Collections.emptyList();
		}
	}

	@Override
	public Collection<WikiPage> getAllPages() throws ProviderException {
		if (_log.isDebugEnabled()) {
			_log.debug("Invoking getAllPages()");
		}

		List<WikiPage> jspWikiPages = new ArrayList<>();

		try {
			int count = WikiPageLocalServiceUtil.getPagesCount(_nodeId, true);

			List<com.liferay.wiki.model.WikiPage> pages =
				WikiPageLocalServiceUtil.getPages(_nodeId, true, 0, count);

			for (com.liferay.wiki.model.WikiPage page : pages) {
				jspWikiPages.add(toJSPWikiPage(page, _engine));
			}
		}
		catch (SystemException se) {
			throw new ProviderException(se.toString());
		}

		return jspWikiPages;
	}

	@Override
	public int getPageCount() throws ProviderException {
		if (_log.isDebugEnabled()) {
			_log.debug("Invoking getPageCount()");
		}

		try {
			return WikiPageLocalServiceUtil.getPagesCount(_nodeId);
		}
		catch (SystemException se) {
			throw new ProviderException(se.toString());
		}
	}

	@Override
	public WikiPage getPageInfo(String title, int version)
		throws ProviderException {

		if (_log.isDebugEnabled()) {
			_log.debug(
				StringBundler.concat(
					"Invoking getPageInfo(", title, ", ",
					String.valueOf(version), ")"));
		}

		try {
			com.liferay.wiki.model.WikiPage page =
				WikiPageLocalServiceUtil.getPage(_nodeId, title);

			return toJSPWikiPage(page, _engine);
		}
		catch (NoSuchPageException nspe) {
			return null;
		}
		catch (Exception e) {
			throw new ProviderException(e.toString());
		}
	}

	@Override
	public String getPageText(String title, int version)
		throws ProviderException {

		if (_log.isDebugEnabled()) {
			_log.debug(
				StringBundler.concat(
					"Invoking getPageText(", title, ", ",
					String.valueOf(version), ")"));
		}

		try {
			com.liferay.wiki.model.WikiPage page =
				WikiPageLocalServiceUtil.getPage(_nodeId, title);

			return page.getContent();
		}
		catch (Exception e) {
			throw new ProviderException(e.toString());
		}
	}

	@Override
	public String getProviderInfo() {
		if (_log.isDebugEnabled()) {
			_log.debug("Invoking getProviderInfo()");
		}

		return LiferayPageProvider.class.getName();
	}

	@Override
	public List<WikiPage> getVersionHistory(String title) {
		if (_log.isDebugEnabled()) {
			_log.debug("Invoking getVersionHistory(" + title + ")");
		}

		return Collections.emptyList();
	}

	@Override
	public void initialize(WikiEngine engine, Properties props) {
		if (_log.isDebugEnabled()) {
			_log.debug(
				StringBundler.concat(
					"Invoking initialize(", String.valueOf(engine), ", ",
					String.valueOf(props), ")"));
		}

		_engine = engine;
		_nodeId = GetterUtil.getLong(props.getProperty("nodeId"));
	}

	@Override
	public void movePage(String from, String to) {
		if (_log.isDebugEnabled()) {
			_log.debug(
				StringBundler.concat(
					"Invoking movePage(", from, ", ", to, ")"));
		}
	}

	@Override
	public boolean pageExists(String title) {
		if (_log.isDebugEnabled()) {
			_log.debug("Invoking pageExists(" + title + ")");
		}

		try {
			int count = WikiPageLocalServiceUtil.getPagesCount(
				_nodeId, JSPWikiEngine.decodeJSPWikiName(title), true);

			if (count > 0) {
				return true;
			}

			return false;
		}
		catch (Exception e) {
			_log.error(e, e);
		}

		return false;
	}

	@Override
	public void putPageText(WikiPage page, String text) {
		if (_log.isDebugEnabled()) {
			_log.debug(
				StringBundler.concat(
					"Invoking putPageText(", String.valueOf(page), ", ", text,
					")"));
		}
	}

	private static final Log _log = LogFactoryUtil.getLog(
		LiferayPageProvider.class);

	private WikiEngine _engine;
	private long _nodeId;

}