/**
 * SPDX-FileCopyrightText: (c) 2025 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.site.cms.site.initializer.internal.display.context;

import com.liferay.layout.constants.LayoutTypeSettingsConstants;
import com.liferay.layout.page.template.model.LayoutPageTemplateEntry;
import com.liferay.layout.page.template.service.LayoutPageTemplateEntryLocalServiceUtil;
import com.liferay.object.admin.rest.dto.v1_0.ObjectDefinition;
import com.liferay.object.admin.rest.dto.v1_0.util.ObjectDefinitionUtil;
import com.liferay.object.admin.rest.resource.v1_0.ObjectDefinitionResource;
import com.liferay.object.constants.ObjectFolderConstants;
import com.liferay.object.service.ObjectRelationshipLocalServiceUtil;
import com.liferay.petra.string.StringBundler;
import com.liferay.portal.kernel.json.JSONFactory;
import com.liferay.portal.kernel.json.JSONObject;
import com.liferay.portal.kernel.json.JSONUtil;
import com.liferay.portal.kernel.model.Group;
import com.liferay.portal.kernel.model.GroupConstants;
import com.liferay.portal.kernel.model.Layout;
import com.liferay.portal.kernel.service.GroupLocalServiceUtil;
import com.liferay.portal.kernel.service.LayoutLocalServiceUtil;
import com.liferay.portal.kernel.theme.ThemeDisplay;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.HashMapBuilder;
import com.liferay.portal.kernel.util.ListUtil;
import com.liferay.portal.kernel.util.ParamUtil;
import com.liferay.portal.kernel.util.PortalUtil;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.kernel.util.WebKeys;
import com.liferay.portal.vulcan.pagination.Page;

import jakarta.servlet.http.HttpServletRequest;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * @author Eudaldo Alonso
 */
public class StructureBuilderDisplayContext {

	public StructureBuilderDisplayContext(
		HttpServletRequest httpServletRequest, JSONFactory jsonFactory,
		ObjectDefinitionResource.Factory objectDefinitionResourceFactory) {

		_httpServletRequest = httpServletRequest;
		_jsonFactory = jsonFactory;
		_objectDefinitionResourceFactory = objectDefinitionResourceFactory;

		_themeDisplay = (ThemeDisplay)httpServletRequest.getAttribute(
			WebKeys.THEME_DISPLAY);
	}

	public Map<String, Object> getProps() throws Exception {
		return HashMapBuilder.<String, Object>put(
			"config",
			JSONUtil.put(
				"autogeneratedDisplayPage",
				() -> {
					ObjectDefinition objectDefinition = _getObjectDefinition();

					if ((objectDefinition == null) ||
						!GetterUtil.getBoolean(_objectDefinition.getActive())) {

						return true;
					}

					Group group = GroupLocalServiceUtil.getGroup(
						_themeDisplay.getCompanyId(), GroupConstants.CMS);

					LayoutPageTemplateEntry layoutPageTemplateEntry =
						LayoutPageTemplateEntryLocalServiceUtil.
							fetchDefaultLayoutPageTemplateEntry(
								group.getGroupId(),
								PortalUtil.getClassNameId(
									objectDefinition.getClassName()),
								0);

					if (layoutPageTemplateEntry == null) {
						return true;
					}

					Layout draftLayout =
						LayoutLocalServiceUtil.fetchDraftLayout(
							layoutPageTemplateEntry.getPlid());

					if ((draftLayout == null) ||
						!GetterUtil.getBoolean(
							draftLayout.getTypeSettingsProperty(
								LayoutTypeSettingsConstants.
									KEY_AUTOGENERATED))) {

						return false;
					}

					return true;
				}
			).put(
				"editStructureDisplayPageURL",
				() -> StringBundler.concat(
					_themeDisplay.getPortalURL(), _themeDisplay.getPathMain(),
					"/cms/edit_structure_display_page")
			).put(
				"isReferenced",
				() -> {
					ObjectDefinition objectDefinition = _getObjectDefinition();

					if ((objectDefinition == null) ||
						!GetterUtil.getBoolean(_objectDefinition.getActive())) {

						return false;
					}

					return ListUtil.isNotEmpty(
						ObjectRelationshipLocalServiceUtil.
							getObjectRelationshipsByObjectDefinitionId2(
								objectDefinition.getId()));
				}
			).put(
				"objectFolderExternalReferenceCode",
				_getObjectFolderExternalReferenceCode()
			).put(
				"resetStructureDisplayPageURL",
				() -> StringBundler.concat(
					_themeDisplay.getPortalURL(), _themeDisplay.getPathMain(),
					"/cms/reset_structure_display_page")
			).put(
				"structureBuilderURL",
				() -> PortalUtil.getLayoutFullURL(
					LayoutLocalServiceUtil.getLayoutByFriendlyURL(
						_themeDisplay.getScopeGroupId(), false,
						"/structure-builder"),
					_themeDisplay)
			)
		).put(
			"state",
			JSONUtil.put(
				"mainObjectDefinition",
				_getObjectDefinitionJSONObject(_getObjectDefinition())
			).put(
				"objectDefinitions", _getObjectDefinitionsJSONObject()
			)
		).build();
	}

	private ObjectDefinition _getObjectDefinition() throws Exception {
		if (_objectDefinition != null) {
			return _objectDefinition;
		}

		long objectDefinitionId = ParamUtil.getLong(
			_httpServletRequest, "objectDefinitionId");

		if (objectDefinitionId <= 0) {
			return null;
		}

		ObjectDefinitionResource.Builder builder =
			_objectDefinitionResourceFactory.create();

		ObjectDefinitionResource objectDefinitionResource = builder.user(
			_themeDisplay.getUser()
		).build();

		_objectDefinition = objectDefinitionResource.getObjectDefinition(
			objectDefinitionId);

		return _objectDefinition;
	}

	private JSONObject _getObjectDefinitionJSONObject(
			ObjectDefinition objectDefinition)
		throws Exception {

		if (objectDefinition == null) {
			return null;
		}

		ObjectDefinitionUtil.prepareObjectDefinitionForExport(
			_jsonFactory, objectDefinition);

		return _jsonFactory.createJSONObject(objectDefinition.toString());
	}

	private List<ObjectDefinition> _getObjectDefinitions() throws Exception {
		if (_objectDefinitions != null) {
			return _objectDefinitions;
		}

		ObjectDefinitionResource.Builder builder =
			_objectDefinitionResourceFactory.create();

		ObjectDefinitionResource objectDefinitionResource = builder.user(
			_themeDisplay.getUser()
		).build();

		Page<ObjectDefinition> page =
			objectDefinitionResource.getObjectDefinitionsPage(
				null, null,
				objectDefinitionResource.toFilter(
					StringBundler.concat(
						"(objectFolderExternalReferenceCode eq '",
						ObjectFolderConstants.
							EXTERNAL_REFERENCE_CODE_CONTENT_STRUCTURES,
						"') or (objectFolderExternalReferenceCode eq '",
						ObjectFolderConstants.
							EXTERNAL_REFERENCE_CODE_FILE_TYPES,
						"')"),
					Collections.emptyMap()),
				null, null);

		_objectDefinitions = new ArrayList<>(page.getItems());

		return _objectDefinitions;
	}

	private JSONObject _getObjectDefinitionsJSONObject() throws Exception {
		List<ObjectDefinition> objectDefinitions = _getObjectDefinitions();

		if (objectDefinitions == null) {
			return null;
		}

		JSONObject jsonObject = _jsonFactory.createJSONObject();

		for (ObjectDefinition objectDefinition : objectDefinitions) {
			JSONObject objectDefinitionJSONObject =
				_getObjectDefinitionJSONObject(objectDefinition);

			if (objectDefinitionJSONObject != null) {
				jsonObject.put(
					objectDefinitionJSONObject.getString(
						"externalReferenceCode"),
					objectDefinitionJSONObject);
			}
		}

		return jsonObject;
	}

	private String _getObjectFolderExternalReferenceCode() throws Exception {
		if (_objectFolderExternalReferenceCode != null) {
			return _objectFolderExternalReferenceCode;
		}

		_objectFolderExternalReferenceCode = ParamUtil.getString(
			_httpServletRequest, "objectFolderExternalReferenceCode");

		if (Validator.isNotNull(_objectFolderExternalReferenceCode)) {
			return _objectFolderExternalReferenceCode;
		}

		ObjectDefinition objectDefinition = _getObjectDefinition();

		if (objectDefinition != null) {
			_objectFolderExternalReferenceCode =
				objectDefinition.getObjectFolderExternalReferenceCode();
		}

		return _objectFolderExternalReferenceCode;
	}

	private final HttpServletRequest _httpServletRequest;
	private final JSONFactory _jsonFactory;
	private ObjectDefinition _objectDefinition;
	private final ObjectDefinitionResource.Factory
		_objectDefinitionResourceFactory;
	private List<ObjectDefinition> _objectDefinitions;
	private String _objectFolderExternalReferenceCode;
	private final ThemeDisplay _themeDisplay;

}