/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.segments.internal.odata;

import com.fasterxml.jackson.databind.util.ISO8601Utils;

import com.liferay.petra.string.StringBundler;
import com.liferay.portal.kernel.json.JSONArray;
import com.liferay.portal.kernel.json.JSONFactoryUtil;
import com.liferay.portal.kernel.json.JSONObject;
import com.liferay.portal.kernel.json.JSONUtil;
import com.liferay.portal.kernel.util.LocaleUtil;
import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.odata.entity.CollectionEntityField;
import com.liferay.portal.odata.entity.ComplexEntityField;
import com.liferay.portal.odata.entity.EntityField;
import com.liferay.portal.odata.entity.EntityModel;
import com.liferay.portal.odata.filter.expression.BinaryExpression;
import com.liferay.portal.odata.filter.expression.CollectionPropertyExpression;
import com.liferay.portal.odata.filter.expression.ComplexPropertyExpression;
import com.liferay.portal.odata.filter.expression.Expression;
import com.liferay.portal.odata.filter.expression.ExpressionVisitException;
import com.liferay.portal.odata.filter.expression.ExpressionVisitor;
import com.liferay.portal.odata.filter.expression.LambdaFunctionExpression;
import com.liferay.portal.odata.filter.expression.LambdaVariableExpression;
import com.liferay.portal.odata.filter.expression.ListExpression;
import com.liferay.portal.odata.filter.expression.LiteralExpression;
import com.liferay.portal.odata.filter.expression.MemberExpression;
import com.liferay.portal.odata.filter.expression.MethodExpression;
import com.liferay.portal.odata.filter.expression.PrimitivePropertyExpression;
import com.liferay.portal.odata.filter.expression.PropertyExpression;
import com.liferay.portal.odata.filter.expression.UnaryExpression;

import java.text.ParseException;
import java.text.ParsePosition;

import java.time.Duration;
import java.time.Instant;
import java.time.format.DateTimeParseException;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author Cristina González
 */
public class ExpressionVisitorImpl implements ExpressionVisitor<Object> {

	public ExpressionVisitorImpl(int groupCount, EntityModel entityModel) {
		_groupCount = groupCount;
		_entityModel = entityModel;
	}

	@Override
	public Object visitBinaryExpressionOperation(
			BinaryExpression.Operation operation, Object left, Object right)
		throws ExpressionVisitException {

		if (Objects.equals(BinaryExpression.Operation.AND, operation) ||
			Objects.equals(BinaryExpression.Operation.OR, operation)) {

			return _getConjunctionJSONObject(
				operation, (JSONObject)left, (JSONObject)right);
		}
		else if (Objects.equals(BinaryExpression.Operation.EQ, operation) ||
				 Objects.equals(BinaryExpression.Operation.GE, operation) ||
				 Objects.equals(BinaryExpression.Operation.GT, operation) ||
				 Objects.equals(BinaryExpression.Operation.LE, operation) ||
				 Objects.equals(BinaryExpression.Operation.LT, operation) ||
				 Objects.equals(BinaryExpression.Operation.NE, operation)) {

			return _getOperationJSONObject(
				String.valueOf(operation), left, right);
		}
		else if (Objects.equals(BinaryExpression.Operation.SUB, operation)) {
			return _sub(left, right);
		}

		throw new UnsupportedOperationException(
			"Unsupported method visitBinaryExpressionOperation with " +
				"operation " + operation);
	}

	@Override
	public Object visitCollectionPropertyExpression(
			CollectionPropertyExpression collectionPropertyExpression)
		throws ExpressionVisitException {

		LambdaFunctionExpression lambdaFunctionExpression =
			collectionPropertyExpression.getLambdaFunctionExpression();

		Map<String, EntityField> entityFieldsMap =
			_entityModel.getEntityFieldsMap();

		CollectionEntityField collectionEntityField =
			(CollectionEntityField)entityFieldsMap.get(
				collectionPropertyExpression.getName());

		return lambdaFunctionExpression.accept(
			new ExpressionVisitorImpl(
				0,
				new EntityModel() {

					@Override
					public Map<String, EntityField> getEntityFieldsMap() {
						return Collections.singletonMap(
							lambdaFunctionExpression.getVariableName(),
							collectionEntityField.getEntityField());
					}

					@Override
					public String getName() {
						return collectionEntityField.getName();
					}

				}));
	}

	public Object visitComplexPropertyExpression(
		ComplexPropertyExpression complexPropertyExpression) {

		Map<String, EntityField> entityFieldsMap =
			_entityModel.getEntityFieldsMap();

		ComplexEntityField complexEntityField =
			(ComplexEntityField)entityFieldsMap.get(
				complexPropertyExpression.getName());

		Map<String, EntityField> complexEntityFieldFieldsMap =
			complexEntityField.getEntityFieldsMap();

		PropertyExpression propertyExpression =
			complexPropertyExpression.getPropertyExpression();

		EntityField entityField = complexEntityFieldFieldsMap.get(
			propertyExpression.getName());

		return complexEntityField.getName() + "/" + entityField.getName();
	}

	@Override
	public Object visitLambdaFunctionExpression(
			LambdaFunctionExpression.Type type, String variable,
			Expression expression)
		throws ExpressionVisitException {

		if (type == LambdaFunctionExpression.Type.ANY) {
			return expression.accept(this);
		}

		throw new UnsupportedOperationException(
			"Unsupported type visitLambdaFunctionExpression with type " + type);
	}

	@Override
	public EntityField visitLambdaVariableExpression(
			LambdaVariableExpression lambdaVariableExpression)
		throws ExpressionVisitException {

		Map<String, EntityField> entityFieldsMap =
			_entityModel.getEntityFieldsMap();

		EntityField entityField = entityFieldsMap.get(
			lambdaVariableExpression.getVariableName());

		if (entityField == null) {
			throw new ExpressionVisitException(
				"Invoked visitLambdaVariableExpression when no entity field " +
					"is stored for lambda variable name " +
						lambdaVariableExpression.getVariableName());
		}

		return entityField;
	}

	@Override
	public Object visitListExpressionOperation(
			ListExpression.Operation operation, Object left, List<Object> right)
		throws ExpressionVisitException {

		if (operation == ListExpression.Operation.IN) {
			return _getOperationJSONObject(
				String.valueOf(operation), left, right);
		}

		throw new UnsupportedOperationException(
			"Unsupported method visitListExpressionOperation with operation " +
				operation);
	}

	@Override
	public Object visitLiteralExpression(LiteralExpression literalExpression)
		throws ExpressionVisitException {

		if (Objects.equals(
				LiteralExpression.Type.DURATION, literalExpression.getType())) {

			return _getDuration(literalExpression.getText());
		}

		return StringUtil.unquote(literalExpression.getText());
	}

	@Override
	public Object visitMemberExpression(MemberExpression memberExpression)
		throws ExpressionVisitException {

		Expression expression = memberExpression.getExpression();

		return expression.accept(this);
	}

	@Override
	public Object visitMethodExpression(
		List<Object> expressions, MethodExpression.Type type) {

		if (type == MethodExpression.Type.CONTAINS) {
			if (expressions.size() != 2) {
				throw new UnsupportedOperationException(
					StringBundler.concat(
						"Unsupported method visitMethodExpression with method ",
						"type ", type, " and ", expressions.size(), "params"));
			}

			return _getOperationJSONObject(
				String.valueOf(type), expressions.get(0), expressions.get(1));
		}
		else if (type == MethodExpression.Type.NOW) {
			if (!expressions.isEmpty()) {
				throw new UnsupportedOperationException(
					StringBundler.concat(
						"Unsupported method visitMethodExpression with method ",
						"type ", type, " and ", expressions.size(), "params"));
			}

			return MethodType.NOW;
		}

		throw new UnsupportedOperationException(
			"Unsupported method visitMethodExpression with method type " +
				type);
	}

	@Override
	public Object visitPrimitivePropertyExpression(
		PrimitivePropertyExpression primitivePropertyExpression) {

		Map<String, EntityField> entityFieldsMap =
			_entityModel.getEntityFieldsMap();

		return entityFieldsMap.get(primitivePropertyExpression.getName());
	}

	@Override
	public JSONObject visitUnaryExpressionOperation(
		UnaryExpression.Operation operation, Object operand) {

		if (Objects.equals(UnaryExpression.Operation.NOT, operation)) {
			JSONObject jsonObject = (JSONObject)operand;

			jsonObject.put(
				"operatorName",
				StringUtil.lowerCase(
					UnaryExpression.Operation.NOT + "-" +
						jsonObject.getString("operatorName")));

			return jsonObject;
		}

		throw new UnsupportedOperationException(
			"Unsupported method visitUnaryExpressionOperation with operation " +
				operation);
	}

	public enum MethodType {

		NOW

	}

	private JSONObject _getConjunctionJSONObject(
		BinaryExpression.Operation operation, JSONObject leftJSONObject,
		JSONObject rightJSONObject) {

		String conjunctionName = leftJSONObject.getString("conjunctionName");

		_groupCount++;

		if (Validator.isNotNull(conjunctionName)) {
			String operationString = operation.toString();

			if (Objects.equals(
					conjunctionName.toLowerCase(LocaleUtil.ROOT),
					operationString.toLowerCase(LocaleUtil.ROOT))) {

				return JSONUtil.put(
					"conjunctionName",
					StringUtil.lowerCase(String.valueOf(operation))
				).put(
					"groupId", "group_" + _groupCount
				).put(
					"items",
					leftJSONObject.getJSONArray(
						"items"
					).put(
						rightJSONObject
					)
				);
			}
		}

		return JSONUtil.put(
			"conjunctionName", StringUtil.lowerCase(String.valueOf(operation))
		).put(
			"groupId", "group_" + _groupCount
		).put(
			"items", JSONUtil.putAll(leftJSONObject, rightJSONObject)
		);
	}

	private Duration _getDuration(String literal)
		throws ExpressionVisitException {

		literal = StringUtil.unquote(
			StringUtil.removeSubstring(literal, "duration"));

		try {
			return Duration.parse(literal);
		}
		catch (DateTimeParseException dateTimeParseException) {
			throw new ExpressionVisitException(
				"Invalid duration: " + dateTimeParseException.getMessage());
		}
	}

	private JSONObject _getOperationJSONObject(
		String operatorName, Object object, List<Object> fieldValues) {

		JSONArray jsonArray = JSONFactoryUtil.createJSONArray();

		fieldValues.forEach(
			fieldValue -> jsonArray.put(String.valueOf(fieldValue)));

		return JSONUtil.put(
			"operatorName", StringUtil.lowerCase(operatorName)
		).put(
			"propertyName", _getPropertyName(object)
		).put(
			"value", jsonArray
		);
	}

	private JSONObject _getOperationJSONObject(
		String operatorName, Object object, Object fieldValue) {

		return JSONUtil.put(
			"operatorName", StringUtil.lowerCase(operatorName)
		).put(
			"propertyName", _getPropertyName(object)
		).put(
			"value", fieldValue
		);
	}

	private String _getPropertyName(Object object) {
		if (object instanceof EntityField) {
			EntityField entityField = (EntityField)object;

			return entityField.getName();
		}

		return String.valueOf(object);
	}

	private Object _sub(Object left, Object right)
		throws ExpressionVisitException {

		if ((left instanceof MethodType || left instanceof String) &&
			(right instanceof Duration)) {

			try {
				Date date = null;

				if (Objects.equals(MethodType.NOW, left)) {
					date = _date;
				}
				else {
					date = ISO8601Utils.parse(
						String.valueOf(left), new ParsePosition(0));
				}

				Duration duration = (Duration)right;

				Instant instant = date.toInstant();

				instant = instant.minusMillis(duration.toMillis());

				return ISO8601Utils.format(Date.from(instant));
			}
			catch (ParseException parseException) {
				throw new ExpressionVisitException(
					"Only a date with ISO 8601 format is supported as a left " +
						"operator " + parseException.getMessage());
			}
		}

		throw new UnsupportedOperationException(
			StringBundler.concat(
				"Unsupported types in _sub with arithmetic operator SUB with ",
				"left type ", left.getClass(), " and right type ",
				right.getClass()));
	}

	private final Date _date = new Date();
	private final EntityModel _entityModel;
	private int _groupCount;

}