/**
 * SPDX-FileCopyrightText: (c) 2025 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.portal.tools.rest.builder.internal.typescript;

import com.liferay.petra.string.StringBundler;
import com.liferay.petra.string.StringPool;
import com.liferay.portal.kernel.util.HashMapBuilder;
import com.liferay.portal.kernel.util.ListUtil;
import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.portal.kernel.util.StringUtil_IW;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.tools.rest.builder.internal.freemarker.tool.FreeMarkerTool;
import com.liferay.portal.tools.rest.builder.internal.freemarker.tool.java.JavaMethodSignature;
import com.liferay.portal.tools.rest.builder.internal.freemarker.tool.java.parser.ResourceOpenAPIParser;
import com.liferay.portal.tools.rest.builder.internal.freemarker.tool.java.parser.util.OpenAPIParserUtil;
import com.liferay.portal.tools.rest.builder.internal.freemarker.util.FreeMarkerUtil;
import com.liferay.portal.tools.rest.builder.internal.util.FileUtil;
import com.liferay.portal.tools.rest.builder.internal.yaml.config.Application;
import com.liferay.portal.tools.rest.builder.internal.yaml.config.ConfigYAML;
import com.liferay.portal.tools.rest.builder.internal.yaml.openapi.Components;
import com.liferay.portal.tools.rest.builder.internal.yaml.openapi.Content;
import com.liferay.portal.tools.rest.builder.internal.yaml.openapi.Discriminator;
import com.liferay.portal.tools.rest.builder.internal.yaml.openapi.Info;
import com.liferay.portal.tools.rest.builder.internal.yaml.openapi.Items;
import com.liferay.portal.tools.rest.builder.internal.yaml.openapi.OpenAPIYAML;
import com.liferay.portal.tools.rest.builder.internal.yaml.openapi.Operation;
import com.liferay.portal.tools.rest.builder.internal.yaml.openapi.Parameter;
import com.liferay.portal.tools.rest.builder.internal.yaml.openapi.PathItem;
import com.liferay.portal.tools.rest.builder.internal.yaml.openapi.RequestBody;
import com.liferay.portal.tools.rest.builder.internal.yaml.openapi.Response;
import com.liferay.portal.tools.rest.builder.internal.yaml.openapi.ResponseCode;
import com.liferay.portal.tools.rest.builder.internal.yaml.openapi.Schema;

import java.io.File;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * @author Daniel Raposo
 */
public class TypeScriptClientUtil {

	public static void generateTypeScriptClient(
			File configDir, ConfigYAML configYAML, File copyrightFile,
			String yamlString)
		throws Exception {

		File baseClientDir = new File(
			StringUtil.removeLast(configDir.getPath(), "-impl") + "-client-js");
		List<File> files = new ArrayList<>();

		_createBuildGradleFile(baseClientDir, files);
		_createNodeScriptsConfigFile(baseClientDir, files);
		_createPackageJSONFile(baseClientDir, files);

		OpenAPIYAML openAPIYAML = OpenAPIParserUtil.loadOpenAPIYAML(
			_addAutogeneratedVulcanSchemas(configYAML, yamlString));

		Map<String, Map<String, Object>> apiContexts = _buildAPIContexts(
			configYAML, openAPIYAML);

		for (Map.Entry<String, Map<String, Object>> entry :
				apiContexts.entrySet()) {

			_createFile(
				entry.getValue(), configYAML, copyrightFile, files,
				"typescript/api",
				StringBundler.concat(
					baseClientDir.getPath(), "/src/apis/", entry.getKey(),
					"API.ts"));
		}

		Components components = openAPIYAML.getComponents();

		Map<String, Schema> schemas = components.getSchemas();

		if (schemas != null) {
			Set<String> processedRelatedSchemaModels = new HashSet<>();

			for (Map.Entry<String, Schema> entry : schemas.entrySet()) {
				_createFile(
					_buildModelContext(entry.getKey(), entry.getValue()),
					configYAML, copyrightFile, files, "typescript/model",
					StringBundler.concat(
						baseClientDir.getPath(), "/src/models/", entry.getKey(),
						".ts"));
				_createRelatedSchemaModels(
					baseClientDir, configYAML, copyrightFile, files, "",
					processedRelatedSchemaModels, entry.getValue());
			}
		}

		_createFile(
			HashMapBuilder.<String, Object>put(
				"apiContexts", apiContexts.values()
			).put(
				"schemas", schemas
			).build(),
			configYAML, copyrightFile, files, "typescript/index",
			baseClientDir.getPath() + "/src/index.ts");
		_createFile(
			Collections.singletonMap("schemas", schemas), configYAML,
			copyrightFile, files, "typescript/serdes",
			baseClientDir.getPath() + "/src/utils/SerDes.ts");

		FileUtil.deleteFiles(baseClientDir.getPath(), files);
	}

	private static String _addAutogeneratedVulcanSchemas(
			ConfigYAML configYAML, String yamlString)
		throws Exception {

		OpenAPIYAML openAPIYAML = OpenAPIParserUtil.loadOpenAPIYAML(yamlString);
		Set<String> processedSchemaNames = new HashSet<>();

		FreeMarkerTool freeMarkerTool = FreeMarkerTool.getInstance();

		Map<String, Schema> schemas = freeMarkerTool.getSchemas(openAPIYAML);

		for (String schemaName : schemas.keySet()) {
			List<JavaMethodSignature> javaMethodSignatures =
				freeMarkerTool.getResourceJavaMethodSignatures(
					configYAML, openAPIYAML, schemaName);

			for (JavaMethodSignature javaMethodSignature :
					ResourceOpenAPIParser.
						getResourceGetPageJavaMethodSignatures(
							javaMethodSignatures)) {

				String returnType = StringUtil.removeSubstrings(
					javaMethodSignature.getReturnType(),
					"com.liferay.portal.vulcan.pagination.Page<", ">");

				String returnSchemaName = StringUtil.extractLast(
					returnType, ".");

				if (processedSchemaNames.add(returnSchemaName)) {
					if (processedSchemaNames.add("Facet")) {
						yamlString = _addSchema(
							FreeMarkerUtil.processTemplate(
								null, null, "facet_yaml", null),
							yamlString);
					}

					yamlString = _addSchema(
						FreeMarkerUtil.processTemplate(
							null, null, "page_yaml",
							HashMapBuilder.<String, Object>put(
								"schemaName", returnSchemaName
							).build()),
						yamlString);

					if (StringUtil.equals(
							"com.liferay.portal.vulcan.permission.Permission",
							returnType)) {

						yamlString = _addSchema(
							FreeMarkerUtil.processTemplate(
								null, null, "permission_yaml", null),
							yamlString);
					}
				}

				int index = StringUtil.indexOfAny(
					yamlString,
					new String[] {
						"\"" + javaMethodSignature.getPath() + "\":",
						" " + javaMethodSignature.getPath() + ":"
					});

				String httpMethod = OpenAPIParserUtil.getHTTPMethod(
					javaMethodSignature.getOperation());

				index = yamlString.indexOf(httpMethod + ":", index);

				String oldYAMLString = yamlString.substring(
					index, yamlString.indexOf("tags:", index));

				String newYAMLString = oldYAMLString.replaceAll(
					StringBundler.concat(
						"schema:\n([ \t]+)items:\n[ \t]+",
						"\\$ref: \"#/components/schemas/", returnSchemaName,
						"\"\n[ \t]+type: array"),
					StringBundler.concat(
						"schema:\n$1\\$ref: \"#/components/schemas/Page",
						returnSchemaName, "\""));

				yamlString = StringUtil.replace(
					yamlString, oldYAMLString, newYAMLString);
			}
		}

		return yamlString;
	}

	private static String _addSchema(
		String schemaYAMLString, String yamlString) {

		schemaYAMLString = StringUtil.replace(
			schemaYAMLString, new String[] {"$", "\t", "\n"},
			new String[] {"\\$", "$1", "\n$1$1"});

		return yamlString.replaceAll(
			"([ \\t]+)schemas:", "$1schemas:\n$1$1" + schemaYAMLString);
	}

	private static Map<String, Map<String, Object>> _buildAPIContexts(
		ConfigYAML configYAML, OpenAPIYAML openAPIYAML) {

		Map<String, Map<String, Object>> apiContexts = new HashMap<>();

		Map<String, Set<String>> importsMap = new HashMap<>();
		Map<String, List<Map<String, Object>>> operationDatasMap =
			new HashMap<>();

		Map<String, PathItem> pathItems = openAPIYAML.getPathItems();

		for (Map.Entry<String, PathItem> entry : pathItems.entrySet()) {
			for (Operation operation :
					OpenAPIParserUtil.getOperations(entry.getValue())) {

				List<String> tags = operation.getTags();

				List<Map<String, Object>> operationDatas =
					operationDatasMap.computeIfAbsent(
						tags.get(0), tag -> new ArrayList<>());

				operationDatas.add(
					_buildOperationData(
						configYAML,
						importsMap.computeIfAbsent(
							tags.get(0), tag -> new HashSet<>()),
						openAPIYAML, operation, entry.getKey()));
			}
		}

		for (Map.Entry<String, List<Map<String, Object>>> entry :
				operationDatasMap.entrySet()) {

			apiContexts.put(
				entry.getKey(),
				HashMapBuilder.<String, Object>put(
					"className", entry.getKey() + "API"
				).put(
					"importClasses",
					importsMap.getOrDefault(entry.getKey(), new HashSet<>())
				).put(
					"operationDatas", entry.getValue()
				).build());
		}

		return apiContexts;
	}

	private static Map<String, Object> _buildModelContext(
		String modelName, Schema schema) {

		Set<String> importClasses = new HashSet<>();
		String parentClass = null;
		List<Map<String, Object>> properties = new ArrayList<>();

		if (schema.getAllOfSchemas() != null) {
			List<Schema> allOfSchemas = schema.getAllOfSchemas();

			Schema parentSchema = allOfSchemas.get(0);

			if (parentSchema.getReference() != null) {
				String parentSchemaReference = parentSchema.getReference();

				parentClass = parentSchemaReference.substring(
					parentSchemaReference.lastIndexOf('/') + 1);

				importClasses.add(parentClass);

				for (Schema curSchema : schema.getAllOfSchemas()) {
					if (curSchema.getPropertySchemas() == null) {
						continue;
					}

					Map<String, Schema> propertySchemas =
						curSchema.getPropertySchemas();

					propertySchemas.forEach(
						(name, propertySchema) -> properties.add(
							HashMapBuilder.<String, Object>put(
								"dataType",
								_getDataType(importClasses, propertySchema)
							).put(
								"name", StringUtil.replace(name, '-', '_')
							).build()));
				}
			}
		}

		Map<String, Schema> propertySchemas = schema.getPropertySchemas();

		if (propertySchemas != null) {
			propertySchemas.forEach(
				(name, propertySchema) -> properties.add(
					HashMapBuilder.<String, Object>put(
						"dataType", _getDataType(importClasses, propertySchema)
					).put(
						"name", StringUtil.replace(name, '-', '_')
					).build()));
		}

		return HashMapBuilder.<String, Object>put(
			"description", schema.getDescription()
		).put(
			"discriminator",
			() -> {
				Discriminator discriminator = schema.getDiscriminator();

				if (discriminator == null) {
					return null;
				}

				String propertyName = discriminator.getPropertyName();

				if (Validator.isNull(propertyName)) {
					return null;
				}

				return propertyName;
			}
		).put(
			"importClasses", importClasses
		).put(
			"modelName", modelName
		).put(
			"parentClass", parentClass
		).put(
			"properties", properties
		).build();
	}

	private static Map<String, Object> _buildOperationData(
		ConfigYAML configYAML, Set<String> importClasses,
		OpenAPIYAML openAPIYAML, Operation operation, String path) {

		Map<ResponseCode, Response> responses = operation.getResponses();

		return HashMapBuilder.<String, Object>put(
			"bodyParameters",
			() -> {
				RequestBody requestBody = operation.getRequestBody();

				if (requestBody == null) {
					return null;
				}

				Map<String, Content> requestBodyContent =
					requestBody.getContent();

				if ((requestBodyContent == null) ||
					requestBodyContent.isEmpty()) {

					return null;
				}

				Map<String, List<Map<String, Object>>> bodyParameters =
					new HashMap<>();

				for (Map.Entry<String, Content> entry :
						requestBodyContent.entrySet()) {

					if (entry.getValue() == null) {
						continue;
					}

					Content content = entry.getValue();

					Schema schema = content.getSchema();

					if (schema == null) {
						continue;
					}

					Map<String, Schema> propertySchemas =
						schema.getPropertySchemas();

					List<Map<String, Object>> list = new ArrayList<>();

					if (propertySchemas != null) {
						propertySchemas.forEach(
							(name, propertySchema) -> list.add(
								HashMapBuilder.<String, Object>put(
									"dataType",
									_getDataType(importClasses, propertySchema)
								).put(
									"name", StringUtil.replace(name, '-', '_')
								).put(
									"required", false
								).put(
									"type", "form"
								).build()));
					}
					else {
						String dataType = _getDataType(importClasses, schema);

						list.add(
							HashMapBuilder.<String, Object>put(
								"dataType", dataType
							).put(
								"name",
								() -> {
									if (Validator.isNull(
											schema.getReference())) {

										return "body";
									}

									return StringUtil.replace(
										StringUtil.lowerCaseFirstLetter(
											dataType),
										'-', '_');
								}
							).put(
								"required", false
							).put(
								"type", "body"
							).build());
					}

					bodyParameters.put(entry.getKey(), list);
				}

				return bodyParameters;
			}
		).put(
			"description", operation.getDescription()
		).put(
			"httpMethod",
			StringUtil.toUpperCase(OpenAPIParserUtil.getHTTPMethod(operation))
		).put(
			"operationId", operation.getOperationId()
		).put(
			"parameters",
			() -> {
				if (operation.getParameters() == null) {
					return null;
				}

				List<Map<String, Object>> parameterDatas = new ArrayList<>();

				for (Parameter parameter : operation.getParameters()) {
					parameterDatas.add(
						HashMapBuilder.<String, Object>put(
							"dataType",
							_getDataType(importClasses, parameter.getSchema())
						).put(
							"name",
							StringUtil.replace(
								parameter.getName(), '-', StringPool.UNDERLINE)
						).put(
							"required",
							Validator.isNotNull(parameter.isRequired()) &&
							parameter.isRequired()
						).put(
							"type", parameter.getIn()
						).build());
				}

				return parameterDatas;
			}
		).put(
			"path",
			() -> {
				Application application = configYAML.getApplication();
				Info info = openAPIYAML.getInfo();

				return StringBundler.concat(
					application.getBaseURI(), "/", info.getVersion(), path);
			}
		).put(
			"responseContentTypes",
			() -> {
				if (responses == null) {
					return null;
				}

				Set<String> responseContentTypes = new LinkedHashSet<>();

				for (Response response : responses.values()) {
					Map<String, Content> content = response.getContent();

					if (content != null) {
						responseContentTypes.addAll(content.keySet());
					}
				}

				if (responseContentTypes.isEmpty()) {
					return null;
				}

				return responseContentTypes;
			}
		).put(
			"returnDataType",
			() -> {
				if (responses == null) {
					return null;
				}

				for (Map.Entry<ResponseCode, Response> entry :
						responses.entrySet()) {

					ResponseCode responseCode = entry.getKey();

					if (!Objects.equals(responseCode.getHttpCode(), 200)) {
						continue;
					}

					Response response = entry.getValue();

					Map<String, Content> contentsMap = response.getContent();

					if ((contentsMap == null) || contentsMap.isEmpty()) {
						continue;
					}

					Collection<Content> contents = contentsMap.values();

					Iterator<Content> iterator = contents.iterator();

					Content content = iterator.next();

					if ((content == null) || (content.getSchema() == null)) {
						continue;
					}

					return _getDataType(importClasses, content.getSchema());
				}

				return null;
			}
		).build();
	}

	private static void _createBuildGradleFile(
			File baseClientDir, List<File> files)
		throws Exception {

		File file = new File(baseClientDir, "build.gradle");

		files.add(file);

		FileUtil.write(file, StringPool.BLANK);
	}

	private static void _createFile(
			Map<String, Object> context, ConfigYAML configYAML,
			File copyrightFile, List<File> files, String name, String path)
		throws Exception {

		File file = new File(path);

		files.add(file);

		if (context != null) {
			context = HashMapBuilder.<String, Object>putAll(
				context
			).put(
				"configYAML", configYAML
			).put(
				"stringUtil", StringUtil_IW.getInstance()
			).build();
		}
		else {
			context = Collections.singletonMap("configYAML", configYAML);
		}

		FileUtil.write(
			file,
			FreeMarkerUtil.processTemplate(
				copyrightFile, FileUtil.getCopyrightYear(file), name, context));
	}

	private static void _createNodeScriptsConfigFile(
			File basePath, List<File> files)
		throws Exception {

		File file = new File(basePath, "node-scripts.config.js");

		files.add(file);

		FileUtil.write(
			file,
			FreeMarkerUtil.processTemplate(
				null, null, "typescript/node_scripts_config_js", null));
	}

	private static void _createPackageJSONFile(File basePath, List<File> files)
		throws Exception {

		File file = new File(basePath, "package.json");

		files.add(file);

		FileUtil.write(
			file,
			FreeMarkerUtil.processTemplate(
				null, null, "typescript/package_json",
				HashMapBuilder.<String, Object>put(
					"clientName", basePath.getName()
				).build()));
	}

	private static void _createRelatedSchemaModels(
			File baseClientDir, ConfigYAML configYAML, File copyrightFile,
			List<File> files, String parentYAMLPath,
			Set<String> processedReferences, Schema schema)
		throws Exception {

		Map<String, Schema> propertySchemas = schema.getPropertySchemas();

		if (propertySchemas == null) {
			return;
		}

		for (Schema propertySchema : propertySchemas.values()) {
			List<String> references = new ArrayList<>();

			if (propertySchema.getReference() != null) {
				references.add(propertySchema.getReference());
			}

			Items items = propertySchema.getItems();

			if (items != null) {
				Schema itemsSchema = items.toSchema();

				if (itemsSchema.getReference() != null) {
					references.add(itemsSchema.getReference());
				}
			}

			for (String reference : references) {
				boolean local = reference.startsWith("#");

				if ((local && Validator.isNull(parentYAMLPath)) ||
					!processedReferences.add(reference)) {

					continue;
				}

				String referencedSchemaName = null;

				if (local) {
					referencedSchemaName = reference.substring(
						reference.lastIndexOf("/") + 1);
				}
				else {
					referencedSchemaName = reference.split("#")[1];

					referencedSchemaName = referencedSchemaName.substring(
						referencedSchemaName.lastIndexOf("/") + 1);
				}

				File referencedYAMLFile = null;

				if (local) {
					referencedYAMLFile = new File(parentYAMLPath);
				}
				else {
					String referencedYAMLFileName = parentYAMLPath.substring(
						0, parentYAMLPath.lastIndexOf("/") + 1);

					referencedYAMLFileName += reference.split("#")[0];

					referencedYAMLFile = new File(referencedYAMLFileName);
				}

				files.add(referencedYAMLFile);

				OpenAPIYAML referencedOpenAPIYAML =
					OpenAPIParserUtil.loadOpenAPIYAML(
						FileUtil.read(referencedYAMLFile));

				Components referencedComponents =
					referencedOpenAPIYAML.getComponents();

				Map<String, Schema> referencedSchemas =
					referencedComponents.getSchemas();

				Schema referencedSchema = referencedSchemas.get(
					referencedSchemaName);

				_createFile(
					_buildModelContext(referencedSchemaName, referencedSchema),
					configYAML, copyrightFile, files, "typescript/model",
					StringBundler.concat(
						baseClientDir.getPath(), "/src/models/",
						referencedSchemaName, ".ts"));
				_createRelatedSchemaModels(
					baseClientDir, configYAML, copyrightFile, files,
					referencedYAMLFile.getAbsolutePath(), processedReferences,
					referencedSchema);
			}
		}
	}

	private static String _getDataType(
		Set<String> importClasses, Schema schema) {

		if (schema == null) {
			return "any";
		}

		if (schema.getReference() != null) {
			String dataType = null;

			String schemaReference = schema.getReference();

			if (schemaReference.startsWith("#")) {
				dataType = schemaReference.substring(
					schemaReference.lastIndexOf('/') + 1);
			}
			else {
				dataType = schemaReference.substring(
					schemaReference.lastIndexOf('#') + 1);
			}

			importClasses.add(dataType);

			return dataType;
		}

		String type = schema.getType();

		if (type.equals("array")) {
			Items items = schema.getItems();

			return "Array<" + _getDataType(importClasses, items.toSchema()) +
				">";
		}
		else if (type.equals("boolean")) {
			return "boolean";
		}
		else if (type.equals("integer") || type.equals("number")) {
			return "number";
		}
		else if (type.equals("object")) {
			Schema additionalPropertySchema =
				schema.getAdditionalPropertySchema();

			if (additionalPropertySchema == null) {
				return "object";
			}

			if (additionalPropertySchema.getAdditionalPropertySchema() !=
					null) {

				String dataType = _getDataType(
					importClasses,
					additionalPropertySchema.getAdditionalPropertySchema());

				return "{[key: string]: {[key: string]: " + dataType + ";};}";
			}

			return "{[key: string]: " +
				_getDataType(importClasses, additionalPropertySchema) + ";}";
		}
		else if (type.equals("permission")) {
			importClasses.add("Permission");

			return "Permission";
		}
		else if (type.equals("string")) {
			List<String> values = schema.getEnumValues();

			if (ListUtil.isNotNull(values)) {
				StringBuilder sb = new StringBuilder();

				for (String value : values) {
					if (sb.length() > 0) {
						sb.append(" | ");
					}

					sb.append("'");
					sb.append(value);
					sb.append("'");
				}

				return sb.toString();
			}

			String format = schema.getFormat();

			if (Validator.isNotNull(format)) {
				if (format.equals("date") || format.equals("date-time")) {
					return "Date";
				}
				else if (format.equals("binary")) {
					return "File";
				}
			}

			return "string";
		}

		return "any";
	}

}