/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.portal.tools.bundle.support.maven;

import com.liferay.portal.tools.bundle.support.commands.DistBundleCommand;
import com.liferay.portal.tools.bundle.support.constants.BundleSupportConstants;
import com.liferay.portal.tools.bundle.support.internal.util.BundleSupportUtil;
import com.liferay.portal.tools.bundle.support.internal.util.FileUtil;
import com.liferay.portal.tools.bundle.support.internal.util.MavenUtil;

import java.io.File;

import java.nio.file.Path;
import java.nio.file.Paths;

import org.apache.maven.model.Build;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;

/**
 * @author David Truong
 * @author Andrea Di Giorgi
 */
@Mojo(name = "dist")
public class DistBundleMojo extends InitBundleMojo {

	@Override
	public void execute() throws MojoExecutionException {
		MavenProject rootProject = MavenUtil.getRootProject(project);

		Build build = rootProject.getBuild();

		if (archiveFileName == null) {
			archiveFileName = rootProject.getArtifactId();
		}

		String archiveLocation = build.getDirectory() + "/" + archiveFileName;

		setLiferayHome(archiveLocation);

		File archiveFile = new File(archiveLocation + "." + format);

		String packaging = project.getPackaging();

		try {
			if (packaging.equals("jar") || packaging.equals("war")) {
				String deployDirName = BundleSupportUtil.getDeployDirName(
					deployFile.getName());

				if (includeFolder) {
					deployDirName = archiveFileName + "/" + deployDirName;
				}

				Path entryPath = Paths.get(deployDirName, outputFileName);

				if (format.equals("zip")) {
					FileUtil.appendZip(deployFile, entryPath, archiveFile);
				}
				else if (format.equals("gz") || format.equals("tar") ||
						 format.equals("tar.gz") || format.equals("tgz")) {

					FileUtil.appendTar(deployFile, entryPath, archiveFile);
				}
				else {
					throw new IllegalArgumentException(
						"Please specify either zip or tar.gz or tgz");
				}
			}
			else if (!project.hasParent()) {
				archiveFile.delete();

				File liferayHomeDir = getLiferayHomeDir();

				super.execute();

				DistBundleCommand distBundleCommand = new DistBundleCommand();

				distBundleCommand.setFormat(format);
				distBundleCommand.setIncludeFolder(includeFolder);
				distBundleCommand.setLiferayHomeDir(getLiferayHomeDir());
				distBundleCommand.setOutputFile(archiveFile);

				distBundleCommand.execute();

				FileUtil.deleteDirectory(liferayHomeDir.toPath());
			}
		}
		catch (MojoExecutionException mee) {
			throw mee;
		}
		catch (Exception e) {
			throw new MojoExecutionException(
				"Unable to create distributable bundle", e);
		}
	}

	@Parameter
	protected String archiveFileName;

	@Parameter(
		defaultValue = "${project.build.directory}/${project.build.finalName}.${project.packaging}",
		required = true
	)
	protected File deployFile;

	@Parameter(
		defaultValue = BundleSupportConstants.DEFAULT_BUNDLE_FORMAT,
		required = true
	)
	protected String format;

	@Parameter(
		defaultValue = "" + BundleSupportConstants.DEFAULT_INCLUDE_FOLDER,
		required = true
	)
	protected boolean includeFolder;

	@Parameter(
		defaultValue = "${project.artifactId}.${project.packaging}",
		required = true
	)
	protected String outputFileName;

}