/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.portal.background.task.internal;

import com.liferay.portal.kernel.backgroundtask.BackgroundTaskStatus;
import com.liferay.portal.kernel.backgroundtask.BackgroundTaskStatusMessageTranslator;
import com.liferay.portal.kernel.backgroundtask.BackgroundTaskStatusRegistry;
import com.liferay.portal.kernel.backgroundtask.BackgroundTaskStatusRegistryUtil;
import com.liferay.portal.kernel.cluster.ClusterMasterExecutor;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.util.MethodHandler;

import java.util.AbstractMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Michael C. Han
 */
@Component(service = BackgroundTaskStatusRegistry.class)
public class BackgroundTaskStatusRegistryImpl
	implements BackgroundTaskStatusRegistry {

	@Override
	public BackgroundTaskStatus getBackgroundTaskStatus(long backgroundTaskId) {
		if (!_clusterMasterExecutor.isMaster()) {
			return _getMasterBackgroundTaskStatus(backgroundTaskId);
		}

		Map.Entry<BackgroundTaskStatus, BackgroundTaskStatusMessageTranslator>
			entry = _entries.get(backgroundTaskId);

		if (entry == null) {
			return null;
		}

		return entry.getKey();
	}

	@Override
	public BackgroundTaskStatusMessageTranslator
		getBackgroundTaskStatusMessageTranslator(long backgroundTaskId) {

		Map.Entry<BackgroundTaskStatus, BackgroundTaskStatusMessageTranslator>
			entry = _entries.get(backgroundTaskId);

		if (entry == null) {
			return null;
		}

		return entry.getValue();
	}

	@Override
	public BackgroundTaskStatus registerBackgroundTaskStatus(
		long backgroundTaskId,
		BackgroundTaskStatusMessageTranslator
			backgroundTaskStatusMessageTranslator) {

		Map.Entry<BackgroundTaskStatus, BackgroundTaskStatusMessageTranslator>
			entry = _entries.computeIfAbsent(
				backgroundTaskId,
				key -> new AbstractMap.SimpleImmutableEntry<>(
					new BackgroundTaskStatus(),
					backgroundTaskStatusMessageTranslator));

		return entry.getKey();
	}

	@Override
	public BackgroundTaskStatus unregisterBackgroundTaskStatus(
		long backgroundTaskId) {

		Map.Entry<BackgroundTaskStatus, BackgroundTaskStatusMessageTranslator>
			entry = _entries.remove(backgroundTaskId);

		if (entry == null) {
			return null;
		}

		return entry.getKey();
	}

	private BackgroundTaskStatus _getMasterBackgroundTaskStatus(
		long backgroundTaskId) {

		try {
			MethodHandler methodHandler = new MethodHandler(
				BackgroundTaskStatusRegistryUtil.class.getDeclaredMethod(
					"getBackgroundTaskStatus", long.class),
				backgroundTaskId);

			Future<BackgroundTaskStatus> future =
				_clusterMasterExecutor.executeOnMaster(methodHandler);

			return future.get();
		}
		catch (Exception exception) {
			_log.error(
				"Unable to get master background task status", exception);
		}

		return null;
	}

	private static final Log _log = LogFactoryUtil.getLog(
		BackgroundTaskStatusRegistryImpl.class);

	@Reference
	private ClusterMasterExecutor _clusterMasterExecutor;

	private final Map
		<Long,
		 Map.Entry<BackgroundTaskStatus, BackgroundTaskStatusMessageTranslator>>
			_entries = new ConcurrentHashMap<>();

}