/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.object.internal.security.permission.resource;

import com.liferay.object.constants.ObjectConstants;
import com.liferay.object.model.ObjectDefinition;
import com.liferay.object.service.ObjectDefinitionLocalService;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.security.auth.PrincipalException;
import com.liferay.portal.kernel.security.permission.PermissionChecker;
import com.liferay.portal.kernel.security.permission.resource.ModelResourcePermission;
import com.liferay.portal.kernel.security.permission.resource.PortletResourcePermission;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Gabriel Albuquerque
 */
@Component(
	immediate = true,
	property = "model.class.name=com.liferay.object.model.ObjectDefinition",
	service = ModelResourcePermission.class
)
public class ObjectDefinitionModelResourcePermission
	implements ModelResourcePermission<ObjectDefinition> {

	@Override
	public void check(
			PermissionChecker permissionChecker, long objectDefinitionId,
			String actionId)
		throws PortalException {

		if (!contains(permissionChecker, objectDefinitionId, actionId)) {
			throw new PrincipalException.MustHavePermission(
				permissionChecker, ObjectDefinition.class.getName(),
				objectDefinitionId, actionId);
		}
	}

	@Override
	public void check(
			PermissionChecker permissionChecker,
			ObjectDefinition objectDefinition, String actionId)
		throws PortalException {

		if (!contains(permissionChecker, objectDefinition, actionId)) {
			throw new PrincipalException.MustHavePermission(
				permissionChecker, ObjectDefinition.class.getName(),
				objectDefinition.getPrimaryKey(), actionId);
		}
	}

	@Override
	public boolean contains(
			PermissionChecker permissionChecker, long objectDefinitionId,
			String actionId)
		throws PortalException {

		ObjectDefinition objectDefinition =
			_objectDefinitionLocalService.getObjectDefinition(
				objectDefinitionId);

		return contains(permissionChecker, objectDefinition, actionId);
	}

	@Override
	public boolean contains(
			PermissionChecker permissionChecker,
			ObjectDefinition objectDefinition, String actionId)
		throws PortalException {

		if (permissionChecker.hasOwnerPermission(
				permissionChecker.getCompanyId(),
				ObjectDefinition.class.getName(),
				objectDefinition.getObjectDefinitionId(),
				objectDefinition.getUserId(), actionId) ||
			permissionChecker.hasPermission(
				null, ObjectDefinition.class.getName(),
				objectDefinition.getPrimaryKey(), actionId)) {

			return true;
		}

		return false;
	}

	@Override
	public String getModelName() {
		return ObjectDefinition.class.getName();
	}

	@Override
	public PortletResourcePermission getPortletResourcePermission() {
		return _portletResourcePermission;
	}

	@Reference
	private ObjectDefinitionLocalService _objectDefinitionLocalService;

	@Reference(target = "(resource.name=" + ObjectConstants.RESOURCE_NAME + ")")
	private PortletResourcePermission _portletResourcePermission;

}