/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.object.service.impl;

import com.liferay.dynamic.data.mapping.expression.CreateExpressionRequest;
import com.liferay.dynamic.data.mapping.expression.DDMExpressionFactory;
import com.liferay.object.constants.ObjectValidationRuleConstants;
import com.liferay.object.constants.ObjectValidationRuleSettingConstants;
import com.liferay.object.definition.util.ObjectDefinitionUtil;
import com.liferay.object.exception.DuplicateObjectValidationRuleExternalReferenceCodeException;
import com.liferay.object.exception.ObjectValidationRuleEngineException;
import com.liferay.object.exception.ObjectValidationRuleNameException;
import com.liferay.object.exception.ObjectValidationRuleOutputTypeException;
import com.liferay.object.exception.ObjectValidationRuleScriptException;
import com.liferay.object.exception.ObjectValidationRuleSettingNameException;
import com.liferay.object.exception.ObjectValidationRuleSettingValueException;
import com.liferay.object.exception.ObjectValidationRuleSystemException;
import com.liferay.object.internal.action.util.ObjectEntryVariablesUtil;
import com.liferay.object.internal.validation.rule.FunctionObjectValidationRuleEngineImpl;
import com.liferay.object.internal.validation.rule.UniqueCompositeKeyObjectValidationRuleEngineImpl;
import com.liferay.object.model.ObjectDefinition;
import com.liferay.object.model.ObjectField;
import com.liferay.object.model.ObjectValidationRule;
import com.liferay.object.model.ObjectValidationRuleSetting;
import com.liferay.object.scripting.exception.ObjectScriptingException;
import com.liferay.object.scripting.validator.ObjectScriptingValidator;
import com.liferay.object.service.ObjectValidationRuleSettingLocalService;
import com.liferay.object.service.base.ObjectValidationRuleLocalServiceBaseImpl;
import com.liferay.object.service.persistence.ObjectDefinitionPersistence;
import com.liferay.object.service.persistence.ObjectFieldPersistence;
import com.liferay.object.service.persistence.ObjectValidationRuleSettingPersistence;
import com.liferay.object.system.SystemObjectDefinitionManagerRegistry;
import com.liferay.object.validation.rule.ObjectValidationRuleEngine;
import com.liferay.object.validation.rule.ObjectValidationRuleEngineRegistry;
import com.liferay.object.validation.rule.ObjectValidationRuleResult;
import com.liferay.petra.function.transform.TransformUtil;
import com.liferay.portal.aop.AopService;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.feature.flag.FeatureFlagManagerUtil;
import com.liferay.portal.kernel.json.JSONObject;
import com.liferay.portal.kernel.language.Language;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.model.BaseModel;
import com.liferay.portal.kernel.model.SystemEventConstants;
import com.liferay.portal.kernel.model.User;
import com.liferay.portal.kernel.search.Indexable;
import com.liferay.portal.kernel.search.IndexableType;
import com.liferay.portal.kernel.service.UserLocalService;
import com.liferay.portal.kernel.systemevent.SystemEvent;
import com.liferay.portal.kernel.transaction.Transactional;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.ListUtil;
import com.liferay.portal.kernel.util.LocaleUtil;
import com.liferay.portal.kernel.util.SetUtil;
import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.vulcan.dto.converter.DTOConverterRegistry;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Marco Leo
 */
@Component(
	property = "model.class.name=com.liferay.object.model.ObjectValidationRule",
	service = AopService.class
)
public class ObjectValidationRuleLocalServiceImpl
	extends ObjectValidationRuleLocalServiceBaseImpl {

	@Indexable(type = IndexableType.REINDEX)
	@Override
	public ObjectValidationRule addObjectValidationRule(
			String externalReferenceCode, long userId, long objectDefinitionId,
			boolean active, String engine, Map<Locale, String> errorLabelMap,
			Map<Locale, String> nameMap, String outputType, String script,
			boolean system,
			List<ObjectValidationRuleSetting> objectValidationRuleSettings)
		throws PortalException {

		_validateInvokerBundle(
			"Only allowed bundles can create system object validation rules",
			system);

		User user = _userLocalService.getUser(userId);

		_validate(
			externalReferenceCode, 0, objectDefinitionId, user.getCompanyId(),
			engine, nameMap, outputType, script, objectValidationRuleSettings);

		ObjectValidationRule objectValidationRule =
			objectValidationRulePersistence.create(
				counterLocalService.increment());

		if (Validator.isNull(externalReferenceCode)) {
			externalReferenceCode = objectValidationRule.getUuid();
		}

		objectValidationRule.setExternalReferenceCode(externalReferenceCode);
		objectValidationRule.setCompanyId(user.getCompanyId());
		objectValidationRule.setUserId(user.getUserId());
		objectValidationRule.setUserName(user.getFullName());
		objectValidationRule.setObjectDefinitionId(objectDefinitionId);
		objectValidationRule.setActive(active);
		objectValidationRule.setEngine(engine);
		objectValidationRule.setErrorLabelMap(errorLabelMap);
		objectValidationRule.setNameMap(nameMap);
		objectValidationRule.setOutputType(outputType);
		objectValidationRule.setScript(script);
		objectValidationRule.setSystem(system);

		objectValidationRule = objectValidationRulePersistence.update(
			objectValidationRule);

		long objectValidationRuleId =
			objectValidationRule.getObjectValidationRuleId();

		objectValidationRule.setObjectValidationRuleSettings(
			TransformUtil.transform(
				objectValidationRuleSettings,
				objectValidationRuleSetting ->
					_objectValidationRuleSettingLocalService.
						addObjectValidationRuleSetting(
							userId, objectValidationRuleId,
							objectValidationRuleSetting.getName(),
							objectValidationRuleSetting.getValue())));

		return objectValidationRule;
	}

	@Indexable(type = IndexableType.DELETE)
	@Override
	public ObjectValidationRule deleteObjectValidationRule(
			long objectValidationRuleId)
		throws PortalException {

		ObjectValidationRule objectValidationRule =
			objectValidationRulePersistence.findByPrimaryKey(
				objectValidationRuleId);

		return deleteObjectValidationRule(objectValidationRule);
	}

	@Indexable(type = IndexableType.DELETE)
	@Override
	@SystemEvent(type = SystemEventConstants.TYPE_DELETE)
	public ObjectValidationRule deleteObjectValidationRule(
			ObjectValidationRule objectValidationRule)
		throws PortalException {

		_validateInvokerBundle(
			"Only allowed bundles can delete system object validation rules",
			objectValidationRule.isSystem());

		objectValidationRule = objectValidationRulePersistence.remove(
			objectValidationRule);

		_objectValidationRuleSettingPersistence.removeByObjectValidationRuleId(
			objectValidationRule.getObjectValidationRuleId());

		return objectValidationRule;
	}

	@Override
	public void deleteObjectValidationRules(Long objectDefinitionId)
		throws PortalException {

		for (ObjectValidationRule objectValidationRule :
				objectValidationRulePersistence.findByObjectDefinitionId(
					objectDefinitionId)) {

			objectValidationRuleLocalService.deleteObjectValidationRule(
				objectValidationRule);
		}
	}

	@Override
	public ObjectValidationRule fetchObjectValidationRule(
		String externalReferenceCode, long objectDefinitionId) {

		ObjectDefinition objectDefinition =
			_objectDefinitionPersistence.fetchByPrimaryKey(objectDefinitionId);

		if (objectDefinition == null) {
			return null;
		}

		return objectValidationRulePersistence.fetchByERC_C_ODI(
			externalReferenceCode, objectDefinition.getCompanyId(),
			objectDefinitionId);
	}

	@Override
	public ObjectValidationRule getObjectValidationRule(
			long objectValidationRuleId)
		throws PortalException {

		ObjectValidationRule objectValidationRule =
			objectValidationRulePersistence.findByPrimaryKey(
				objectValidationRuleId);

		objectValidationRule.setObjectValidationRuleSettings(
			_objectValidationRuleSettingPersistence.
				findByObjectValidationRuleId(objectValidationRuleId));

		return objectValidationRule;
	}

	@Override
	public List<ObjectValidationRule> getObjectValidationRules(
		long objectDefinitionId) {

		return _getObjectValidationRules(
			objectValidationRulePersistence.findByObjectDefinitionId(
				objectDefinitionId));
	}

	@Override
	public List<ObjectValidationRule> getObjectValidationRules(
		long objectDefinitionId, boolean active) {

		return _getObjectValidationRules(
			objectValidationRulePersistence.findByODI_A(
				objectDefinitionId, active));
	}

	@Override
	public List<ObjectValidationRule> getObjectValidationRules(
		long objectDefinitionId, String engine) {

		return _getObjectValidationRules(
			objectValidationRulePersistence.findByODI_E(
				objectDefinitionId, engine));
	}

	@Override
	public int getObjectValidationRulesCount(
		long objectDefinitionId, boolean active) {

		return objectValidationRulePersistence.countByODI_A(
			objectDefinitionId, active);
	}

	@Override
	public void unassociateObjectField(ObjectField objectField) {
		for (ObjectValidationRule objectValidationRule :
				objectValidationRulePersistence.findByODI_O(
					objectField.getObjectDefinitionId(),
					ObjectValidationRuleConstants.
						OUTPUT_TYPE_PARTIAL_VALIDATION)) {

			ObjectValidationRuleSetting objectValidationRuleSetting =
				_objectValidationRuleSettingPersistence.fetchByOVRI_N_V(
					objectValidationRule.getObjectValidationRuleId(),
					ObjectValidationRuleSettingConstants.
						NAME_OUTPUT_OBJECT_FIELD_ID,
					String.valueOf(objectField.getObjectFieldId()));

			if (objectValidationRuleSetting == null) {
				continue;
			}

			_objectValidationRuleSettingPersistence.remove(
				objectValidationRuleSetting);

			int count = _objectValidationRuleSettingPersistence.countByOVRI_N(
				objectValidationRule.getObjectValidationRuleId(),
				ObjectValidationRuleSettingConstants.
					NAME_OUTPUT_OBJECT_FIELD_ID);

			if (count == 0) {
				objectValidationRule.setOutputType(
					ObjectValidationRuleConstants.OUTPUT_TYPE_FULL_VALIDATION);

				objectValidationRulePersistence.update(objectValidationRule);
			}
		}
	}

	@Indexable(type = IndexableType.REINDEX)
	@Override
	public ObjectValidationRule updateObjectValidationRule(
			String externalReferenceCode, long objectValidationRuleId,
			boolean active, String engine, Map<Locale, String> errorLabelMap,
			Map<Locale, String> nameMap, String outputType, String script,
			List<ObjectValidationRuleSetting> objectValidationRuleSettings)
		throws PortalException {

		ObjectValidationRule objectValidationRule =
			objectValidationRulePersistence.findByPrimaryKey(
				objectValidationRuleId);

		_validateInvokerBundle(
			"Only allowed bundles can edit system object validation rules",
			objectValidationRule.isSystem());

		_validate(
			externalReferenceCode,
			objectValidationRule.getObjectValidationRuleId(),
			objectValidationRule.getObjectDefinitionId(),
			objectValidationRule.getCompanyId(), engine, nameMap, outputType,
			script, objectValidationRuleSettings);

		objectValidationRule.setExternalReferenceCode(externalReferenceCode);
		objectValidationRule.setActive(active);
		objectValidationRule.setEngine(engine);
		objectValidationRule.setErrorLabelMap(errorLabelMap);
		objectValidationRule.setNameMap(nameMap);
		objectValidationRule.setOutputType(outputType);
		objectValidationRule.setScript(script);

		objectValidationRule = objectValidationRulePersistence.update(
			objectValidationRule);

		_objectValidationRuleSettingPersistence.removeByObjectValidationRuleId(
			objectValidationRuleId);

		objectValidationRule.setObjectValidationRuleSettings(
			_updateObjectValidationRuleSettings(
				objectValidationRuleSettings, objectValidationRule));

		return objectValidationRule;
	}

	@Override
	@Transactional(readOnly = true)
	public void validate(
			BaseModel<?> baseModel, long objectDefinitionId,
			JSONObject payloadJSONObject, long userId)
		throws PortalException {

		if (baseModel == null) {
			return;
		}

		List<ObjectValidationRule> objectValidationRules =
			objectValidationRuleLocalService.getObjectValidationRules(
				objectDefinitionId, true);

		if (ListUtil.isEmpty(objectValidationRules)) {
			return;
		}

		ObjectDefinition objectDefinition =
			_objectDefinitionPersistence.fetchByPrimaryKey(objectDefinitionId);

		Map<String, Object> variables = ObjectEntryVariablesUtil.getVariables(
			_dtoConverterRegistry, objectDefinition, payloadJSONObject,
			_systemObjectDefinitionManagerRegistry);

		List<ObjectValidationRuleResult> objectValidationRuleResults =
			new ArrayList<>();

		for (ObjectValidationRule objectValidationRule :
				objectValidationRules) {

			Map<String, Object> results = new HashMap<>();

			ObjectValidationRuleEngine objectValidationRuleEngine =
				_objectValidationRuleEngineRegistry.
					getObjectValidationRuleEngine(
						objectValidationRule.getCompanyId(),
						objectValidationRule.getEngine());

			if (StringUtil.equals(
					objectValidationRuleEngine.getKey(),
					ObjectValidationRuleConstants.ENGINE_TYPE_COMPOSITE_KEY)) {

				variables.put("objectValidationRule", objectValidationRule);

				results = objectValidationRuleEngine.execute(variables, null);
			}
			else if (StringUtil.equals(
						objectValidationRuleEngine.getKey(),
						ObjectValidationRuleConstants.ENGINE_TYPE_DDM)) {

				results = objectValidationRuleEngine.execute(
					variables, objectValidationRule.getScript());
			}
			else if (StringUtil.equals(
						objectValidationRuleEngine.getKey(),
						ObjectValidationRuleConstants.ENGINE_TYPE_GROOVY)) {

				results = objectValidationRuleEngine.execute(
					(Map<String, Object>)variables.get("baseModel"),
					objectValidationRule.getScript());
			}
			else {
				results = objectValidationRuleEngine.execute(
					(Map<String, Object>)variables.get("entryDTO"), null);
			}

			Locale locale = LocaleUtil.getMostRelevantLocale();

			User user = _userLocalService.fetchUser(userId);

			if (user != null) {
				locale = user.getLocale();
			}

			String errorMessage = null;

			if (!GetterUtil.getBoolean(results.get("validationCriteriaMet"))) {
				errorMessage = objectValidationRule.getErrorLabel(locale);
			}
			else if (GetterUtil.getBoolean(results.get("invalidScript"))) {
				errorMessage = _language.get(
					locale, "there-was-an-error-validating-your-data");
			}

			if (Validator.isNull(errorMessage)) {
				continue;
			}

			if (objectValidationRule.compareOutputType(
					ObjectValidationRuleConstants.
						OUTPUT_TYPE_PARTIAL_VALIDATION)) {

				for (ObjectValidationRuleSetting objectValidationRuleSetting :
						_objectValidationRuleSettingPersistence.findByOVRI_N(
							objectValidationRule.getObjectValidationRuleId(),
							ObjectValidationRuleSettingConstants.
								NAME_OUTPUT_OBJECT_FIELD_ID)) {

					ObjectField objectField =
						_objectFieldPersistence.fetchByPrimaryKey(
							GetterUtil.getLong(
								objectValidationRuleSetting.getValue()));

					if (objectField == null) {
						continue;
					}

					objectValidationRuleResults.add(
						new ObjectValidationRuleResult(
							errorMessage, objectField.getName()));
				}
			}
			else {
				objectValidationRuleResults.add(
					new ObjectValidationRuleResult(errorMessage));
			}
		}

		if (ListUtil.isNotEmpty(objectValidationRuleResults)) {
			throw new ObjectValidationRuleEngineException(
				objectValidationRuleResults);
		}
	}

	private List<ObjectValidationRule> _getObjectValidationRules(
		List<ObjectValidationRule> objectValidationRules) {

		for (ObjectValidationRule objectValidationRule :
				objectValidationRules) {

			objectValidationRule.setObjectValidationRuleSettings(
				_objectValidationRuleSettingPersistence.
					findByObjectValidationRuleId(
						objectValidationRule.getObjectValidationRuleId()));
		}

		return objectValidationRules;
	}

	private List<ObjectValidationRuleSetting>
			_updateObjectValidationRuleSettings(
				List<ObjectValidationRuleSetting>
					newObjectValidationRuleSettings,
				ObjectValidationRule objectValidationRule)
		throws PortalException {

		Set<ObjectValidationRuleSetting> objectValidationRuleSettings =
			SetUtil.fromCollection(newObjectValidationRuleSettings);

		for (ObjectValidationRuleSetting oldObjectValidationRuleSetting :
				_objectValidationRuleSettingPersistence.
					findByObjectValidationRuleId(
						objectValidationRule.getObjectValidationRuleId())) {

			boolean delete = true;

			for (ObjectValidationRuleSetting newObjectValidationRuleSetting :
					newObjectValidationRuleSettings) {

				if (StringUtil.equals(
						oldObjectValidationRuleSetting.getName(),
						newObjectValidationRuleSetting.getName()) &&
					StringUtil.equals(
						oldObjectValidationRuleSetting.getValue(),
						newObjectValidationRuleSetting.getValue())) {

					delete = false;

					objectValidationRuleSettings.remove(
						newObjectValidationRuleSetting);

					break;
				}
			}

			if (delete) {
				_objectValidationRuleSettingPersistence.remove(
					oldObjectValidationRuleSetting);
			}
		}

		for (ObjectValidationRuleSetting objectValidationRuleSetting :
				objectValidationRuleSettings) {

			_objectValidationRuleSettingLocalService.
				addObjectValidationRuleSetting(
					objectValidationRule.getUserId(),
					objectValidationRule.getObjectValidationRuleId(),
					objectValidationRuleSetting.getName(),
					objectValidationRuleSetting.getValue());
		}

		return newObjectValidationRuleSettings;
	}

	private void _validate(
			String externalReferenceCode, long objectValidationRuleId,
			long objectDefinitionId, long companyId, String engine,
			Map<Locale, String> nameMap, String outputType, String script,
			List<ObjectValidationRuleSetting> objectValidationRuleSettings)
		throws PortalException {

		if (Validator.isNotNull(externalReferenceCode)) {
			ObjectValidationRule objectValidationRule =
				objectValidationRulePersistence.fetchByERC_C_ODI(
					externalReferenceCode, companyId, objectDefinitionId);

			if ((objectValidationRule != null) &&
				(objectValidationRule.getObjectValidationRuleId() !=
					objectValidationRuleId)) {

				throw new DuplicateObjectValidationRuleExternalReferenceCodeException();
			}
		}

		if (Validator.isNull(engine)) {
			throw new ObjectValidationRuleEngineException.MustNotBeNull();
		}

		ObjectValidationRuleEngine objectValidationRuleEngine =
			_objectValidationRuleEngineRegistry.getObjectValidationRuleEngine(
				companyId, engine);

		if (objectValidationRuleEngine == null) {
			throw new ObjectValidationRuleEngineException.NoSuchEngine(engine);
		}

		Locale locale = LocaleUtil.getSiteDefault();

		if ((nameMap == null) || Validator.isNull(nameMap.get(locale))) {
			throw new ObjectValidationRuleNameException(
				"Name is null for locale " + locale.getDisplayName());
		}

		if (!StringUtil.equals(
				outputType,
				ObjectValidationRuleConstants.OUTPUT_TYPE_FULL_VALIDATION) &&
			!StringUtil.equals(
				outputType,
				ObjectValidationRuleConstants.OUTPUT_TYPE_PARTIAL_VALIDATION)) {

			throw new ObjectValidationRuleOutputTypeException(
				"Invalid output type " + outputType);
		}

		if (Validator.isNull(script) &&
			!(objectValidationRuleEngine instanceof
				FunctionObjectValidationRuleEngineImpl ||
			  objectValidationRuleEngine instanceof
				  UniqueCompositeKeyObjectValidationRuleEngineImpl)) {

			throw new ObjectValidationRuleScriptException(
				"The script is required", "required");
		}

		try {
			if (Objects.equals(
					engine, ObjectValidationRuleConstants.ENGINE_TYPE_DDM)) {

				_ddmExpressionFactory.createExpression(
					CreateExpressionRequest.Builder.newBuilder(
						script
					).build());
			}
			else if (Objects.equals(
						engine,
						ObjectValidationRuleConstants.ENGINE_TYPE_GROOVY)) {

				_objectScriptingValidator.validate("groovy", script);
			}
		}
		catch (PortalException portalException) {
			if (_log.isDebugEnabled()) {
				_log.debug(portalException);
			}

			if (portalException instanceof ObjectScriptingException) {
				ObjectScriptingException objectScriptingException =
					(ObjectScriptingException)portalException;

				throw new ObjectValidationRuleScriptException(
					objectScriptingException.getMessage(),
					objectScriptingException.getMessageKey());
			}

			throw new ObjectValidationRuleScriptException(
				"The script syntax is invalid", "syntax-error");
		}

		if (StringUtil.equals(
				outputType,
				ObjectValidationRuleConstants.OUTPUT_TYPE_PARTIAL_VALIDATION) &&
			ListUtil.isEmpty(objectValidationRuleSettings)) {

			throw new ObjectValidationRuleSettingNameException.
				MissingRequiredName(
					ObjectValidationRuleSettingConstants.
						NAME_OUTPUT_OBJECT_FIELD_ID);
		}

		for (ObjectValidationRuleSetting objectValidationRuleSetting :
				objectValidationRuleSettings) {

			if (FeatureFlagManagerUtil.isEnabled("LPS-187854") &&
				!(objectValidationRuleSetting.compareName(
					ObjectValidationRuleSettingConstants.
						NAME_COMPOSITE_KEY_OBJECT_FIELD_ID) ||
				  objectValidationRuleSetting.compareName(
					  ObjectValidationRuleSettingConstants.
						  NAME_OUTPUT_OBJECT_FIELD_ID))) {

				throw new ObjectValidationRuleSettingNameException.
					NotAllowedName(objectValidationRuleSetting.getName());
			}

			if (!FeatureFlagManagerUtil.isEnabled("LPS-187854") &&
				(StringUtil.equals(
					outputType,
					ObjectValidationRuleConstants.
						OUTPUT_TYPE_FULL_VALIDATION) ||
				 !objectValidationRuleSetting.compareName(
					 ObjectValidationRuleSettingConstants.
						 NAME_OUTPUT_OBJECT_FIELD_ID))) {

				throw new ObjectValidationRuleSettingNameException.
					NotAllowedName(objectValidationRuleSetting.getName());
			}

			ObjectField objectField = _objectFieldPersistence.fetchByPrimaryKey(
				GetterUtil.getLong(objectValidationRuleSetting.getValue()));

			if ((objectField == null) || objectField.isSystem()) {
				throw new ObjectValidationRuleSettingValueException.
					InvalidValue(
						objectValidationRuleSetting.getName(),
						objectValidationRuleSetting.getValue());
			}
		}
	}

	private void _validateInvokerBundle(String message, boolean system)
		throws PortalException {

		if (!system || ObjectDefinitionUtil.isInvokerBundleAllowed()) {
			return;
		}

		throw new ObjectValidationRuleSystemException(message);
	}

	private static final Log _log = LogFactoryUtil.getLog(
		ObjectValidationRuleLocalServiceImpl.class);

	@Reference
	private DDMExpressionFactory _ddmExpressionFactory;

	@Reference
	private DTOConverterRegistry _dtoConverterRegistry;

	@Reference
	private Language _language;

	@Reference
	private ObjectDefinitionPersistence _objectDefinitionPersistence;

	@Reference
	private ObjectFieldPersistence _objectFieldPersistence;

	@Reference
	private ObjectScriptingValidator _objectScriptingValidator;

	@Reference
	private ObjectValidationRuleEngineRegistry
		_objectValidationRuleEngineRegistry;

	@Reference
	private ObjectValidationRuleSettingLocalService
		_objectValidationRuleSettingLocalService;

	@Reference
	private ObjectValidationRuleSettingPersistence
		_objectValidationRuleSettingPersistence;

	@Reference
	private SystemObjectDefinitionManagerRegistry
		_systemObjectDefinitionManagerRegistry;

	@Reference
	private UserLocalService _userLocalService;

}