/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.oauth2.provider.service.impl;

import com.liferay.oauth2.provider.model.OAuth2ApplicationScopeAliases;
import com.liferay.oauth2.provider.model.OAuth2ScopeGrant;
import com.liferay.oauth2.provider.scope.liferay.LiferayOAuth2Scope;
import com.liferay.oauth2.provider.scope.liferay.ScopeLocator;
import com.liferay.oauth2.provider.service.base.OAuth2ApplicationScopeAliasesLocalServiceBaseImpl;
import com.liferay.petra.string.StringPool;
import com.liferay.portal.kernel.dao.orm.QueryUtil;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.util.ListUtil;
import com.liferay.portal.kernel.util.OrderByComparator;
import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.spring.extender.service.ServiceReference;

import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import org.osgi.framework.Bundle;

/**
 * @author Brian Wing Shun Chan
 */
public class OAuth2ApplicationScopeAliasesLocalServiceImpl
	extends OAuth2ApplicationScopeAliasesLocalServiceBaseImpl {

	@Override
	public OAuth2ApplicationScopeAliases addOAuth2ApplicationScopeAliases(
			long companyId, long userId, String userName,
			long oAuth2ApplicationId, List<String> scopeAliasesList)
		throws PortalException {

		long oAuth2ApplicationScopeAliasesId = counterLocalService.increment(
			OAuth2ApplicationScopeAliases.class.getName());

		OAuth2ApplicationScopeAliases oAuth2ApplicationScopeAliases =
			createOAuth2ApplicationScopeAliases(
				oAuth2ApplicationScopeAliasesId);

		oAuth2ApplicationScopeAliases.setCompanyId(companyId);
		oAuth2ApplicationScopeAliases.setUserId(userId);
		oAuth2ApplicationScopeAliases.setUserName(userName);
		oAuth2ApplicationScopeAliases.setCreateDate(new Date());
		oAuth2ApplicationScopeAliases.setOAuth2ApplicationId(
			oAuth2ApplicationId);
		oAuth2ApplicationScopeAliases.setScopeAliasesList(scopeAliasesList);

		oAuth2ApplicationScopeAliases =
			oAuth2ApplicationScopeAliasesPersistence.update(
				oAuth2ApplicationScopeAliases);

		createScopeGrants(
			companyId, oAuth2ApplicationScopeAliasesId, scopeAliasesList);

		return oAuth2ApplicationScopeAliases;
	}

	@Override
	public OAuth2ApplicationScopeAliases deleteOAuth2ApplicationScopeAliases(
			long oAuth2ApplicationScopeAliasesId)
		throws PortalException {

		Collection<OAuth2ScopeGrant> oAuth2ScopeGrants =
			oAuth2ScopeGrantLocalService.getOAuth2ScopeGrants(
				oAuth2ApplicationScopeAliasesId, QueryUtil.ALL_POS,
				QueryUtil.ALL_POS, null);

		for (OAuth2ScopeGrant oAuth2ScopeGrant : oAuth2ScopeGrants) {
			oAuth2ScopeGrantLocalService.deleteOAuth2ScopeGrant(
				oAuth2ScopeGrant.getOAuth2ScopeGrantId());
		}

		return oAuth2ApplicationScopeAliasesPersistence.remove(
			oAuth2ApplicationScopeAliasesId);
	}

	@Override
	public OAuth2ApplicationScopeAliases fetchOAuth2ApplicationScopeAliases(
		long oAuth2ApplicationId, List<String> scopeAliasesList) {

		String scopeAliases = StringUtil.merge(
			ListUtil.sort(
				ListUtil.filter(scopeAliasesList, Validator::isNotNull)),
			StringPool.SPACE);

		List<OAuth2ApplicationScopeAliases> oAuth2ApplicationScopeAliasesList =
			oAuth2ApplicationScopeAliasesPersistence.findByO_S(
				oAuth2ApplicationId, scopeAliases.hashCode());

		for (OAuth2ApplicationScopeAliases oAuth2ApplicationScopeAliases :
				oAuth2ApplicationScopeAliasesList) {

			if (scopeAliases.equals(
					oAuth2ApplicationScopeAliases.getScopeAliases())) {

				return oAuth2ApplicationScopeAliases;
			}
		}

		return null;
	}

	@Override
	public List<OAuth2ApplicationScopeAliases>
		getOAuth2ApplicationScopeAliaseses(
			long oAuth2ApplicationId, int start, int end,
			OrderByComparator<OAuth2ApplicationScopeAliases>
				orderByComparator) {

		return oAuth2ApplicationScopeAliasesPersistence.
			findByOAuth2ApplicationId(
				oAuth2ApplicationId, start, end, orderByComparator);
	}

	@Override
	public OAuth2ApplicationScopeAliases updateOAuth2ApplicationScopeAliases(
		OAuth2ApplicationScopeAliases oAuth2ApplicationScopeAliases) {

		if (_hasUpToDateScopeGrants(oAuth2ApplicationScopeAliases)) {
			return oAuth2ApplicationScopeAliases;
		}

		try {
			return addOAuth2ApplicationScopeAliases(
				oAuth2ApplicationScopeAliases.getCompanyId(),
				oAuth2ApplicationScopeAliases.getUserId(),
				oAuth2ApplicationScopeAliases.getUserName(),
				oAuth2ApplicationScopeAliases.getOAuth2ApplicationId(),
				oAuth2ApplicationScopeAliases.getScopeAliasesList());
		}
		catch (PortalException pe) {
			throw new IllegalArgumentException(pe);
		}
	}

	protected void createScopeGrants(
			long companyId, long oAuth2ApplicationScopeAliasesId,
			List<String> scopeAliasesList)
		throws PortalException {

		Set<LiferayOAuth2Scope> liferayOAuth2Scopes = new HashSet<>();

		for (String scopeAlias : scopeAliasesList) {
			liferayOAuth2Scopes.addAll(
				_scopeLocator.getLiferayOAuth2Scopes(companyId, scopeAlias));
		}

		for (LiferayOAuth2Scope liferayOAuth2Scope : liferayOAuth2Scopes) {
			Bundle bundle = liferayOAuth2Scope.getBundle();

			oAuth2ScopeGrantLocalService.createOAuth2ScopeGrant(
				companyId, oAuth2ApplicationScopeAliasesId,
				liferayOAuth2Scope.getApplicationName(),
				bundle.getSymbolicName(), liferayOAuth2Scope.getScope());
		}
	}

	private boolean _hasUpToDateScopeGrants(
		OAuth2ApplicationScopeAliases oAuth2ApplicationScopeAliases) {

		List<String> scopeAliasesList =
			oAuth2ApplicationScopeAliases.getScopeAliasesList();

		Collection<LiferayOAuth2Scope> liferayOAuth2Scopes = new HashSet<>();

		for (String scopeAlias : scopeAliasesList) {
			liferayOAuth2Scopes.addAll(
				_scopeLocator.getLiferayOAuth2Scopes(
					oAuth2ApplicationScopeAliases.getCompanyId(), scopeAlias));
		}

		Collection<OAuth2ScopeGrant> oAuth2ScopeGrants =
			oAuth2ScopeGrantLocalService.getOAuth2ScopeGrants(
				oAuth2ApplicationScopeAliases.
					getOAuth2ApplicationScopeAliasesId(),
				QueryUtil.ALL_POS, QueryUtil.ALL_POS, null);

		if (liferayOAuth2Scopes.size() != oAuth2ScopeGrants.size()) {
			return false;
		}

		for (OAuth2ScopeGrant oAuth2ScopeGrant : oAuth2ScopeGrants) {
			boolean found = liferayOAuth2Scopes.removeIf(
				liferayOAuth2Scope -> {
					Bundle bundle = liferayOAuth2Scope.getBundle();

					if (Objects.equals(
							liferayOAuth2Scope.getApplicationName(),
							oAuth2ScopeGrant.getApplicationName()) &&
						Objects.equals(
							bundle.getSymbolicName(),
							oAuth2ScopeGrant.getBundleSymbolicName()) &&
						Objects.equals(
							liferayOAuth2Scope.getScope(),
							oAuth2ScopeGrant.getScope())) {

						return true;
					}

					return false;
				});

			if (!found) {
				return false;
			}
		}

		return true;
	}

	@ServiceReference(type = ScopeLocator.class)
	private ScopeLocator _scopeLocator;

}