/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.gradle.plugins.workspace.task;

import de.undercouch.gradle.tasks.download.Verify;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import org.gradle.api.GradleException;
import org.gradle.api.tasks.TaskAction;

/**
 * @author Simon Jiang
 */
public class VerifyBundleTask extends Verify {

	@Override
	@TaskAction
	public void verify() throws IOException, NoSuchAlgorithmException {
		if (getSrc() == null) {
			throw new IllegalArgumentException(
				"Please provide a file to verify");
		}

		if (getAlgorithm() == null) {
			throw new IllegalArgumentException(
				"Please provide the algorithm to use to calculate the " +
					"checksum");
		}

		if (getChecksum() == null) {
			throw new IllegalArgumentException(
				"Please provide a checksum to verify against");
		}

		String calculatedChecksum = null;
		MessageDigest messageDigest = MessageDigest.getInstance(getAlgorithm());

		try (FileInputStream fileInputStream = new FileInputStream(getSrc())) {
			byte[] bytes = new byte[1024];
			int read = 0;

			while ((read = fileInputStream.read(bytes)) != -1) {
				messageDigest.update(bytes, 0, read);
			}

			calculatedChecksum = _toHex(messageDigest.digest());
		}

		if (!calculatedChecksum.equalsIgnoreCase(getChecksum())) {
			File srcFile = getSrc();

			throw new GradleException(
				"Invalid checksum for " + srcFile.getName() + ". Expected " +
					getChecksum().toLowerCase() + ", but got " +
						calculatedChecksum.toLowerCase() + ". Please remove " +
							srcFile.getAbsolutePath() + " and try again.");
		}
	}

	private String _toHex(byte[] bytes) {
		StringBuilder stringBuilder = new StringBuilder();

		for (byte b : bytes) {
			stringBuilder.append(String.format("%02X", b));
		}

		return stringBuilder.toString();
	}

}