/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.gradle.plugins.workspace.configurators;

import com.liferay.gradle.plugins.LiferayThemePlugin;
import com.liferay.gradle.plugins.extensions.LiferayExtension;
import com.liferay.gradle.plugins.theme.builder.BuildThemeTask;
import com.liferay.gradle.plugins.theme.builder.ThemeBuilderPlugin;
import com.liferay.gradle.plugins.workspace.ProjectConfigurator;
import com.liferay.gradle.plugins.workspace.WorkspaceExtension;
import com.liferay.gradle.plugins.workspace.WorkspacePlugin;
import com.liferay.gradle.plugins.workspace.internal.util.GradleUtil;

import groovy.json.JsonSlurper;

import groovy.lang.Closure;

import java.io.File;
import java.io.IOException;

import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;

import org.gradle.api.Project;
import org.gradle.api.Task;
import org.gradle.api.file.ConfigurableFileCollection;
import org.gradle.api.file.CopySpec;
import org.gradle.api.initialization.Settings;
import org.gradle.api.plugins.BasePlugin;
import org.gradle.api.plugins.BasePluginConvention;
import org.gradle.api.plugins.ExtensionAware;
import org.gradle.api.plugins.WarPluginConvention;
import org.gradle.api.tasks.Copy;

/**
 * @author Andrea Di Giorgi
 * @author David Truong
 */
public class ThemesProjectConfigurator extends BaseProjectConfigurator {

	public ThemesProjectConfigurator(Settings settings) {
		super(settings);

		_javaBuild = GradleUtil.getProperty(
			settings, WorkspacePlugin.PROPERTY_PREFIX + NAME + ".java.build",
			_JAVA_BUILD);
	}

	@Override
	public void apply(Project project) {
		WorkspaceExtension workspaceExtension = GradleUtil.getExtension(
			(ExtensionAware)project.getGradle(), WorkspaceExtension.class);

		if (isJavaBuild()) {
			ProjectConfigurator projectConfigurator =
				workspaceExtension.propertyMissing(
					WarsProjectConfigurator.NAME);

			projectConfigurator.apply(project);

			GradleUtil.applyPlugin(project, ThemeBuilderPlugin.class);

			_configureTaskBuildTheme(project);
			_configureWar(project);
		}
		else {
			GradleUtil.applyPlugin(project, LiferayThemePlugin.class);

			_configureLiferay(project, workspaceExtension);

			final Task assembleTask = GradleUtil.getTask(
				project, BasePlugin.ASSEMBLE_TASK_NAME);

			_configureRootTaskDistBundle(assembleTask);

			Callable<ConfigurableFileCollection> warSourcePath =
				new Callable<ConfigurableFileCollection>() {

					@Override
					public ConfigurableFileCollection call() throws Exception {
						Project project = assembleTask.getProject();

						ConfigurableFileCollection configurableFileCollection =
							project.files(_getWarFile(project));

						return configurableFileCollection.builtBy(assembleTask);
					}

				};

			addTaskDockerDeploy(project, warSourcePath, workspaceExtension);
		}
	}

	@Override
	public String getName() {
		return NAME;
	}

	public boolean isJavaBuild() {
		return _javaBuild;
	}

	public void setJavaBuild(boolean javaBuild) {
		_javaBuild = javaBuild;
	}

	@Override
	protected Iterable<File> doGetProjectDirs(File rootDir) throws Exception {
		final Set<File> projectDirs = new HashSet<>();

		Files.walkFileTree(
			rootDir.toPath(),
			new SimpleFileVisitor<Path>() {

				@Override
				public FileVisitResult preVisitDirectory(
						Path dirPath, BasicFileAttributes basicFileAttributes)
					throws IOException {

					Path dirNamePath = dirPath.getFileName();

					String dirName = dirNamePath.toString();

					if (dirName.equals("build") ||
						dirName.equals("build_gradle") ||
						dirName.equals("node_modules") ||
						dirName.equals("node_modules_cache")) {

						return FileVisitResult.SKIP_SUBTREE;
					}

					if (Files.exists(dirPath.resolve("package.json"))) {
						projectDirs.add(dirPath.toFile());

						return FileVisitResult.SKIP_SUBTREE;
					}

					return FileVisitResult.CONTINUE;
				}

			});

		return projectDirs;
	}

	protected static final String NAME = "themes";

	private void _configureLiferay(
		Project project, WorkspaceExtension workspaceExtension) {

		LiferayExtension liferayExtension = GradleUtil.getExtension(
			project, LiferayExtension.class);

		liferayExtension.setAppServerParentDir(workspaceExtension.getHomeDir());
	}

	@SuppressWarnings({"serial", "unused"})
	private void _configureRootTaskDistBundle(final Task assembleTask) {
		Project project = assembleTask.getProject();

		Copy copy = (Copy)GradleUtil.getTask(
			project.getRootProject(),
			RootProjectConfigurator.DIST_BUNDLE_TASK_NAME);

		copy.dependsOn(assembleTask);

		copy.into(
			"osgi/war",
			new Closure<Void>(project) {

				public void doCall(final CopySpec copySpec) {
					Project project = assembleTask.getProject();

					File warFile = _getWarFile(project);

					ConfigurableFileCollection configurableFileCollection =
						project.files(warFile);

					configurableFileCollection.builtBy(assembleTask);

					copySpec.from(warFile);
				}

			});
	}

	private void _configureTaskBuildTheme(Project project) {
		File packageJsonFile = project.file("package.json");

		if (!packageJsonFile.exists()) {
			return;
		}

		BuildThemeTask buildThemeTask = (BuildThemeTask)GradleUtil.getTask(
			project, ThemeBuilderPlugin.BUILD_THEME_TASK_NAME);

		JsonSlurper jsonSlurper = new JsonSlurper();

		Map<String, Object> packageJsonMap =
			(Map<String, Object>)jsonSlurper.parse(packageJsonFile);

		Map<String, String> liferayThemeMap =
			(Map<String, String>)packageJsonMap.get("liferayTheme");

		String baseTheme = liferayThemeMap.get("baseTheme");

		if (baseTheme.equals("styled") || baseTheme.equals("unstyled")) {
			baseTheme = "_" + baseTheme;
		}

		String templateLanguage = liferayThemeMap.get("templateLanguage");

		buildThemeTask.setParentName(baseTheme);
		buildThemeTask.setTemplateExtension(templateLanguage);
	}

	private void _configureWar(Project project) {
		WarPluginConvention warPluginConvention = GradleUtil.getConvention(
			project, WarPluginConvention.class);

		warPluginConvention.setWebAppDirName("src");
	}

	private File _getWarFile(Project project) {
		BasePluginConvention basePluginConvention = GradleUtil.getConvention(
			project, BasePluginConvention.class);

		return project.file(
			"dist/" + basePluginConvention.getArchivesBaseName() + ".war");
	}

	private static final boolean _JAVA_BUILD = false;

	private boolean _javaBuild;

}