/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.gradle.plugins.defaults.tasks;

import com.liferay.gradle.plugins.defaults.internal.util.GradleUtil;
import com.liferay.gradle.util.Validator;

import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;

import java.nio.charset.StandardCharsets;
import java.nio.file.Files;

import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.Set;

import org.gradle.api.DefaultTask;
import org.gradle.api.Project;
import org.gradle.api.file.FileCollection;
import org.gradle.api.specs.Spec;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.InputFiles;
import org.gradle.api.tasks.Optional;
import org.gradle.api.tasks.OutputFile;
import org.gradle.api.tasks.SkipWhenEmpty;
import org.gradle.api.tasks.TaskAction;
import org.gradle.util.GUtil;

/**
 * @author Andrea Di Giorgi
 */
public class MergeFilesTask extends DefaultTask {

	@Input
	@Optional
	public String getFooter() {
		return GradleUtil.toString(_footer);
	}

	@Input
	@Optional
	public String getHeader() {
		return GradleUtil.toString(_header);
	}

	@InputFiles
	@SkipWhenEmpty
	public FileCollection getInputFiles() {
		Project project = getProject();

		return project.files(_inputFiles);
	}

	@OutputFile
	public File getOutputFile() {
		return GradleUtil.toFile(getProject(), _outputFile);
	}

	@Input
	public String getSeparator() {
		return GradleUtil.toString(_separator);
	}

	public MergeFilesTask inputFiles(Iterable<?> inputFiles) {
		GUtil.addToCollection(_inputFiles, inputFiles);

		return this;
	}

	public MergeFilesTask inputFiles(Object... inputFiles) {
		return inputFiles(Arrays.asList(inputFiles));
	}

	@TaskAction
	public void mergeFiles() throws IOException {
		FileCollection inputFiles = getInputFiles();

		inputFiles = inputFiles.filter(
			new Spec<File>() {

				@Override
				public boolean isSatisfiedBy(File file) {
					return file.exists();
				}

			});

		File outputFile = getOutputFile();

		if (inputFiles.isEmpty()) {
			outputFile.delete();

			return;
		}

		String separator = getSeparator();

		try (BufferedWriter bufferedWriter = Files.newBufferedWriter(
				outputFile.toPath(), StandardCharsets.UTF_8)) {

			String header = getHeader();

			if (Validator.isNotNull(header)) {
				bufferedWriter.write(header);
			}

			for (File inputFile : inputFiles) {
				bufferedWriter.write(separator);

				String content = new String(
					Files.readAllBytes(inputFile.toPath()),
					StandardCharsets.UTF_8);

				bufferedWriter.write(content);
			}

			String footer = getFooter();

			if (Validator.isNotNull(footer)) {
				bufferedWriter.write(separator);
				bufferedWriter.write(footer);
			}
		}
	}

	public void setFooter(Object footer) {
		_footer = footer;
	}

	public void setHeader(Object header) {
		_header = header;
	}

	public void setInputFiles(Iterable<?> inputFiles) {
		_inputFiles.clear();

		inputFiles(inputFiles);
	}

	public void setInputFiles(Object... inputFiles) {
		setInputFiles(Arrays.asList(inputFiles));
	}

	public void setOutputFile(Object outputFile) {
		_outputFile = outputFile;
	}

	public void setSeparator(Object separator) {
		_separator = separator;
	}

	private Object _footer;
	private Object _header;
	private final Set<Object> _inputFiles = new LinkedHashSet<>();
	private Object _outputFile;
	private Object _separator = System.lineSeparator();

}