/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

import React, {useContext, useRef} from 'react';

const ConfigContext = React.createContext({});

ConfigContext.displayName = 'ConfigContext';

/**
 * A provider to store any configuration or property that has no
 * side effect during the life cycle of the application.
 *
 * Maintaining configuration properties with side effect properties
 * in the same `store` may be rendering unnecessary components
 * that use only configuration properties.
 */
export const ConfigProvider = ({children, initialConfig, ...otherProps}) => {
	const {config = initialConfig} = otherProps;

	// Use `useRef` to avoid causing a new rendering of components that
	// consume context data. We do not want to cause a new rendering after
	// it initializes the app, this data will not change during the life
	// cycle of the application.

	const configRef = useRef({...initialConfig, ...config});

	return (
		<ConfigContext.Provider value={configRef.current}>
			{children}
		</ConfigContext.Provider>
	);
};

ConfigProvider.displayName = 'ConfigProvider';

export const useConfig = () => {
	return useContext(ConfigContext);
};
