/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

import {usePrevious} from 'frontend-js-react-web';
import React, {useEffect, useState} from 'react';

import {EVENT_TYPES} from '../../actions/eventTypes.es';
import {useForm} from '../../hooks/useForm.es';
import {usePage} from '../../hooks/usePage.es';
import {PagesVisitor} from '../../util/visitors.es';

export const PageHeader = ({
	description: initialDescription,
	placeholder,
	title: initialTitle,
}) => {
	const [description, setDescription] = useState(initialDescription);
	const [title, setTitle] = useState(initialTitle);

	const dispatch = useForm();
	const {editingLanguageId, pageIndex, pages} = usePage();

	const prevEditingLanguageId = usePrevious(editingLanguageId);

	useEffect(() => {
		if (prevEditingLanguageId !== editingLanguageId) {
			setDescription(initialDescription);
			setTitle(initialTitle);
		}
	}, [
		editingLanguageId,
		initialDescription,
		initialTitle,
		prevEditingLanguageId,
	]);

	const handlePageDescriptionChanged = (value) => {
		const visitor = new PagesVisitor(pages);

		dispatch({
			payload: visitor.mapPages((page, index) => {
				if (index === pageIndex) {
					page = {
						...page,
						description: value,
						localizedDescription: {
							...page.localizedDescription,
							[editingLanguageId]: value,
						},
					};
				}

				return page;
			}),
			type: EVENT_TYPES.PAGE_UPDATED,
		});
	};

	const handlePageTitleChanged = (value) => {
		const visitor = new PagesVisitor(pages);

		dispatch({
			payload: visitor.mapPages((page, index) => {
				if (index === pageIndex) {
					page = {
						...page,
						localizedTitle: {
							...page.localizedTitle,
							[editingLanguageId]: value,
						},
						title: value,
					};
				}

				return page;
			}),
			type: EVENT_TYPES.PAGE_UPDATED,
		});
	};

	return (
		<div>
			<input
				className="form-builder-page-header-title form-control p-0"
				maxLength="120"
				onChange={(event) => {
					const {value} = event.target;

					setTitle(value);
					handlePageTitleChanged(value);
				}}
				placeholder={placeholder}
				value={title}
			/>
			<input
				className="form-builder-page-header-description form-control p-0"
				maxLength="120"
				onChange={(event) => {
					const {value} = event.target;

					setDescription(value);
					handlePageDescriptionChanged(value);
				}}
				placeholder={Liferay.Language.get(
					'add-a-short-description-for-this-page'
				)}
				value={description}
			/>
		</div>
	);
};
