/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

import {ClayIconSpriteContext} from '@clayui/icon';
import Soy from 'metal-soy';
import React from 'react';

import {PageProvider} from '../../hooks/usePage.es';
import {useFieldTypesResource} from '../../hooks/useResource.es';
import {getConnectedReactComponentAdapter} from '../../util/ReactComponentAdapter.es';
import {Field} from './Field.es';
import templates from './ReactFieldAdapter.soy';

/**
 * This creates a compatibility layer for the Field component on React, allowing
 * it to be called by Metal+Soy file.
 *
 * This component creates a mask for the Field component, some DDM fields access
 * the `usePage` and we need to bring the `usePage` here since this component
 * will not have a React context above it.
 */
export const ReactFieldAdapter = ({fieldType, instance, ...field}) => {
	const {resource: fieldTypes} = useFieldTypesResource();

	if (!fieldType || fieldType === '') {
		return null;
	}

	const emit = (name, event) => {
		instance.emit(name, {
			...event,
			fieldInstance: {
				...event.fieldInstance,
				element: instance.element,
			},
		});
	};

	return (
		<PageProvider value={{...field, fieldTypes}}>
			<ClayIconSpriteContext.Provider value={field.spritemap}>
				<Field
					field={{
						...field,
						type: fieldType,
					}}
					onBlur={(event) => emit('fieldBlurred', event)}
					onChange={(event) => emit('fieldEdited', event)}
					onFocus={(event) => emit('fieldFocused', event)}
				/>
			</ClayIconSpriteContext.Provider>
		</PageProvider>
	);
};

const ReactComponentAdapter = getConnectedReactComponentAdapter(
	ReactFieldAdapter
);

Soy.register(ReactComponentAdapter, templates);

export default ReactComponentAdapter;
