/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

import * as FormSupport from 'dynamic-data-mapping-form-renderer/js/components/FormRenderer/FormSupport.es';
import ClayButton from 'clay-button';
import ClayModal from 'clay-modal';
import Component from 'metal-jsx';
import dom from 'metal-dom';
import {Config} from 'metal-state';
import {EventHandler} from 'metal-events';
import {focusedFieldStructure, pageStructure} from '../../util/config.es';

class Actions extends Component {
	render() {
		const {spritemap} = this.props;

		return (
			<div class="ddm-field-actions-container" ref="actionsContainer">
				<ClayButton
					editable={true}
					events={{
						click: this._handleDuplicateButtonClicked.bind(this)
					}}
					icon="paste"
					monospaced={true}
					size="sm"
					spritemap={spritemap}
					style="secondary"
				/>

				<ClayButton
					editable={true}
					events={{
						click: this._handleDeleteButtonClicked.bind(this)
					}}
					icon="trash"
					monospaced={true}
					size="sm"
					spritemap={spritemap}
					style="secondary"
				/>
			</div>
		);
	}

	_handleDeleteButtonClicked(event) {
		const indexes = FormSupport.getIndexes(
			dom.closest(event.target, '.col-ddm')
		);

		this.emit('fieldDeleted', {indexes});
	}

	_handleDuplicateButtonClicked(event) {
		const indexes = FormSupport.getIndexes(
			dom.closest(event.target, '.col-ddm')
		);

		this.emit('fieldDuplicated', {
			indexes
		});
	}
}

Actions.PROPS = {
	/**
	 * @default undefined
	 * @instance
	 * @memberof Actions
	 * @type {!string}
	 */

	spritemap: Config.string().required()
};

const withActionableFields = ChildComponent => {
	class ActionableFields extends Component {
		attached() {
			this._eventHandler = new EventHandler();

			this._eventHandler.add(
				this.delegate(
					'mouseenter',
					'.ddm-field-container',
					this._handleMouseEnterField.bind(this)
				)
			);
		}

		disposeInternal() {
			super.disposeInternal();

			this._eventHandler.removeAllListeners();
		}

		getEvents() {
			return {
				fieldDeleted: this._handleDeleteRequest.bind(this),
				fieldDuplicated: this._handleDuplicateRequest.bind(this)
			};
		}

		isActionsEnabled() {
			const {defaultLanguageId, editingLanguageId} = this.props;

			return defaultLanguageId === editingLanguageId;
		}

		render() {
			const {spritemap} = this.props;

			return (
				<div>
					<ClayModal
						body={Liferay.Language.get(
							'are-you-sure-you-want-to-delete-this-field'
						)}
						events={{
							clickButton: this._handleDeleteConfirmationModalButtonClicked.bind(
								this
							)
						}}
						footerButtons={[
							{
								alignment: 'right',
								label: Liferay.Language.get('dismiss'),
								style: 'primary',
								type: 'close'
							},
							{
								alignment: 'right',
								label: Liferay.Language.get('delete'),
								style: 'primary',
								type: 'button'
							}
						]}
						ref="deleteModal"
						size="sm"
						spritemap={spritemap}
						title={Liferay.Language.get(
							'delete-field-dialog-title'
						)}
					/>

					<ChildComponent {...this.props} events={this.getEvents()} />

					{this.isActionsEnabled() && (
						<Actions
							events={this.getEvents()}
							portalElement={this.element}
							ref="actions"
							spritemap={spritemap}
						/>
					)}
				</div>
			);
		}

		showDeleteConfirmationModal() {
			const {deleteModal} = this.refs;

			deleteModal.show();
		}

		_handleDeleteConfirmationModalButtonClicked(event) {
			const {store} = this.context;
			const {target} = event;
			const {deleteModal} = this.refs;
			const {indexes} = this.state;

			event.stopPropagation();

			deleteModal.emit('hide');

			if (!target.classList.contains('close-modal')) {
				store.emit('fieldDeleted', {indexes});
			}
		}

		_handleDeleteRequest({indexes}) {
			this.setState({
				indexes
			});

			this.showDeleteConfirmationModal();
		}

		_handleDuplicateRequest(indexes) {
			this._handleFieldDuplicated(indexes);
		}

		_handleFieldDuplicated(indexes) {
			const {store} = this.context;

			store.emit('fieldDuplicated', indexes);
		}

		_handleMouseEnterField({delegateTarget}) {
			if (this.isActionsEnabled()) {
				dom.append(delegateTarget, this.refs.actions.element);
			}
		}
	}

	ActionableFields.STATE = {
		/**
		 * @default undefined
		 * @instance
		 * @memberof ActionableFields
		 * @type {?array<string>}
		 */

		indexes: Config.object()
	};

	ActionableFields.PROPS = {
		/**
		 * @default
		 * @instance
		 * @memberof FormBuilder
		 * @type {?number}
		 */

		activePage: Config.number().value(0),

		/**
		 * @default undefined
		 * @instance
		 * @memberof FormBuilder
		 * @type {?string}
		 */

		defaultLanguageId: Config.string(),

		/**
		 * @default undefined
		 * @instance
		 * @memberof FormBuilder
		 * @type {?string}
		 */

		editingLanguageId: Config.string(),

		/**
		 * @default undefined
		 * @instance
		 * @memberof FormBuilder
		 * @type {?string}
		 */

		fieldSetDefinitionURL: Config.string(),

		/**
		 * @default []
		 * @instance
		 * @memberof FormBuilder
		 * @type {?(array|undefined)}
		 */

		fieldSets: Config.array().value([]),

		/**
		 * @default []
		 * @instance
		 * @memberof FormBuilder
		 * @type {?(array|undefined)}
		 */

		fieldTypes: Config.array().value([]),

		/**
		 * @default {}
		 * @instance
		 * @memberof FormBuilder
		 * @type {?object}
		 */

		focusedField: focusedFieldStructure.value({}),

		/**
		 * @default []
		 * @instance
		 * @memberof FormBuilder
		 * @type {?array<object>}
		 */

		pages: Config.arrayOf(pageStructure).value([]),

		/**
		 * @instance
		 * @memberof FormBuilder
		 * @type {string}
		 */

		paginationMode: Config.string().required(),

		/**
		 * @instance
		 * @memberof FormBuilder
		 * @type {string}
		 */

		portletNamespace: Config.string().required(),

		/**
		 * @default undefined
		 * @instance
		 * @memberof FormRenderer
		 * @type {!string}
		 */

		spritemap: Config.string().required(),

		/**
		 * @instance
		 * @memberof FormBuilder
		 * @type {object}
		 */

		successPageSettings: Config.shapeOf({
			body: Config.object(),
			enabled: Config.bool(),
			title: Config.object()
		}).value({}),

		/**
		 * @default undefined
		 * @instance
		 * @memberof FormBuilder
		 * @type {?string}
		 */

		view: Config.string()
	};

	return ActionableFields;
};

export default withActionableFields;
