/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.change.tracking.store.internal;

import com.liferay.change.tracking.constants.CTConstants;
import com.liferay.change.tracking.model.CTEntry;
import com.liferay.change.tracking.service.CTEntryLocalService;
import com.liferay.change.tracking.spi.exception.CTEventException;
import com.liferay.change.tracking.spi.listener.CTEventListener;
import com.liferay.change.tracking.store.model.CTSContent;
import com.liferay.change.tracking.store.service.CTSContentLocalService;
import com.liferay.document.library.kernel.store.Store;
import com.liferay.osgi.service.tracker.collections.map.ServiceTrackerMap;
import com.liferay.osgi.service.tracker.collections.map.ServiceTrackerMapFactory;
import com.liferay.petra.lang.SafeCloseable;
import com.liferay.portal.change.tracking.sql.CTSQLModeThreadLocal;
import com.liferay.portal.kernel.change.tracking.CTCollectionThreadLocal;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.service.ClassNameLocalService;

import java.util.ArrayList;
import java.util.List;

import org.osgi.framework.BundleContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Shuyang Zhou
 */
@Component(service = CTEventListener.class)
public class CTStoreCTEventListener implements CTEventListener {

	@Override
	public void onAfterPublish(long ctCollectionId) throws CTEventException {
		List<CTEntry> ctEntries = _ctEntryLocalService.getCTEntries(
			ctCollectionId, _ctsContentClassNameId);

		if (ctEntries.isEmpty()) {
			return;
		}

		List<CTEntry> deletedCTEnties = new ArrayList<>();

		List<CTEntry> addOrModifiedCTEntries = new ArrayList<>();

		for (CTEntry ctEntry : ctEntries) {
			if (ctEntry.getChangeType() ==
					CTConstants.CT_CHANGE_TYPE_DELETION) {

				deletedCTEnties.add(ctEntry);
			}
			else {
				addOrModifiedCTEntries.add(ctEntry);
			}
		}

		// Deleted CTEntries need to read CTSContent from CTCollection

		if (!deletedCTEnties.isEmpty()) {
			try (SafeCloseable safeCloseable1 =
					CTSQLModeThreadLocal.setCTSQLModeWithSafeCloseable(
						CTSQLModeThreadLocal.CTSQLMode.CT_ONLY);
				SafeCloseable safeCloseable2 =
					CTCollectionThreadLocal.setCTCollectionIdWithSafeCloseable(
						ctCollectionId)) {

				for (CTEntry ctEntry : deletedCTEnties) {
					CTSContent ctsContent =
						_ctsContentLocalService.getCTSContent(
							ctEntry.getModelClassPK());

					Store store = _storeServiceTrackerMap.getService(
						ctsContent.getStoreType());

					store.deleteFile(
						ctsContent.getCompanyId(), ctsContent.getRepositoryId(),
						ctsContent.getPath(), ctsContent.getVersion());
				}
			}
			catch (PortalException portalException) {
				throw new CTEventException(portalException);
			}
		}

		// Add or modifed CTEntries need to read CTSContent from production

		if (!addOrModifiedCTEntries.isEmpty()) {
			try (SafeCloseable safeCloseable =
					CTCollectionThreadLocal.
						setProductionModeWithSafeCloseable()) {

				for (CTEntry ctEntry : addOrModifiedCTEntries) {
					CTSContent ctsContent =
						_ctsContentLocalService.getCTSContent(
							ctEntry.getModelClassPK());

					Store store = _storeServiceTrackerMap.getService(
						ctsContent.getStoreType());

					store.addFile(
						ctsContent.getCompanyId(), ctsContent.getRepositoryId(),
						ctsContent.getPath(), ctsContent.getVersion(),
						_ctsContentLocalService.openDataInputStream(
							ctsContent.getCtsContentId()));
				}
			}
			catch (PortalException portalException) {
				throw new CTEventException(portalException);
			}
		}
	}

	@Override
	public void onBeforeRemove(long ctCollectionId) throws CTEventException {
		List<Long> ctsContentIds =
			_ctEntryLocalService.getExclusiveModelClassPKs(
				ctCollectionId, _ctsContentClassNameId);

		if (ctsContentIds.isEmpty()) {
			return;
		}

		try (SafeCloseable safeCloseable =
				CTCollectionThreadLocal.setProductionModeWithSafeCloseable()) {

			for (long ctsContentId : ctsContentIds) {
				CTSContent ctsContent = _ctsContentLocalService.fetchCTSContent(
					ctsContentId);

				if (ctsContent != null) {
					_ctsContentLocalService.deleteCTSContent(ctsContent);
				}
			}
		}
	}

	@Activate
	protected void activate(BundleContext bundleContext) {
		_ctsContentClassNameId = _classNameLocalService.getClassNameId(
			CTSContent.class);

		_storeServiceTrackerMap = ServiceTrackerMapFactory.openSingleValueMap(
			bundleContext, Store.class, "store.type");
	}

	@Reference
	private ClassNameLocalService _classNameLocalService;

	@Reference
	private CTEntryLocalService _ctEntryLocalService;

	private long _ctsContentClassNameId;

	@Reference
	private CTSContentLocalService _ctsContentLocalService;

	private ServiceTrackerMap<String, Store> _storeServiceTrackerMap;

}