/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.change.tracking.internal;

import com.liferay.petra.string.StringBundler;
import com.liferay.portal.kernel.dao.db.DBManagerUtil;
import com.liferay.portal.kernel.dao.db.DBType;
import com.liferay.portal.kernel.service.persistence.change.tracking.CTPersistence;

import java.sql.Blob;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

/**
 * @author Preston Crary
 */
public class CTRowUtil {

	public static int copyCTRows(
			CTPersistence<?> ctPersistence, Connection connection,
			String selectSQL)
		throws SQLException {

		Map<String, Integer> tableColumnsMap =
			ctPersistence.getTableColumnsMap();

		if (_isPostgresBlobTable(tableColumnsMap)) {
			StringBundler sb = new StringBundler(
				(3 * tableColumnsMap.size()) + 4);

			sb.append("insert into ");
			sb.append(ctPersistence.getTableName());
			sb.append(" (");

			for (String columnName : tableColumnsMap.keySet()) {
				sb.append(columnName);
				sb.append(", ");
			}

			sb.setStringAt(") values (?", sb.index() - 1);

			for (int i = 1; i < tableColumnsMap.size(); i++) {
				sb.append(", ?");
			}

			sb.append(")");

			try (PreparedStatement selectPreparedStatement =
					connection.prepareStatement(selectSQL);
				PreparedStatement insertPreparedStatement =
					connection.prepareStatement(sb.toString());
				ResultSet resultSet = selectPreparedStatement.executeQuery()) {

				while (resultSet.next()) {
					int parameterIndex = 1;

					for (int type : tableColumnsMap.values()) {
						if (type == Types.BLOB) {
							Blob blob = resultSet.getBlob(parameterIndex);

							insertPreparedStatement.setBlob(
								parameterIndex, blob.getBinaryStream());
						}
						else {
							insertPreparedStatement.setObject(
								parameterIndex,
								resultSet.getObject(parameterIndex));
						}

						parameterIndex++;
					}

					insertPreparedStatement.addBatch();
				}

				int result = 0;

				for (int count : insertPreparedStatement.executeBatch()) {
					result += count;
				}

				return result;
			}
		}

		StringBundler sb = new StringBundler((2 * tableColumnsMap.size()) + 4);

		sb.append("insert into ");
		sb.append(ctPersistence.getTableName());
		sb.append(" (");

		for (String name : tableColumnsMap.keySet()) {
			sb.append(name);
			sb.append(", ");
		}

		sb.setStringAt(") ", sb.index() - 1);

		sb.append(selectSQL);

		try (PreparedStatement preparedStatement = connection.prepareStatement(
				sb.toString())) {

			return preparedStatement.executeUpdate();
		}
	}

	public static String getConstraintConflictsSQL(
		String tableName, String primaryColumnName,
		String[] uniqueIndexColumnNames, long targetCTCollectionId) {

		StringBundler sb = new StringBundler(
			(9 * uniqueIndexColumnNames.length) + 17);

		sb.append("select ");
		sb.append(primaryColumnName);
		sb.append(" from ");
		sb.append(tableName);
		sb.append(" where ctCollectionId = ");
		sb.append(targetCTCollectionId);
		sb.append(" and ");
		sb.append(primaryColumnName);
		sb.append(" != ?");

		for (String uniqueIndexColumnName : uniqueIndexColumnNames) {
			sb.append(" and ");
			sb.append(uniqueIndexColumnName);
			sb.append(" = ?");
		}

		return sb.toString();
	}

	public static String getConstraintEntitiesSQL(
		String tableName, String primaryColumnName,
		String[] uniqueIndexColumnNames, long ctCollectionId,
		Set<Long> primaryKeys) {

		StringBundler sb = new StringBundler(
			(9 * uniqueIndexColumnNames.length) + 17);

		sb.append("select ");
		sb.append(primaryColumnName);
		sb.append(", ");

		for (String uniqueIndexColumnName : uniqueIndexColumnNames) {
			sb.append(uniqueIndexColumnName);
			sb.append(", ");
		}

		sb.setIndex(sb.index() - 1);

		sb.append(" from ");
		sb.append(tableName);
		sb.append(" where ctCollectionId = ");
		sb.append(ctCollectionId);
		sb.append(" and (");

		for (long primaryKey : primaryKeys) {
			sb.append("(");
			sb.append(primaryColumnName);
			sb.append(" = ");
			sb.append(primaryKey);
			sb.append(")");
			sb.append(" or ");
		}

		sb.setIndex(sb.index() - 1);

		sb.append(")");

		return sb.toString();
	}

	private static boolean _isPostgresBlobTable(
		Map<String, Integer> tableColumnsMap) {

		if (DBManagerUtil.getDBType() != DBType.POSTGRESQL) {
			return false;
		}

		Collection<Integer> values = tableColumnsMap.values();

		if (values.contains(Types.BLOB)) {
			return true;
		}

		return false;
	}

	private CTRowUtil() {
	}

}