/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.change.tracking.internal.model.listener;

import com.liferay.change.tracking.internal.background.task.CTPublishBackgroundTaskExecutor;
import com.liferay.change.tracking.model.CTCollection;
import com.liferay.change.tracking.model.CTProcess;
import com.liferay.change.tracking.service.CTCollectionLocalService;
import com.liferay.change.tracking.service.CTSchemaVersionLocalService;
import com.liferay.portal.background.task.model.BackgroundTask;
import com.liferay.portal.kernel.backgroundtask.constants.BackgroundTaskConstants;
import com.liferay.portal.kernel.exception.ModelListenerException;
import com.liferay.portal.kernel.model.BaseModelListener;
import com.liferay.portal.kernel.model.ModelListener;
import com.liferay.portal.kernel.search.Indexer;
import com.liferay.portal.kernel.search.IndexerRegistryUtil;
import com.liferay.portal.kernel.search.SearchException;
import com.liferay.portal.kernel.util.MapUtil;
import com.liferay.portal.kernel.workflow.WorkflowConstants;

import java.util.Objects;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Preston Crary
 */
@Component(service = ModelListener.class)
public class BackgroundTaskModelListener
	extends BaseModelListener<BackgroundTask> {

	@Override
	public void onAfterUpdate(
		BackgroundTask originalBackgroundTask, BackgroundTask backgroundTask) {

		if ((backgroundTask == null) ||
			!Objects.equals(
				backgroundTask.getTaskExecutorClassName(),
				CTPublishBackgroundTaskExecutor.class.getName())) {

			return;
		}

		if ((backgroundTask.getStatus() ==
				BackgroundTaskConstants.STATUS_CANCELLED) ||
			(backgroundTask.getStatus() ==
				BackgroundTaskConstants.STATUS_FAILED)) {

			String name = backgroundTask.getName();

			if (name.contains("_")) {
				return;
			}

			CTCollection ctCollection =
				_ctCollectionLocalService.fetchCTCollection(
					Long.valueOf(backgroundTask.getName()));

			if (ctCollection != null) {
				int status = WorkflowConstants.STATUS_DRAFT;

				if (!_ctSchemaVersionLocalService.isLatestCTSchemaVersion(
						ctCollection.getSchemaVersionId())) {

					status = WorkflowConstants.STATUS_EXPIRED;
				}

				ctCollection.setStatus(status);

				_ctCollectionLocalService.updateCTCollection(ctCollection);
			}
		}

		if (backgroundTask.getStatus() != originalBackgroundTask.getStatus()) {
			_reindexCTProcess(backgroundTask);
		}
	}

	private void _reindexCTProcess(BackgroundTask backgroundTask) {
		long ctProcessId = MapUtil.getLong(
			backgroundTask.getTaskContextMap(), "ctProcessId");

		if (ctProcessId <= 0) {
			return;
		}

		try {
			Indexer<CTProcess> indexer = IndexerRegistryUtil.nullSafeGetIndexer(
				CTProcess.class);

			indexer.reindex(CTProcess.class.getName(), ctProcessId);
		}
		catch (SearchException searchException) {
			throw new ModelListenerException(searchException);
		}
	}

	@Reference
	private CTCollectionLocalService _ctCollectionLocalService;

	@Reference
	private CTSchemaVersionLocalService _ctSchemaVersionLocalService;

}