/**
 * SPDX-FileCopyrightText: (c) 2023 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.blade.cli.gradle;

import com.liferay.blade.cli.LiferayBundleDeployer;
import com.liferay.gogo.shell.client.GogoShellClient;

import java.io.IOException;

import java.net.URI;

import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;

import java.util.Collection;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.osgi.framework.Bundle;
import org.osgi.framework.dto.BundleDTO;

/**
 * @author Christopher Bryan Boyd
 * @author Gregory Amerson
 */
public class LiferayBundleDeployerImpl implements LiferayBundleDeployer {

	public LiferayBundleDeployerImpl(final String host, final int port) throws IOException {
		_client = new GogoShellClient(host, port);
	}

	@Override
	public void close() throws Exception {
		if (_client != null) {
			_client.close();
		}
	}

	@Override
	public BundleDTO getBundle(long id) throws Exception {
		String response = _client.send("lb -s -u | grep '" + id + "\\|'");

		String[] lines = _parseGogoResponse(response);

		return Stream.of(
			lines
		).skip(
			1
		).limit(
			1
		).map(
			LiferayBundleDeployerImpl::_parseGogoLine
		).filter(
			x -> x.id == id
		).findAny(
		).orElseThrow(
			() -> new NoSuchElementException("No bundle matching the specified ID " + id)
		);
	}

	@Override
	public long getBundleId(Collection<BundleDTO> bundles, String bsn) throws Exception {
		long bundleId;

		if (Objects.nonNull(bsn)) {
			Stream<BundleDTO> stream = bundles.stream();

			bundleId = stream.filter(
				Objects::nonNull
			).filter(
				bundle -> Objects.equals(bundle.symbolicName, bsn)
			).map(
				bundle -> bundle.id
			).findAny(
			).orElse(
				-1L
			);
		}
		else {
			bundleId = -1L;
		}

		return bundleId;
	}

	@Override
	public Collection<BundleDTO> getBundles() throws Exception {
		return _getBundles(_client);
	}

	@Override
	public long install(URI uri) throws Exception {
		final String installString;

		Path uriPath = Paths.get(uri);

		if (_WAR_FILE_GLOB.matches(uriPath)) {
			installString = "install " + _getWarString(uriPath);
		}
		else {
			installString = "install " + uri.toASCIIString();
		}

		String response = _sendGogo(installString);

		try {
			Matcher matcher = _installResponse.matcher(response);

			matcher.matches();

			return Integer.parseInt(matcher.group(1));
		}
		catch (Exception exception) {
			throw new Exception("Unable to install bundle.  Unexpected response: \n" + response, exception);
		}
	}

	@Override
	public void refresh(long id) throws Exception {
		_sendGogo(String.format("refresh %s", id));
	}

	@Override
	public void start(long id) throws Exception {
		String request = String.format("start %s", id);

		String response = _sendGogo(request);

		_verify(request, response);
	}

	@Override
	public void stop(long id) throws Exception {
		_sendGogo(String.format("stop %s", id));
	}

	@Override
	public void uninstall(long id) throws Exception {
		_sendGogo(String.format("uninstall %s", id));
	}

	@Override
	public void update(long id, URI uri) throws Exception {
		final String installString;

		Path uriPath = Paths.get(uri);

		if (_WAR_FILE_GLOB.matches(uriPath)) {
			installString = "update " + _getWarString(uriPath);
		}
		else {
			installString = _sendGogo(String.format("update %s %s", id, uri.toASCIIString()));
		}

		_sendGogo(installString);
	}

	private static final int _getState(String state) {
		String bundleState = state.toUpperCase();

		if (Objects.equals(bundleState, "ACTIVE")) {
			return Bundle.ACTIVE;
		}
		else if (Objects.equals(bundleState, "INSTALLED")) {
			return Bundle.INSTALLED;
		}
		else if (Objects.equals(bundleState, "RESOLVED")) {
			return Bundle.RESOLVED;
		}
		else if (Objects.equals(bundleState, "STARTING")) {
			return Bundle.STARTING;
		}
		else if (Objects.equals(bundleState, "STOPPING")) {
			return Bundle.STOPPING;
		}
		else if (Objects.equals(bundleState, "UNINSTALLED")) {
			return Bundle.UNINSTALLED;
		}

		return 0;
	}

	private static final BundleDTO _newBundleDTO(Long id, int state, String symbolicName) {
		BundleDTO bundle = new BundleDTO();

		bundle.id = id;
		bundle.state = state;
		bundle.symbolicName = symbolicName;

		return bundle;
	}

	private static final BundleDTO _parseGogoLine(String line) {
		String[] fields = line.split("\\|");

		Long id = Long.parseLong(fields[0].trim());

		int state = _getState(fields[1].trim());

		String symbolicName = fields[3];

		return _newBundleDTO(id, state, symbolicName);
	}

	private List<BundleDTO> _getBundles(GogoShellClient client) throws Exception {
		String response = client.send("lb -s -u");

		String[] lines = _parseGogoResponse(response);

		return Stream.of(
			lines
		).skip(
			3
		).map(
			LiferayBundleDeployerImpl::_parseGogoLine
		).collect(
			Collectors.toList()
		);
	}

	private String _getWarString(Path path) throws Exception {
		if (!_WAR_FILE_GLOB.matches(path)) {
			throw new IllegalArgumentException("Must provide a valid WAR file");
		}

		String fileNameString = String.valueOf(path.getFileName());

		Matcher matcher = _versionPattern.matcher(fileNameString);

		if (matcher.find()) {
			fileNameString = matcher.replaceFirst(".war");
		}

		URI uri = path.toUri();

		return String.format(_WAR_STRING_TEMPLATE, uri.toASCIIString(), fileNameString, fileNameString);
	}

	private final String[] _parseGogoResponse(String response) {
		return response.split("\\r?\\n");
	}

	private String _sendGogo(String data) throws Exception {
		return _client.send(data);
	}

	private void _verify(String request, String response) throws Exception {
		Objects.requireNonNull(request, "Request cannot be null");
		Objects.requireNonNull(request, "Response cannot be null");
		request = request.trim();
		response = response.trim();

		String requestWithoutBreaks = request.replace(System.lineSeparator(), "");
		String responseWithoutBreaks = response.replace(System.lineSeparator(), "");

		int requestLineBreakCount = request.length() - requestWithoutBreaks.length();
		int responseLineBreakCount = response.length() - responseWithoutBreaks.length();

		if (requestLineBreakCount != responseLineBreakCount) {
			String exceptionString =
				"Unexpected Exception exceptionncountered while processing command \"" + request + "\":" +
					System.lineSeparator() + response;

			throw new Exception(exceptionString);
		}

		String[] requestSplit = request.split(" ");
		String[] responseSplit = response.split(" ");

		if (requestSplit.length != responseSplit.length) {
			String exceptionString =
				"Unexpected response encountered while processing command \"" + request + "\":" +
					System.lineSeparator() + response;

			throw new Exception(exceptionString);
		}

		for (int x = 0; x < requestSplit.length; x++) {
			if (!Objects.equals(requestSplit[0], responseSplit[0])) {
				String exceptionString =
					"Unexpected response encountered while processing command \"" + request + "\":" +
						System.lineSeparator() + response;

				throw new Exception(exceptionString);
			}
		}
	}

	private static final FileSystem _FILE_SYSTEM = FileSystems.getDefault();

	private static final PathMatcher _WAR_FILE_GLOB = _FILE_SYSTEM.getPathMatcher("glob:**.war");

	private static final String _WAR_STRING_TEMPLATE = "webbundle:%s?Bundle-SymbolicName=%s&Web-ContextPath=/%s";

	private static final Pattern _installResponse = Pattern.compile(
		".*Bundle ID: (.*$).*", Pattern.DOTALL | Pattern.MULTILINE);
	private static final Pattern _versionPattern = Pattern.compile("-[\\d]+((\\.[\\d]+)+(-.+)*)\\.war$");

	private GogoShellClient _client;

}