package com.juphoon.cloud;

import android.support.annotation.IntDef;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.nio.ByteBuffer;
import java.util.List;

/**
 * 设备模块
 *
 * @author juphoon
 */
public abstract class JCMediaDevice {

    /**
     * 渲染方式
     */
    @IntDef({RENDER_FULL_SCREEN, RENDER_FULL_CONTENT, RENDER_FULL_AUTO})
    @Retention(RetentionPolicy.SOURCE)
    public @interface RenderType {
    }

    /**
     * 铺满窗口
     */
    public static final int RENDER_FULL_SCREEN = 0;
    /**
     * 全图像显示，会有黑边
     */
    public static final int RENDER_FULL_CONTENT = 1;
    /**
     * 自适应
     */
    public static final int RENDER_FULL_AUTO = 2;

    /**
     * 摄像头类型
     */
    @IntDef({CAMERA_NONE, CAMERA_FRONT, CAMERA_BACK, CAMERA_UNKNOWN})
    @Retention(RetentionPolicy.SOURCE)
    public @interface CameraType {
    }

    /**
     * 未获取到摄像头
     */
    public static final int CAMERA_NONE = 0;
    /**
     * 前置摄像头
     */
    public static final int CAMERA_FRONT = 1;
    /**
     * 后置摄像头
     */
    public static final int CAMERA_BACK = 2;
    /**
     * 未知摄像头
     */
    public static final int CAMERA_UNKNOWN = 3;

    /**
     * 回声消除模式
     */
    @IntDef({AEC_OS, AEC_SDE, AEC_OS_SDE})
    @Retention(RetentionPolicy.SOURCE)
    public @interface AecMode {
    }

    /**
     * OS
     */
    public static final int AEC_OS = 0;
    /**
     * SDE
     */
    public static final int AEC_SDE = 1;
    /**
     * OS SDE
     */
    public static final int AEC_OS_SDE = 2;

    /**
     * 视频通话呼叫模式
     */
    @IntDef({VIDEO_MODE_S_S, VIDEO_MODE_S_L, VIDEO_MODE_L_S, VIDEO_MODE_L_L})
    @Retention(RetentionPolicy.SOURCE)
    public @interface VideoMode {
    }

    /**
     * 小屏呼小屏，例如 本端手表 - 远端手表 通话
     */
    public static final int VIDEO_MODE_S_S = 0;
    /**
     * 小屏呼大屏，例如 本端手表 - 远端手机 通话
     */
    public static final int VIDEO_MODE_S_L = 1;
    /**
     * 大屏呼小屏，例如 本端手机 - 远端手表 通话
     */
    public static final int VIDEO_MODE_L_S = 2;
    /**
     * 大屏呼大屏，例如 本端手机 - 远端手机 通话
     */
    public static final int VIDEO_MODE_L_L = 3;


    /**
     * 视频像素格式
     */
    @IntDef({I420, IYUV, RGB24, ABGR, ARGB, ARGB444, RGB565, ARGB1555, YUY2, YV12, UYVY, MJPG, NV21, NV12, BGRA, H264})
    @Retention(RetentionPolicy.SOURCE)
    public @interface VideoPixelFormat {
    }

    /**
     * I420
     */
    public static final int I420 = 1;
    /**
     * IYUV
     */
    public static final int IYUV = 2;
    /**
     * RGB24
     */
    public static final int RGB24 = 3;
    /**
     * ABGR
     */
    public static final int ABGR = 4;
    /**
     * ARGB
     */
    public static final int ARGB = 5;
    /**
     * ARGB444
     */
    public static final int ARGB444 = 6;
    /**
     * RGB565
     */
    public static final int RGB565 = 7;
    /**
     * ARGB1555
     */
    public static final int ARGB1555 = 8;
    /**
     * YUY2
     */
    public static final int YUY2 = 9;
    /**
     * YV12
     */
    public static final int YV12 = 10;
    /**
     * UYVY
     */
    public static final int UYVY = 11;
    /**
     * MJPG
     */
    public static final int MJPG = 12;
    /**
     * NV21
     */
    public static final int NV21 = 13;
    /**
     * NV12
     */
    public static final int NV12 = 14;
    /**
     * BGRA
     */
    public static final int BGRA = 15;
    /**
     * H264
     */
    public static final int H264 = 100;

    private static JCMediaDevice sMediaDevice;

    /**
     * 视频窗口是否跟着手机自动旋转，默认为false
     */
    public boolean autoRotate = false;

    /**
     * setVideoMode 额外参数
     */
    public static class VideoModeOtherParam {
        /**
         * 声音增益，VIDEO_MODE_S_S 默认关，VIDEO_MODE_S_L 默认开，VIDEO_MODE_L_S 默认关，VIDEO_MODE_L_L 默认开
         *
         * -1 不会按照此 agc 设置，按照 VideoMode 的默认值设置
         * 0 表示关闭 agc
         * 1 表示开启 agc
         */
        public int agc;

        public VideoModeOtherParam() {
            agc = -1;
        }

    }

    /**
     * 创建 JCMediaDevice 对象
     *
     * @param client    JCClient 对象
     * @param callback  JCMediaDeviceCallback 回调接口，用于接收 JCMediaDevice 相关通知
     * @return          返回 JCMediaDevice 对象
     */
    public static JCMediaDevice create(JCClient client, JCMediaDeviceCallback callback) {
        if (sMediaDevice != null) {
            return sMediaDevice;
        }
        sMediaDevice = new JCMediaDeviceImpl(client, callback);
        return sMediaDevice;
    }

    /**
     * 销毁 JCMediaDevice 对象
     */
    public static void destroy() {
        if (sMediaDevice != null) {
            JCClientThreadImpl.getInstance().post(new Runnable() {
                @Override
                public void run() {
                    sMediaDevice.destroyObj();
                    sMediaDevice = null;
                }
            });
        }
    }

    /**
     * 销毁对象
     */
    protected abstract void destroyObj();

    /**
     * 摄像头是否打开
     *
     * @return 摄像头是否打开
     */
    public abstract boolean isCameraOpen();

    /**
     * 是否开启扬声器
     *
     * @return 是否开启扬声器
     */
    public abstract boolean isSpeakerOn();

    /**
     * 当前默认的摄像头
     *
     * @return 当前摄像头
     */
    public abstract String getCamera();

    /**
     * 获取摄像头列表
     *
     * @return 摄像头列表
     */
    public abstract List<String> getCameras();

    /**
     * 获得视频预览对象，通过此对象能获得视图用于UI显示
     *
     * @param renderType    渲染模式
     * @return              JCMediaDeviceVideoCanvas 对象
     * @see RenderType
     */
    public abstract JCMediaDeviceVideoCanvas startCameraVideo(@RenderType int renderType);

    /**
     * 获得视频对象，通过此对象能获得视图用于UI显示
     *
     * @param videoSource   渲染标识串，比如 JCMediaChannelParticipant JCCallItem 中的 renderId，当videoSource 为 videoFileId 时，内部会调用 startVideoFile
     * @param renderType    渲染模式
     * @return              JCMediaDeviceVideoCanvas 对象
     * @see RenderType
     */
    public abstract JCMediaDeviceVideoCanvas startVideo(String videoSource, @RenderType int renderType);

    /**
     * 停止视频
     *
     * @param canvas JCMediaDeviceVideoCanvas 对象，由 startVideo 获得
     */
    public abstract void stopVideo(JCMediaDeviceVideoCanvas canvas);

    /**
     * 启动音频，一般正式开启通话前需要调用此接口
     *
     * @return 成功返回 true，失败返回 false
     */
    public abstract boolean startAudio();

    /**
     * 停止音频，一般在通话结束时调用
     *
     * @return 成功返回 true，失败返回 false
     */
    public abstract boolean stopAudio();

    /**
     * 开启摄像头
     *
     * @return 成功返回 true，失败返回 false
     */
    public abstract boolean startCamera();

    /**
     * 关闭摄像头
     *
     * @return 成功返回 true，失败返回 false
     */
    public abstract boolean stopCamera();

    /**
     * 切换摄像头，内部会根据当前摄像头类型来进行切换
     *
     * @return 成功返回 true，失败返回 false
     */
    public abstract boolean switchCamera();

    /**
     * 指定要开启的摄像头
     * @param cameraIndex   摄像头索引
     */
    public abstract void specifyCamera(int cameraIndex);

    /**
     * 设置摄像头采集属性
     * @param width     采集宽度，默认640
     * @param height    采集高度，默认360
     * @param frameRate 采集帧速率，默认30
     */
    public abstract void setCameraProperty(int width, int height, int frameRate);

    /**
     * 设置屏幕共享采集属性
     * @param width     采集宽度，默认640
     * @param height    采集高度，默认360
     * @param frameRate 采集帧速率，默认10
     */
    public abstract void setScreenCaptureProperty(int width, int height, int frameRate);

    /**
     * 获得摄像头类型
     *
     * @param cameraIndex 摄像头队列索引
     * @return 摄像头类型
     */
    public abstract @CameraType int getCameraType(int cameraIndex);

    /**
     * 开启关闭扬声器
     *
     * @param enable 是否开启
     */
    public abstract void enableSpeaker(boolean enable);

    /**
     * 设置视频模式，在登陆成功后, 呼叫前或者收到来电前调用此接口, 因此在调用前要明确本端设备和远端设备类型，并传入合适的类型
     *
     * @param videoMode 视频模式
     * @param otherParam 额外设置参数
     */
    public abstract void setVideoMode(@VideoMode int videoMode, VideoModeOtherParam otherParam);

    /**
     * 获得视频模式
     *
     * @return 视频模式
     */
    public abstract int getVideoMode();

    /**
     * 设置回声消除模式, 登陆前设置
     * @param aecMode 回声消除模式
     */
    public abstract void setAecMode(@AecMode int aecMode);

    /**
     * 获取回声消除模式
     * @return 回声消除模式
     */
    public abstract @AecMode int getAecMode();

    /**
     * 设置抗锯齿
     *
     * @param sawtooth 是否抗锯齿
     */
    public abstract void setSawtooth(boolean sawtooth);

    /**
     * 文件视频源是否开启
     *
     * @return 是否开启文件视频源
     */
    public abstract boolean isVideoFileOpen();

    /**
     * 获得文件视频源渲染id
     *
     * @return 视频源渲染id
     */
    public abstract String getVideoFileId();

    /**
     * 开启视频文件作为视频输入源，文件和摄像头作为视频输入源只能存在一种，当前摄像头开启的话会关闭摄像头
     *
     * @return 是否成功
     */
    public abstract boolean startVideoFile();

    /**
     * 逐帧采集视频画面
     *
     * 当为 H264 格式时
     * 1. 如果是关键帧需要将 0x67 0x68 0x41 的数据作为一帧传入
     * 2. 关键帧要以固定间隔传入，例如5秒，否则一开始可能有几秒对端无法显示视频
     *
     * @param data 画面二进制数据
     * @param format   视频像素格式
     * @param width    宽
     * @param height   高
     * @param angle    90 的倍数
     * @param mirror   0 不镜像，1进行左右镜像
     * @param keyFrame 是否为关键帧，针对 format 为 H264
     */
    public abstract void setVideoFileFrame(byte[] data, @VideoPixelFormat int format, int width, int height, int angle, int mirror, boolean keyFrame);

    /**
     * 关闭逐帧采集画面
     *
     * @return ture表示关闭成功，false表示关闭失败
     */
    public abstract boolean stopVideoFile();

    /**
     * 通知底层需要使用 H264 传输帧数据，可以在 JCMediaDevice 创建完调用此接口
     */
    public abstract void readyForH264();

    /**
     * 开启/关闭屏幕采集
     * @param   enable  是否开启
     * @return  开启/关闭 是否成功
     */
    protected abstract boolean enableScreenCapture(boolean enable);

    /**
     * 添加回调
     *
     * @param callback JCMediaDeviceCallback 接口对象
     */
    protected abstract void addCallback(JCMediaDeviceCallback callback);

    /**
     * 删除回调
     *
     * @param callback JCMediaDeviceCallback 接口对象
     */
    protected abstract void removeCallback(JCMediaDeviceCallback callback);

    /**
     * 针对手表配置媒体参数
     */
    protected abstract void configMeidaParam(VideoModeOtherParam otherParam);
}
