package com.juphoon.cloud;

import android.graphics.Color;

import java.util.ArrayList;
import java.util.List;

/**
 * 涂鸦动作对象类
 *
 * @author juphoon
 */
public class JCDoodleAction {

    private String userId;
    private int actionType;
    private int pageId;
    private int paintColor;
    private float paintStrokeWidth;
    private List<List<String>> intervalPointList;
    private String stickerUnicode;
    private String stickerName;
    private float stickerX;
    private float stickerY;
    private float stickerWidth;
    private float stickerHeight;
    private int stickerRotate;
    private String userDefine;

    private JCDoodleAction(Builder builder) {
        this.actionType = builder.actionType;
        this.userId = builder.userId;
        this.pageId = builder.pageId;
        this.paintColor = builder.paintColor;
        this.paintStrokeWidth = builder.paintStrokeWidth;
        this.intervalPointList = builder.intervalPointList;
        this.stickerUnicode = builder.stickerUnicode;
        this.stickerName = builder.stickerName;
        this.stickerX = builder.stickerX;
        this.stickerY = builder.stickerY;
        this.stickerWidth = builder.stickerWidth;
        this.stickerHeight = builder.stickerHeight;
        this.stickerRotate = builder.stickerRotate;
        this.userDefine = builder.userDefine;
    }

    /**
     * 获取涂鸦动作用户标识
     *
     * @return 用户标识
     */
    public String getUserId() {
        return userId;
    }

    /**
     * 获取涂鸦动作类型
     *
     * @return 涂鸦动作类型
     */
    public int getActionType() {
        return actionType;
    }

    /**
     * 获取涂鸦页码
     *
     * @return 涂鸦页码
     */
    public int getPageId() {
        return pageId;
    }

    /**
     * 获取涂鸦画笔颜色
     *
     * @return 画笔颜色
     */
    public int getPaintColor() {
        return paintColor;
    }

    /**
     * 获取画笔粗细
     *
     * @return 画笔粗细
     */
    public float getPaintStrokeWidth() {
        return paintStrokeWidth;
    }

    /**
     * 获取涂鸦点列表
     *
     * @return 涂鸦点列表
     */
    public List<List<String>> getIntervalPointList() {
        return intervalPointList;
    }

    /**
     * 获取贴纸 unicode 码
     *
     * @return 贴纸 unicode 码
     */
    public String getStickerUnicode() {
        return stickerUnicode;
    }

    /**
     * 获取贴纸名称
     *
     * @return 贴纸名称
     */
    public String getStickerName() {
        return stickerName;
    }

    /**
     * 获取贴纸 x 位置
     *
     * @return 贴纸 x 位置
     */
    public float getStickerX() {
        return stickerX;
    }

    /**
     * 获取贴纸 y 位置
     *
     * @return 贴纸 y 位置
     */
    public float getStickerY() {
        return stickerY;
    }

    /**
     * 获取贴纸宽度
     *
     * @return 贴纸宽度
     */
    public float getStickerWidth() {
        return stickerWidth;
    }

    /**
     * 获取贴纸高度
     *
     * @return 贴纸高度
     */
    public float getStickerHeight() {
        return stickerHeight;
    }

    /**
     * 获取贴纸角度
     *
     * @return 贴纸角度
     */
    public int getStickerRotate() {
        return stickerRotate;
    }

    /**
     * 获取用户自定义数据
     *
     * @return 用户自定义数据
     */
    public String getUserDefine() {
        return userDefine;
    }

    /**
     * 涂鸦动作对象构造器
     *
     * @author juphoon
     */
    public static final class Builder {
        private int actionType;
        private String userId;
        private int pageId = 0;
        private int paintColor = Color.TRANSPARENT;
        private float paintStrokeWidth = 0.0f;
        private List<List<String>> intervalPointList;
        private String stickerUnicode;
        private String stickerName;
        private float stickerX;
        private float stickerY;
        private float stickerWidth;
        private float stickerHeight;
        private int stickerRotate;
        private String userDefine;

        /**
         * {@link JCDoodleAction} 构造器。
         *
         * @param actionType 涂鸦类型。所有 {@link JCDoodleAction} 都必须有涂鸦类型。
         */
        public Builder(int actionType) {
            this.actionType = actionType;
            this.intervalPointList = new ArrayList<>();
        }

        /**
         * 发起者
         *
         * @param userId 发起者标识。
         * @return {@link JCDoodleAction} 的构造器对象
         */
        public Builder userId(String userId) {
            this.userId = userId;
            return this;
        }

        /**
         * 页码。用于多页涂鸦，默认为 0。
         *
         * @param pageId 页码
         * @return {@link JCDoodleAction} 的构造器对象
         */
        public Builder pageId(int pageId) {
            this.pageId = pageId;
            return this;
        }

        /**
         * 设置涂鸦轨迹颜色
         *
         * @param paintColor 颜色
         * @return {@link JCDoodleAction} 的构造器对象。
         */
        public Builder paintColor(int paintColor) {
            this.paintColor = paintColor;
            return this;
        }

        /**
         * 设置涂鸦轨迹或橡皮擦宽度
         *
         * @param paintStrokeWidth 宽度
         * @return {@link JCDoodleAction} 的构造器对象
         */
        public Builder paintStrokeWidth(float paintStrokeWidth) {
            this.paintStrokeWidth = paintStrokeWidth;
            return this;
        }

        /**
         * 添加涂鸦轨迹点。
         *
         * 逻辑坐标点由屏幕坐标点根据屏幕尺寸转化。
         * 假设屏幕坐标点为 (sX, sY), 屏幕尺寸为 size，则逻辑坐标点为 ((sX + 1.0) * size.width / 2, (sY + 1.0) * size.width / 2)。
         *
         * @param interval 与上一个点的时间间隔。
         * @param x 逻辑坐标点 x 值。
         * @param y 逻辑坐标点 y 值。
         * @return {@link JCDoodleAction} 的构造器对象
         */
        public Builder addActionPoint(int interval, float x, float y) {
            List<String> list = new ArrayList<>();
            list.add(String.valueOf(interval));
            list.add(String.valueOf(x));
            list.add(String.valueOf(y));
            intervalPointList.add(list);
            return this;
        }

        /**
         * 添加涂鸦轨迹点。
         *
         * 逻辑坐标点由屏幕坐标点根据屏幕尺寸转化。
         * 假设屏幕坐标点为 (sX, sY), 屏幕尺寸为 size，则逻辑坐标点为 ((sX + 1.0) * size.width / 2, (sY + 1.0) * size.width / 2)。
         *
         * @param interval 与上一个点的时间间隔。
         * @param x 逻辑坐标点 x 值。
         * @param y 逻辑坐标点 y 值。
         * @return {@link JCDoodleAction} 的构造器对象
         */
        Builder addActionPoint(String interval, String x, String y) {
            List<String> list = new ArrayList<>();
            list.add(interval);
            list.add(x);
            list.add(y);
            intervalPointList.add(list);
            return this;
        }

        /**
         * 设置贴图 unicode。当贴图为 emoji 表情时，需要设置 unicode。
         *
         * @param stickerUnicode 贴图的 Unicode 值
         * @return {@link JCDoodleAction} 的构造器对象
         */
        public Builder stickerUnicode(String stickerUnicode) {
            this.stickerUnicode = stickerUnicode;
            return this;
        }

        /**
         * 设置 stickerName。当贴图为自定义图片时，需要设置 stickerName
         *
         * @param stickerName
         * @return {@link JCDoodleAction} 的构造器对象
         */
        public Builder stickerName(String stickerName) {
            this.stickerName = stickerName;
            return this;
        }

        /**
         * 设置贴图逻辑 x 值。
         *
         * 逻辑坐标点由屏幕坐标点根据屏幕尺寸转化。
         *
         * @param stickerX 贴图的逻辑 x 值。
         * @return {@link JCDoodleAction} 的构造器对象
         */
        public Builder stickerX(float stickerX) {
            this.stickerX = stickerX;
            return this;
        }

        /**
         * 设置贴图逻辑 Y 值。
         *
         * 逻辑坐标点由屏幕坐标点根据屏幕尺寸转化。
         *
         * @param stickerY 贴图的逻辑 Y 值。
         * @return {@link JCDoodleAction} 的构造器对象
         */
        public Builder stickerY(float stickerY) {
            this.stickerY = stickerY;
            return this;
        }

        /**
         * 设置贴图宽度
         *
         * @param stickerWidth 贴图宽度
         * @return {@link JCDoodleAction} 的构造器对象
         */
        public Builder stickerWidth(float stickerWidth) {
            this.stickerWidth = stickerWidth;
            return this;
        }

        /**
         * 设置贴图高度
         *
         * @param stickerHeight 贴图高度
         * @return {@link JCDoodleAction} 的构造器对象
         */
        public Builder stickerHeight(float stickerHeight) {
            this.stickerHeight = stickerHeight;
            return this;
        }

        /**
         * 设置贴图角度
         *
         * @param stickerRotate 贴图角度
         * @return {@link JCDoodleAction} 的构造器对象
         */
        public Builder stickerRotate(int stickerRotate) {
            this.stickerRotate = stickerRotate;
            return this;
        }

        /**
         * 设置用户自定义数据
         *
         * @param userDefine 用户自定义数据
         * @return {@link JCDoodleAction} 的构造器对象
         */
        public Builder userDefine(String userDefine) {
            this.userDefine = userDefine;
            return this;
        }

        /**
         * 构造 {@link JCDoodleAction}
         *
         * @return {@link JCDoodleAction} 对象
         */
        public JCDoodleAction build() {
            return new JCDoodleAction(this);
        }
    }
}
