package com.juphoon.cloud;

import android.graphics.Color;
import android.text.TextUtils;
import android.util.Log;

import com.justalk.cloud.lemon.MtcDoodleConstants;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;

import java.lang.ref.WeakReference;
import java.util.List;

public class JCDoodleImpl extends JCDoodle {

    private DoodleEngine mDoodleEngine;
    private JCDoodleCallback mDoodleCallback;
    private WeakReference<JCDoodleInteractor> mDoodleInteractorRef;

    JCDoodleImpl(JCDoodleCallback doodleCallback, DoodleEngine doodleEngine) {
        mDoodleCallback = doodleCallback;
        mDoodleEngine = doodleEngine;
    }

    @Override
    protected void destroyObj() {
        mDoodleCallback = null;
    }

    @Override
    public void bindDoodleInteractor(JCDoodleInteractor interactor) {
        mDoodleInteractorRef = new WeakReference<>(interactor);
    }

    @Override
    public void generateDoodleAction(JCDoodleAction doodleAction) {
        mDoodleCallback.onDoodleActionGenerated(doodleAction);
    }

    @Override
    public String stringFromDoodleAction(JCDoodleAction doodleAction) {
        if (doodleAction == null) {
            return null;
        }
        int actionType = doodleAction.getActionType();
        if (actionType == JCDoodle.ACTION_INVALID) {
            throw new RuntimeException("please set actionType and other parameters first!");
        }
        JSONObject json = new JSONObject();
        try {
            json.put(MtcDoodleConstants.MtcDoodleActionTypeKey, actionType);
            if (!TextUtils.isEmpty(doodleAction.getUserId())) {
                json.put(MtcDoodleConstants.MtcDoodleUserUriKey, doodleAction.getUserId());
            }
            json.put(MtcDoodleConstants.MtcDoodlePageIdKey, doodleAction.getPageId());
            JSONObject brushJson = new JSONObject();
            if (doodleAction.getPaintColor() != Color.TRANSPARENT) {
                brushJson.put(MtcDoodleConstants.MtcDoodleColorKey, doodleAction.getPaintColor());
            }
            if (doodleAction.getPaintStrokeWidth() > 0.0f) {
                brushJson.put(MtcDoodleConstants.MtcDoodleWidthKey, doodleAction.getPaintStrokeWidth());
            }
            json.put(MtcDoodleConstants.MtcDoodleBrushKey, brushJson);
            JSONObject contentJson = new JSONObject();
            if (actionType == ACTION_STICKER) {
                JSONObject stickerContent = new JSONObject();
                stickerContent.put(KEY_CUSTOM_STICKER_EMOJI_UNICODE, doodleAction.getStickerUnicode());
                stickerContent.put(KEY_CUSTOM_STICKER_NAME, doodleAction.getStickerName());
                stickerContent.put(KEY_CUSTOM_STICKER_X, doodleAction.getStickerX());
                stickerContent.put(KEY_CUSTOM_STICKER_Y, doodleAction.getStickerY());
                stickerContent.put(KEY_CUSTOM_STICKER_WIDTH, doodleAction.getStickerWidth());
                stickerContent.put(KEY_CUSTOM_STICKER_HEIGHT, doodleAction.getStickerHeight());
                stickerContent.put(KEY_CUSTOM_STICKER_ROTATE, doodleAction.getStickerRotate());
                contentJson.put(KEY_STICKER_CONTENT, stickerContent.toString());
                contentJson.put(KEY_STICKER_USER_DEFINE, doodleAction.getUserDefine());
                json.put(MtcDoodleConstants.MtcDoodleContentKey, contentJson.toString());
            } else if (!TextUtils.isEmpty(doodleAction.getUserDefine())) {
                json.put(MtcDoodleConstants.MtcDoodleContentKey, doodleAction.getUserDefine());
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        long action = mDoodleEngine.createAction();
        mDoodleEngine.setActionParms(action, json.toString());
        List<List<String>> intervalPointList = doodleAction.getIntervalPointList();
        if (intervalPointList != null && !intervalPointList.isEmpty()) {
            for (List<String> points : intervalPointList) {
                mDoodleEngine.addActionPosition(action, Float.valueOf(points.get(1)),
                        Float.valueOf(points.get(2)), Integer.valueOf(points.get(0)));
            }
        }
        String ret = mDoodleEngine.printAction(action);
        Log.d("Doodle", "stringFromDoodleAction: " + ret + " parms:" + json.toString());
        mDoodleEngine.deleteAction(action);
        return ret;
    }

    @Override
    public JCDoodleAction doodleActionFromString(String doodleActionData) {
        if (!TextUtils.isEmpty(doodleActionData)) {
            try {
                long action = mDoodleEngine.parseAction(doodleActionData);
                String actionInfo = mDoodleEngine.getActionParms(action);
                JSONObject jsonInfo = (JSONObject) new JSONTokener(actionInfo).nextValue();
                Log.d("Doodle", "stringFromDoodleAction: " + doodleActionData + " actionInfo:" + actionInfo);
                int actionType = jsonInfo.getInt(MtcDoodleConstants.MtcDoodleActionTypeKey);
                JCDoodleAction.Builder builder = new JCDoodleAction.Builder(actionType);
                int pageId = jsonInfo.getInt(MtcDoodleConstants.MtcDoodlePageIdKey);
                builder.pageId(pageId);
                String userId = jsonInfo.getString(MtcDoodleConstants.MtcDoodleUserUriKey);
                builder.userId(userId);
                JSONObject brushJson = jsonInfo.getJSONObject(MtcDoodleConstants.MtcDoodleBrushKey);
                float brushWidth = (float) brushJson.getDouble(MtcDoodleConstants.MtcDoodleWidthKey);
                builder.paintStrokeWidth(brushWidth);
                int brushColor = brushJson.getInt(MtcDoodleConstants.MtcDoodleColorKey);
                builder.paintColor(brushColor);
                switch (actionType) {
                    case JCDoodle.ACTION_DRAW:
                    case JCDoodle.ACTION_ERASER:
                        String pathJson = mDoodleEngine.getActionPath(action);
                        JSONArray points = new JSONArray(pathJson);
                        for (int i = 0; i < points.length(); i++) {
                            JSONArray p = (JSONArray) points.get(i);
                            String interval = p.getString(0);
                            String x = p.getString(1);
                            String y = p.getString(2);
                            builder.addActionPoint(interval, x, y);
                        }
                        break;
                    case JCDoodle.ACTION_STICKER:
                        String sticker = jsonInfo.optString(MtcDoodleConstants.MtcDoodleContentKey);
                        if (!TextUtils.isEmpty(sticker)) {
                            JSONObject stickerContent = (JSONObject) new JSONTokener(sticker).nextValue();
                            String stickerJson = stickerContent.optString(KEY_STICKER_CONTENT);
                            JSONObject jsonSticker = (JSONObject) new JSONTokener(stickerJson).nextValue();
                            builder.stickerUnicode(jsonSticker.optString(KEY_CUSTOM_STICKER_EMOJI_UNICODE));
                            builder.stickerName(jsonSticker.optString(KEY_CUSTOM_STICKER_NAME));
                            builder.stickerX(Float.valueOf(jsonSticker.optString(KEY_CUSTOM_STICKER_X)));
                            builder.stickerY(Float.valueOf(jsonSticker.optString(KEY_CUSTOM_STICKER_Y)));
                            builder.stickerWidth(Float.valueOf(jsonSticker.optString(KEY_CUSTOM_STICKER_WIDTH)));
                            builder.stickerHeight(Float.valueOf(jsonSticker.optString(KEY_CUSTOM_STICKER_HEIGHT)));
                            builder.stickerRotate(jsonSticker.optInt(KEY_CUSTOM_STICKER_ROTATE));
                            builder.userDefine(stickerContent.optString(KEY_STICKER_CONTENT));
                        }
                        break;
                    default:
                        builder.userDefine(jsonInfo.optString(MtcDoodleConstants.MtcDoodleContentKey));
                        break;
                }
                JCDoodleAction doodleAction = builder.build();
                JCDoodleInteractor doodleInteractor = getDoodleInteractor();
                if (doodleInteractor != null) {
                    doodleInteractor.onDoodleReceived(doodleAction);
                }
                return doodleAction;
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return null;
    }

    private JCDoodleInteractor getDoodleInteractor() {
        if (mDoodleInteractorRef != null) {
            return mDoodleInteractorRef.get();
        }
        return null;
    }
}
