package com.jibestream.navigationkit;

import android.graphics.Path;
import android.graphics.PointF;
import android.graphics.RectF;
import android.graphics.Region;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.jibestream.jmapandroidsdk.astar.ASNode;
import com.jibestream.jmapandroidsdk.astar.PathPerFloor;
import com.jibestream.jmapandroidsdk.collections.MapCollection;
import com.jibestream.jmapandroidsdk.components.Floor;
import com.jibestream.jmapandroidsdk.components.Map;
import com.jibestream.jmapandroidsdk.components.Waypoint;
import com.jibestream.jmapandroidsdk.jcontroller.JController;
import com.jibestream.jmapandroidsdk.main.Utilities;
import com.jibestream.jmapandroidsdk.rendering_engine.MapLayer;
import com.jibestream.jmapandroidsdk.rendering_engine.drawables.JIconDrawable;
import com.jibestream.jmapandroidsdk.rendering_engine.drawables.JShapeDrawable;
import com.jibestream.jmapandroidsdk.rendering_engine.moving_objects.UserLocation;
import com.jibestream.jmapandroidsdk.styles.JStyle;
import com.jibestream.navigationkit.instructionfactory.Direction;
import com.jibestream.navigationkit.instructionfactory.Instruction;
import com.jibestream.navigationkit.instructionfactory.InstructionFactory;
import com.jibestream.navigationkit.surroundings.SurroundingDefinition;
import com.jibestream.navigationkit.surroundings.SurroundingElements;
import com.jibestream.navigationkit.surroundings.SurroundingIcon;
import com.jibestream.navigationkit.surroundings.SurroundingShape;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;

/**
 * Created by Andrew Adams on 2017-05-15.
 */
public class NavigationKit {

    private JController controller;
    private int angleThreshold = 20;
    private InstructionFactory instructionFactory;

    /**
     * NavigationKit Constructor
     * Pass in {@link JController} to initialize the {@link InstructionFactory}.
     *
     * @param controller
     */
    public NavigationKit(@NonNull JController controller) {
        this.controller = controller;

        instructionFactory = new InstructionFactory(this, controller);
        instructionFactory.addLayerOfInterest(MapLayer.Layer_Units);
//        instructionFactory.addLayerOfInterest(MapLayer.Layer_Amenity_Icons);
//        instructionFactory.addLayerOfInterest(MapLayer.Layer_Path_Type_Icons);
//        instructionFactory.addVisualObstacles(MapLayer.Layer_Elevators);
    }

    /**
     * NavigationKit Constructor
     * Pass in {@link JController} and {@link NavigationKitOptions} to initialize the {@link InstructionFactory}.
     *
     * @param controller
     * @param options
     */
    public NavigationKit(@NonNull JController controller, @Nullable NavigationKitOptions options) {
        this.controller = controller;

        //Null pass in
        if (options == null) {
            new NavigationKit(controller);
            return;
        }

        instructionFactory = new InstructionFactory(this, controller);
        //Extract options
        instructionFactory.setVisualLimitLeft(options.getVisualLimitLeft());
        instructionFactory.setVisualLimitRight(options.getVisualLimitRight());
        instructionFactory.setVisualRange(options.getVisualRange());
        //Default Units layer if empty array
        if (options.getLayersOfInterest().isEmpty()) {
            instructionFactory.addLayerOfInterest(MapLayer.Layer_Units);
        } else {
            for (String layer : options.getLayersOfInterest()) {
                instructionFactory.addLayerOfInterest(layer);
            }
        }
        for (String layer : options.getVisualObstacles()) {
            instructionFactory.addVisualObstacles(layer);
        }
    }

    /**
     * Sets the angles threshold.
     *
     * @param angleThreshold Angle is in degrees.
     */
    public void setAngleThreshold(int angleThreshold) {
        this.angleThreshold = angleThreshold;
    }

    /**
     * Pass in {@link PathPerFloor}s to create a set of basic instructions.
     *
     * @param pathPerFloors
     * @return
     */
    public Instruction[] createInstructionsFromPaths(@NonNull PathPerFloor[] pathPerFloors) {
        // Create empty array for appending instructions
        ArrayList<Instruction> instructions = new ArrayList<>();

        // validate to see if controller exists
        if (controller != null) {
            // Need control access
            MapCollection mapCollection = controller.getActiveVenue().getMaps();

            // Iterate through all paths
            int i = 0;
            for (PathPerFloor pathPerFloor : pathPerFloors) {
                Map map = mapCollection.getById(pathPerFloor.mapId);

                // Get the last waypoint from the path
                Waypoint lastWaypoint = map.getWaypoints().getById(pathPerFloor.points[pathPerFloor.points.length - 1].id);

                // Loop through all points on the floor and make instructions
                ASNode[] points = pathPerFloor.points;

                float distanceCalculation = 0;
                for (int k = 0; k < pathPerFloor.points.length; k++) {
                    Waypoint a = map.getWaypoints().getById(pathPerFloor.points[k].id);
                    Waypoint b = null;
                    Waypoint c = null;

                    if (k + 1 < points.length) {
                        b = map.getWaypoints().getById(pathPerFloor.points[k + 1].id);
                    }

                    if (k + 2 < points.length) {
                        c = map.getWaypoints().getById(pathPerFloor.points[k + 2].id);
                    }

                    // If three consecutive points exist, calculate the direction
                    if (a != null && b != null && c != null) {
                        Direction direction = new Direction(a, b, c, angleThreshold);

                        // If direction is NOT forward, create an instruction
                        if (!direction.getValue().equalsIgnoreCase(Direction.FORWARD)) {
                            Instruction instruction = instructionFactory.createFromDirection(direction);

                            distanceCalculation += instruction.getPixelDistance();
                            instruction.setPixelDistance(distanceCalculation);

                            distanceCalculation = 0;

                            float abAngle = pointPairToBearingDegrees(new PointF(a.getCoordinates()[0], a.getCoordinates()[1]), new PointF(b.getCoordinates()[0], b.getCoordinates()[1]));
                            float acAngle = pointPairToBearingDegrees(new PointF(a.getCoordinates()[0], a.getCoordinates()[1]), new PointF(c.getCoordinates()[0], c.getCoordinates()[1]));
                            instruction.setTurnReferenceAngle(getRelativeAngleWithDegrees(abAngle, acAngle));

                            instructions.add(instruction);
                        } else {
                            distanceCalculation += distanceBetweenPoint(new PointF(a.getCoordinates()[0], a.getCoordinates()[1]), new PointF(b.getCoordinates()[0], b.getCoordinates()[1]));
                        }
                    }
                }

                // If moving to another map after this iteration
                if (i < pathPerFloors.length - 1) {
                    // Current map & next map
                    Map currentMap = mapCollection.getById(pathPerFloor.mapId);
                    Map nextMap = mapCollection.getById(pathPerFloors[i + 1].mapId);

                    // Grab mover name, calculate directions & add to list
                    String type = pathPerFloor.pathType.getName();
                    Floor currentFloor = controller.getCurrentBuilding().getFloors().getByMap(currentMap);
                    Floor nextFloor = controller.getCurrentBuilding().getFloors().getByMap(nextMap);

                    // Create Mover Instruction
                    Instruction instruction = new Instruction();
                    instruction.setDirection(nextFloor.getLevel() > currentFloor.getLevel() ? "up" : "down");
                    instruction.setText("Take the " + type + " " + instruction.getDirection() + " to " + nextFloor.getName());

                    instruction.setCompletionPoint(lastWaypoint);
                    instruction.setPathType(pathPerFloor.pathType);
                    instruction.setNextFloor(nextFloor);

                    instructions.add(instruction);
                } else {
                    // This is the final instruction
                    // Handle case of ending on mover waypoint on next level
                    if (points.length < 2) {
                        Waypoint a = map.getWaypoints().getById(points[0].id);

                        // create direction and instruction
                        //TODO: add u-turn param?
                        Direction direction = new Direction(a, null, null, angleThreshold);

                        Instruction instruction = instructionFactory.createFromDirection(direction);
                        instruction.setTurnReferenceAngle(0);
                        instruction.setPixelDistance(0);
                        instructions.add(instruction);
                    } else {
                        Waypoint a = map.getWaypoints().getById(points[points.length - 2].id);
                        Waypoint b = lastWaypoint;

                        // create direction and instruction
                        //TODO: add u-turn param?
                        Direction direction = new Direction(a, b, null, angleThreshold);

                        Instruction instruction = instructionFactory.createFromDirection(direction);
                        instruction.setTurnReferenceAngle(Direction.getAngle(a, b));
                        instruction.setPixelDistance(distanceCalculation);
                        instructions.add(instruction);
                    }
                }

                i++;
            }
        }

        return instructions.toArray(new Instruction[instructions.size()]);
    }

    /**
     * Get all the surround elements arond a specific point.
     *
     * @param surroundingDefinition
     * @return
     */
    public SurroundingElements getSurroundingElements(@NonNull SurroundingDefinition surroundingDefinition) {
        //Set date for time logging
//        NSDate *startTotal = [NSDate date];
//        NSDate *startShape = [NSDate date];

        //Initialize variables
        float startAngle = surroundingDefinition.getGazeDirection() - surroundingDefinition.getVisualLimitLeft();
        float gazeX = (float) (surroundingDefinition.getPoint().x + surroundingDefinition.getVisualRange() * Math.cos(surroundingDefinition.getGazeDirection() * Math.PI / 180));
        float gazeY = (float) (surroundingDefinition.getPoint().y + surroundingDefinition.getVisualRange() * Math.sin(surroundingDefinition.getGazeDirection() * Math.PI / 180));
        float endAngle = surroundingDefinition.getGazeDirection() + surroundingDefinition.getVisualLimitRight();

        //Gaze's angle
        float gazeDegree = pointPairToBearingDegrees(surroundingDefinition.getPoint(), new PointF(gazeX, gazeY));

        //Create new surrounding element
        SurroundingElements surroundingElements = new SurroundingElements();

        //1. Take point and generate a circle, using the circle find segments of interest via layers of interest
        PointF center = surroundingDefinition.getPoint();
        float radius = surroundingDefinition.getVisualRange();

        //Initialize a segments of interest dictionary
        //dest ID - segment array key-value pairing
        HashMap<String, ArrayList<ArrayList<PointF>>> segmentsOfInterest = new HashMap<>();

        //Create a dictionary for storing the associated shape and Id
        HashMap<ArrayList<PointF>, JShapeDrawable> segmentToShapeDictionary = new HashMap<>();

        //Combine layersOfInterest and obstacles
        ArrayList<String> allShapesArray = new ArrayList<>();
        allShapesArray.addAll(surroundingDefinition.getLayersOfInterest());
        allShapesArray.addAll(surroundingDefinition.getVisualObstacles());

        //Store points for generating poly for icons check later
        ArrayList<PointF> viewPolygon = new ArrayList<>();

        //Get shapes within layers of interest
        for (String layer : allShapesArray) {
            JShapeDrawable[] shapes = controller.getShapesInLayer(layer, surroundingDefinition.getMap());

            //Generate optimized bound
            //Take first ray, gaze ray, last ray to
            //First ray
            float startX = (float) (surroundingDefinition.getPoint().x + surroundingDefinition.getVisualRange() * Math.cos(startAngle * Math.PI / 180));
            float startY = (float) (surroundingDefinition.getPoint().y + surroundingDefinition.getVisualRange() * Math.sin(startAngle * Math.PI / 180));

            //Last ray
            float endX = (float) (surroundingDefinition.getPoint().x + surroundingDefinition.getVisualRange() * Math.cos(endAngle * Math.PI / 180));
            float endY = (float) (surroundingDefinition.getPoint().y + surroundingDefinition.getVisualRange() * Math.sin(endAngle * Math.PI / 180));

            //Have all points required, create a path and get the bound
            Path path = new Path();

            //Move to current point
            PointF p = new PointF(surroundingDefinition.getPoint().x, surroundingDefinition.getPoint().y);
            path.moveTo(p.x, p.y);

            //Move to start point
            p = new PointF(startX, startY);
            path.lineTo(p.x, p.y);

            //Move to gaze point
            p = new PointF(gazeX, gazeY);
            path.lineTo(p.x, p.y);

            //Move to end point
            p = new PointF(endX, endY);
            path.lineTo(p.x, p.y);

            //Close the path
            path.close();

            if (viewPolygon.size() < 1) {
                //Add it to polygon points
                viewPolygon.add(surroundingDefinition.getPoint());
                viewPolygon.add(new PointF(startX, startY));
                viewPolygon.add(new PointF(gazeX, gazeY));
                viewPolygon.add(new PointF(endX, endY));

                /* DEBUG - draw gaze polygon */
//                Path gazePath = new Path();
//                gazePath.moveTo(surroundingDefinition.getPoint().x, surroundingDefinition.getPoint().y);
//                gazePath.lineTo(startX, startY);
//                gazePath.lineTo(gazeX, gazeY);
//                gazePath.lineTo(endX, endY);
//                gazePath.close();
//
//                RayDrawable rayDrawable = new RayDrawable(gazePath);
//                rayDrawable.setPoint(surroundingDefinition.getPoint());
//                Log.d("drawtag", "surroundingDefinition = " + surroundingDefinition.getPoint().x + ", " + surroundingDefinition.getPoint().y);
//                controller.addComponent(rayDrawable, controller.getCurrentMap(), surroundingDefinition.getPoint());
//                */
            }

            //Filter out the shape in bound
            //CGRect bounds = CGRectMake(surroudingDefinition.getPoint().x - surroudingDefinition.getVisualRange(), surroudingDefinition.getPoint().y - surroudingDefinition.getVisualRange(),
            // surroudingDefinition.getVisualRange() * 2, surroudingDefinition.getVisualRange() * 2);
            RectF bounds = new RectF();
            path.computeBounds(bounds, true);

            //Iterate through shapes to see which shape is within bounds
            for (JShapeDrawable shape : shapes) {
                //Check to see if the two bounds intersect
                if (shape.getBounds().intersect(bounds)) {
                    //Extract segment from shape
                    ArrayList<ArrayList<PointF>> segments = shape.getSegments();
                    for (ArrayList<PointF> segment : segments) {
                        //TODO: double check this
                        //get first segment
                        PointF startPt = segment.get(0);
                        //get last segment
                        PointF endPt = segment.get(segment.size() - 1);

                        boolean doesIntersect = doesSegmentWithPoint(startPt, endPt, center, radius);

                        if (doesIntersect) {
                            //Segment intersects with circle, add to dictionary
                            //Check if segment belongs to a shape in layer of interest
                            if (shape.getClassName().equalsIgnoreCase(layer) && surroundingDefinition.getLayersOfInterest().contains(shape.getClassName())) {
                                //Check if id already exists
                                //TODO: double check
                                ArrayList<ArrayList<PointF>> interestLayer = segmentsOfInterest.get(layer);

                                //Exists, append to array
                                if (interestLayer != null) {
                                    interestLayer.add(segment);
                                } else {
                                    //Create it with new object
                                    interestLayer = new ArrayList<>();
                                }

                                //TODO: double check
                                segmentsOfInterest.put(layer, interestLayer);
                                segmentToShapeDictionary.put(segment, shape);
                            } else {
                                //Add if they ever want to do something with obstacles?
                                //Segment belongs to obstacle layer
                                //Check if other segments array exists
                                ArrayList<ArrayList<PointF>> obstacles = segmentsOfInterest.get(MapLayer.Layer_Obstacles);
                                if (obstacles != null) {
                                    obstacles.add(segment);
                                } else {
                                    obstacles = new ArrayList<>();
                                }
                                segmentsOfInterest.put(MapLayer.Layer_Obstacles, obstacles);
                            }
                        }
                    }
                }
            }
        }

        //NSLog(@"gather shapes: %f", [[NSDate date] timeIntervalSinceDate:startShape]);

        //At this point segments of interest should be filled
        //2. Ray cast from point towards gaze direction with vision limits
        //NSDate *startRays = [NSDate date];

        //Final array containing all data to shapes hit
        ArrayList<ArrayList<PointF>> finalSegmentArray = new ArrayList();

        //Get rays to segment edges + 1 ray towards gaze direction
        ArrayList closestGazeSegment = null;
        float closestDistance = Float.MAX_VALUE;

        ArrayList<PointF> gazeRay = new ArrayList();
        gazeRay.add(surroundingDefinition.getPoint());
        gazeRay.add(new PointF(gazeX, gazeY));

        //Find the segment that hits the gazeRay and is the closest
        for (String segmentKey : segmentsOfInterest.keySet()) {
            for (ArrayList<PointF> theSegment : segmentsOfInterest.get(segmentKey)) {
                //Check if theres an intersection
                PointF intersection = getIntersectionWithRay(gazeRay, theSegment);
                if (intersection != null) {
                    float distance = distanceBetweenPoint(intersection, surroundingDefinition.getPoint());
                    if (distance < closestDistance) {
                        closestDistance = distance;
                        closestGazeSegment = theSegment;
                    }
                }
            }
        }


//        /* DEBUG - Display ray cast
        Path trackPath = new Path();
        trackPath.moveTo(surroundingDefinition.getPoint().x, surroundingDefinition.getPoint().y);
        trackPath.lineTo(gazeX, gazeY);
//        */

        //Initialize surroundShapesArray
        ArrayList<SurroundingShape> surroundingShapesArray = new ArrayList<>();

        //At this point we have the gaze ray's closest segment
        //Check all other segment points with other segments of interest
        for (String segmentKey : segmentsOfInterest.keySet()) {
            for (ArrayList<PointF> theSegment : segmentsOfInterest.get(segmentKey)) {

                //Get shape of the segment
                JShapeDrawable shape = segmentToShapeDictionary.get(theSegment);

                Boolean containsShape = false;
                for (SurroundingShape surrShape : surroundingShapesArray) {
                    if (surrShape.getShape() == shape) {
                        containsShape = true;
                        break;
                    }
                }

                //Skip if shape already been addded
                if (containsShape) {
                    continue;
                }

                //Check if closestGazeSegment is equal to theSegment, means there is a line of sight
                if (closestGazeSegment != null && closestGazeSegment.equals(theSegment)) {
                    finalSegmentArray.add(theSegment);

                    if (shape != null) {
                        //Create the shape
                        SurroundingShape surroundingShape = new SurroundingShape();
                        surroundingShape.setShape(shape);
                        surroundingShape.setLayerName(shape.getClassName());

                        //Get closest distance to segment
                        PointF intersectingPoint = new PointF();
                        float distance = distanceToSegment(surroundingDefinition.getPoint(), theSegment, intersectingPoint);

                        //Set distance to shape
                        surroundingShape.setDistance(distance);

                        //Find angle to intersecting point
                        //Shape's angle
                        float shapeDegree = pointPairToBearingDegrees(surroundingDefinition.getPoint(), intersectingPoint);

                        //Store angle to surrounding shape
                        surroundingShape.setAngle(getRelativeAngleWithDegrees(gazeDegree, shapeDegree));

                        surroundingShapesArray.add(surroundingShape);
                    }

                    continue;
                }

                //Check if its within range of vision
                boolean pointInView = false;

                float rayAngle1 = pointPairToBearingDegrees(surroundingDefinition.getPoint(), theSegment.get(0));
                float rayAngle2 = pointPairToBearingDegrees(surroundingDefinition.getPoint(), theSegment.get(theSegment.size() - 1));

                float relativeAngle1 = getRelativeAngleWithDegrees(gazeDegree, rayAngle1);
                float relativeAngle2 = getRelativeAngleWithDegrees(gazeDegree, rayAngle2);

                //Left of gaze of first point
                if (relativeAngle1 < 0 && Math.abs(relativeAngle1) < surroundingDefinition.getVisualLimitLeft()) {
                    //within range
                    pointInView = true;
                }

                //Right of gaze of first point
                if (relativeAngle1 > 0 && Math.abs(relativeAngle1) < surroundingDefinition.getVisualLimitRight()) {
                    //within range
                    pointInView = true;
                }

                //Left of gaze of second point
                if (relativeAngle2 < 0 && Math.abs(relativeAngle2) < surroundingDefinition.getVisualLimitLeft()) {
                    //within range
                    pointInView = true;
                }

                //Right of gaze of second point
                if (relativeAngle2 > 0 && Math.abs(relativeAngle2) < surroundingDefinition.getVisualLimitRight()) {
                    //within range
                    pointInView = true;
                }

                if (!pointInView) {
                    //No points inside view
                    continue;
                }

                outerLoop:
                for (PointF segmentPoint : theSegment) {
                    PointF castPoint = segmentPoint;

                    /* DEBUG - Display ray cast
                    // Move to point
                    Log.d("drawtag", "draw cast to path!");
                    Log.d("drawtag", "castPoint = " + castPoint.x + ", " + castPoint.y);
                    trackPath.moveTo(surroundingDefinition.getPoint().x, surroundingDefinition.getPoint().y);
                    // Link others in line
                    // Add line to next point
                    trackPath.lineTo(castPoint.x, castPoint.y);
                    */

                    ArrayList<PointF> ray = new ArrayList<>();
                    ray.add(surroundingDefinition.getPoint());
                    ray.add(segmentPoint);

                    //Continue with logic
                    //Intersection check
                    boolean lineOfSight = true;

                    //Check line with segments of interest
                    for (String layerName : segmentsOfInterest.keySet()) {
                        for (ArrayList<PointF> segment : segmentsOfInterest.get(layerName)) {

                            //On the same segment, skip
                            if (segment == theSegment) {
                                continue;
                            }

                            //Check if line intersects with segment
                            PointF intersection = getIntersectionWithRay(ray, segment);

                            //There is an intersection
                            if (intersection != null) {
                                //Check if point intersection is segment's connecting point
                                PointF point1 = new PointF((float) Math.ceil(intersection.x), (float) Math.ceil(intersection.y));
                                PointF point2 = new PointF((float) Math.ceil(castPoint.x), (float) Math.ceil(castPoint.y));
                                if (!point1.equals(point2)) {
                                    lineOfSight = false;
                                    break;
                                }
                            }
                        }
                        if (!lineOfSight) {
                            break;
                        }
                    }

                    //There was an intersection, don't add
                    if (!lineOfSight) {
                        continue;
                    }

                    finalSegmentArray.add(theSegment);

                    //Create shape for surrounding shape
                    if (shape != null) {
                        for (SurroundingShape surrShape : surroundingShapesArray) {
                            if (surrShape.getShape() == shape) {
                                continue outerLoop;
                            }
                        }

                        SurroundingShape surroundingShape = new SurroundingShape();
                        surroundingShape.setShape(shape);
                        surroundingShape.setLayerName(shape.getClassName());

                        //Get closest distance to segment
                        PointF intersectingPoint = new PointF();
                        float distance = distanceToSegment(surroundingDefinition.getPoint(), theSegment, intersectingPoint);

                        //Set disntace to shape
                        surroundingShape.setDistance(distance);

                        //Find angle to intersecting point
                        //Shape's angle
                        float shapeDegree = pointPairToBearingDegrees(surroundingDefinition.getPoint(), intersectingPoint);

                        //Store angle to surrounding shape
                        surroundingShape.setAngle(getRelativeAngleWithDegrees(gazeDegree, shapeDegree));

                        surroundingShapesArray.add(surroundingShape);
                    }
                    break;
                }
            }
        }

        /* DEBUG - Display ray cast
        RayDrawable rayDrawable = new RayDrawable(trackPath);
        rayDrawable.setPoint(surroundingDefinition.getPoint());
        Log.d("drawtag", "surroundingDefinition = " + surroundingDefinition.getPoint().x + ", " + surroundingDefinition.getPoint().y);
        controller.addComponent(rayDrawable, controller.getCurrentMap(), surroundingDefinition.getPoint());
        */

        //Sort surroundingShapesArray by distance
        Collections.sort(surroundingShapesArray, new Comparator<SurroundingShape>() {
            @Override
            public int compare(SurroundingShape o1, SurroundingShape o2) {
                Float distance1 = Float.valueOf(o1.getDistance());
                Float distance2 = Float.valueOf(o2.getDistance());
                return distance1.compareTo(distance2);
            }
        });

        //Add to surroundingElements
        surroundingElements.setShapes(surroundingShapesArray.toArray(new SurroundingShape[surroundingShapesArray.size()]));
        //NSLog(@"ray cast shapes: %f", [[NSDate date] timeIntervalSinceDate:startRays]);

        //3. Now for the icons
        //Create a path with polygonPoints
//        NSDate * startIcon =[NSDate date];
        Path path = new Path();
        if (viewPolygon.size() > 0) {
            PointF p0 = viewPolygon.get(0);
            path.moveTo(p0.x, p0.y);

            for (int i = 1; i < viewPolygon.size(); i++) {
                PointF p = viewPolygon.get(i);
                path.lineTo(p.x, p.y);
            }

            //Close the path
            path.close();
        }

        //Create icons array
        ArrayList<SurroundingIcon> containedIconsArray = new ArrayList<>();

        //Iterate through all icons on floor and check if its within this path
        ArrayList<JIconDrawable> iconsOfInterest = new ArrayList<>();

        for (String layerName : surroundingDefinition.getLayersOfInterest()) {
            ArrayList<JIconDrawable> iconsArray = new ArrayList<>();

            //TODO: does this need to be changed to surroundingDefinition.getMap()?
            MapLayer mapLayer = controller.getMapView().getCurrentMapDrawable().getMapLayer(layerName);
            if (mapLayer != null) {
                //add all icon drawables to list
                iconsArray.addAll(new ArrayList<>(Arrays.asList(mapLayer.getIconDrawables())));
            }

            iconsOfInterest.addAll(iconsArray);
        }

        for (JIconDrawable icon : iconsOfInterest) {
            Waypoint waypoint = surroundingDefinition.getMap().getWaypoints().getById(icon.getWaypointId());

            //Exists on map
            if (waypoint != null) {
                PointF wpPoint = new PointF(waypoint.getCoordinates()[0], waypoint.getCoordinates()[1]);
                //check if point exists in path
                RectF rectF = new RectF();
                path.computeBounds(rectF, true);
                Region r = new Region();
                r.setPath(path, new Region((int) rectF.left, (int) rectF.top, (int) rectF.right, (int) rectF.bottom));

                if (r.contains((int) wpPoint.x, (int) wpPoint.y)) {
                    //Is within path
                    //Check if closest segments are blocking line of sight to icons
                    //Create a ray with point to waypoint
                    boolean hasSight = true;
                    ArrayList<PointF> iconRay = new ArrayList<>();
                    iconRay.add(surroundingDefinition.getPoint());
                    iconRay.add(wpPoint);

                    for (ArrayList<PointF> segment : finalSegmentArray) {

                        PointF intersection = getIntersectionWithRay(iconRay, segment);
                        if (intersection != null) {
                            //Move on to next icon, don't add
                            hasSight = false;
                            break;
                        }
                    }

                    if (!hasSight) {
                        continue;
                    }

                    SurroundingIcon surroundingIcon = new SurroundingIcon();
                    surroundingIcon.setIcon(icon);
                    surroundingIcon.setLayerName(icon.getClassName());

                    //Find the angle
                    //Icon's angle
                    float iconDegree = pointPairToBearingDegrees(surroundingDefinition.getPoint(), wpPoint);

                    //Gaze's angle
                    gazeDegree = pointPairToBearingDegrees(surroundingDefinition.getPoint(), new PointF(gazeX, gazeY));

                    //Store angle to surrounding icon
                    surroundingIcon.setAngle(getRelativeAngleWithDegrees(gazeDegree, iconDegree));

                    //Store distance to surrounding icon
                    surroundingIcon.setDistance(distanceBetweenPoint(surroundingDefinition.getPoint(), wpPoint));

                    containedIconsArray.add(surroundingIcon);
                }
            }
        }

        //Sort containedIconsArray by distance
        Collections.sort(containedIconsArray, new Comparator<SurroundingIcon>() {
            @Override
            public int compare(SurroundingIcon o1, SurroundingIcon o2) {
                Float distance1 = Float.valueOf(o1.getDistance());
                Float distance2 = Float.valueOf(o2.getDistance());
                return distance1.compareTo(distance2);
            }
        });

        //Add array to surrounding elements
        surroundingElements.setIcons(containedIconsArray.toArray(new SurroundingIcon[containedIconsArray.size()]));

        //NSLog(@"gather icons: %f", [[NSDate date] timeIntervalSinceDate:startIcon]);
        //NSLog(@"total time: %f", [[NSDate date] timeIntervalSinceDate:startTotal]);
        return surroundingElements;
    }

    /**
     * Creates a path with a generated instruction direction popup on each decision/completion point
     *
     * @param pathPerFloors Array of {@link PathPerFloor} used to compare with user's location
     * @param pathStyle     Desired style for path drawn
     * @param popupStyle    Desired style for popup (including stroke)
     * @param textStyle     Desired style for text within popup
     */
    public void drawPathWithInstructions(@NonNull PathPerFloor[] pathPerFloors, @Nullable JStyle pathStyle, @Nullable JStyle popupStyle, @Nullable JStyle textStyle) {
        //Generate instructions
        Instruction[] instructions = createInstructionsFromPaths(pathPerFloors);

        //Iterate through all instructions
        for (Instruction instruction : instructions) {
            //Create drawable with styling parameters
            NavDirectionPopupDrawable directionPopupDrawable = new NavDirectionPopupDrawable(instruction.getText().toLowerCase().contains("arrive") ? "ARRIVE" : instruction.getDirection()
                    .toUpperCase(), popupStyle, textStyle);

            //Set instrWaypoint to be either the decision/completion point of an instruction
            Waypoint instrWaypoint = instruction.getDecisionPoint() != null ? instruction.getDecisionPoint() : instruction.getCompletionPoint();

            if (instrWaypoint != null) {
                Map currentMap = controller.getActiveVenue().getMaps().getById(instrWaypoint.getMapId());

                if (currentMap != null) {
                    controller.addComponent(directionPopupDrawable, currentMap, instrWaypoint, MapLayer.Layer_Wayfind);
                }
            }
        }

        controller.drawWayfindingPath(pathPerFloors, pathStyle);
    }

    /**
     * Custom the {@link InstructionFactory} such as visible limit and range.
     *
     * @param instructionFactory
     */
    public void setInstructionFactory(@NonNull InstructionFactory instructionFactory) {
        this.instructionFactory = instructionFactory;
    }

    /**
     * Checks if user location is within the {@param threshold} from the closest node in the
     * wayfinding path. Returns true if user location beyond threshold, false otherwise.
     *
     * @param pathPerFloors Array of {@link PathPerFloor} used to compare with user's location
     * @param threshold     Millimeter value used to define acceptable distance from wayfinding path
     */
    public boolean hasUserVeeredOffRoute(@NonNull PathPerFloor[] pathPerFloors, float threshold) {
        UserLocation userLocation = UserLocation.getInstance();
        for (PathPerFloor pathPerFloor : pathPerFloors) {
            if (userLocation.getMapId() == pathPerFloor.mapId && pathPerFloor.points.length >= 2) {
                PointF userLocationPosition = new PointF(userLocation.getX(), userLocation.getY());
                ArrayList<Float> distanceList = new ArrayList<>();
                for (int i = 0; i < pathPerFloor.points.length - 1; i++) {
                    float distance = Utilities.distanceFromPointToLine(userLocationPosition, pathPerFloor.points[i].asPointF(), pathPerFloor.points[i + 1].asPointF());
                    distanceList.add(distance);
                }
                return (Collections.min(distanceList) * controller.getCurrentMap().getMmPerPixel()) > threshold;
            }
        }
        return false;
    }

    private boolean doesSegmentWithPoint(PointF p1, PointF p2, PointF center, float radius) {
        //Before going into logic check if points are inside the circle
        //Case where both points are inside the circle isn't accounted for in the next part of the algorithm
        //Optimize performance
        boolean ptCheck1 = circleWithCenter(center, radius, p1);
        boolean ptCheck2 = circleWithCenter(center, radius, p2);

        if (ptCheck1 || ptCheck2) {
            return true;
        }

        //Continue on case where middle of segment is contained inside the circle
        PointF direction = new PointF(p2.x - p1.x, p2.y - p1.y);
        PointF vector = new PointF(p1.x - center.x, p1.y - center.y);

        float a = dotProductOfPoint(direction, direction);
        float b = 2 * dotProductOfPoint(vector, direction);
        float c = dotProductOfPoint(vector, vector) - (radius * radius);

        float discriminant = b * b - 4 * a * c;

        //No intersections
        if (discriminant < 0) {
            return false;
        } else {
            //Theres some sort of intersection..
            discriminant = (float) Math.sqrt(discriminant);

            float t1 = (-b - discriminant) / (2 * a);
            float t2 = (-b + discriminant) / (2 * a);

            if (t1 >= 0 && t1 <= 1) {
                //t1 intersects, closer than t2
                return true;
            }

            if (t2 >= 0 && t2 <= 1) {
                return true;
            }

            return false;
        }
    }

    //Method to check if point is inside a circle
    private boolean circleWithCenter(PointF center, float radius, PointF point) {
        float ptCheck = (float) Math.sqrt((point.x - center.x) * (point.x - center.x) + (point.y - center.y) * (point.y - center.y));

        if (ptCheck <= radius) {
            return true;
        }
        return false;
    }

    private float dotProductOfPoint(PointF v1, PointF v2) {
        return (v1.x * v2.x + v1.y * v2.y);
    }

    //Method to check if a ray and a segment intersects
    private PointF getIntersectionWithRay(ArrayList<PointF> ray, ArrayList<PointF> segment) {

        PointF rayA = ray.get(0);
        PointF rayB = ray.get(ray.size() - 1);

        float aRayPointX = rayA.x;
        float aRayPointY = rayA.y;

        float bRayPointX = rayB.x;
        float bRayPointY = rayB.y;

        PointF p1 = new PointF(aRayPointX, aRayPointY);
        PointF p2 = new PointF(bRayPointX, bRayPointY);

        PointF segmentA = segment.get(0);
        PointF segmentB = segment.get(segment.size() - 1);

        float aSegmentPointX = segmentA.x;
        float aSegmentPointY = segmentA.y;

        float bSegmentPointX = segmentB.x;
        float bSegmentPointY = segmentB.y;

        PointF p3 = new PointF(aSegmentPointX, aSegmentPointY);
        PointF p4 = new PointF(bSegmentPointX, bSegmentPointY);

        float d = (p2.x - p1.x) * (p4.y - p3.y) - (p2.y - p1.y) * (p4.x - p3.x);
        if (d == 0) {
            return null; // parallel lines
        }
        float u = ((p3.x - p1.x) * (p4.y - p3.y) - (p3.y - p1.y) * (p4.x - p3.x)) / d;
        float v = ((p3.x - p1.x) * (p2.y - p1.y) - (p3.y - p1.y) * (p2.x - p1.x)) / d;
        if (u < 0.0 || u > 1.0) {
            return null; // intersection point not between p1 and p2
        }
        if (v < 0.0 || v > 1.0) {
            return null; // intersection point not between p3 and p4
        }
        PointF intersection = new PointF(p1.x + u * (p2.x - p1.x), p1.y + u * (p2.y - p1.y));

        return intersection;
    }

    //Method to find the distance between 2 points
    private float distanceBetweenPoint(PointF point1, PointF point2) {
        float xDistance = (point2.x - point1.x);
        float yDistance = (point2.y - point1.y);
        float distance = (float) Math.sqrt((xDistance * xDistance) + (yDistance * yDistance));

        return distance;
    }

    //Method for calculating the absolute angle given two points
    private float pointPairToBearingDegrees(PointF startingPoint, PointF endingPoint) {
        //        90
        //        |
        //  q=2   y     q=1
        //        |
        // 180--x-+--- 0 degrees
        //        |
        //  q=3   270   q=4
        //        |
        //        |
        PointF vector = new PointF(endingPoint.x - startingPoint.x, endingPoint.y - startingPoint.y);
        double angleCalc;
        if (vector.y < 0) {
            // upper Half
            angleCalc = Math.atan2(-vector.y, vector.x);
        } else {
            angleCalc = Math.atan2(vector.y, -vector.x) + Math.PI;
        }

        return (float) (angleCalc * (180 / Math.PI));
    }

    //Method to find the angle of the item relative to the gaze direction's angle
    private float getRelativeAngleWithDegrees(float gazeAngle, float itemAngle) {
        //Consider all case scenarios

        //Abnormal cases, 1 is below 0, 1 above 0
        //fabs(itemAngle - gazeAngle) > 180
        if (Math.abs(itemAngle - gazeAngle) > 180) {
            //item below, gaze above
            //itemAngle > gazeAngle
            //360 - item + gazeAngle
            if (itemAngle > gazeAngle) {
                return (float) Math.abs(360.0 - itemAngle + gazeAngle);
            }

            //item above, gaze below
            //itemAngle < gazeAngle
            //360 - gazeAngle + itemAngle
            if (itemAngle < gazeAngle) {
                return -Math.abs(360 - gazeAngle + itemAngle);
            }
        } else {
            //All else regular
            //itemAngle > gazeAngle
            //Left
            if (itemAngle > gazeAngle) {
                return -(itemAngle - gazeAngle);
            }
            //itemAngle < gazeAngle
            //Right
            if (itemAngle < gazeAngle) {
                return gazeAngle - itemAngle;
            }
        }

        //Exactly in front, not likely with float though
        return 0;
    }

    private float getPixelsFromInches(float inches) {
        return getPixelsFromMillimeters((float) (inches * 25.4));
    }

    private float getPixelsFromMillimeters(float millimeters) {
        return millimeters / controller.getCurrentMap().getMmPerPixel();
    }

    private float distanceToSegment(PointF point, ArrayList<PointF> line, PointF intersectPoint) {
        PointF lineP1 = line.get(0);
        PointF lineP2 = line.get(line.size() - 1);

        float l2 = distanceBetweenPoint(lineP1, lineP2);
        if (l2 == 0) {
            intersectPoint.set(lineP2.x, lineP2.y);
            return distanceBetweenPoint(point, lineP2);
        }

        float t = ((point.x - lineP1.x) * (lineP2.x - lineP1.x) + (point.y - lineP1.y) * (lineP2.y - lineP1.y)) / l2;

        if (t < 0) {
            intersectPoint.set(lineP1.x, lineP1.y);
            return distanceBetweenPoint(point, lineP1);
        }
        if (t > 1) {
            intersectPoint.set(lineP2.x, lineP2.y);
            return distanceBetweenPoint(point, lineP2);
        }

        // Point of intersect
        intersectPoint = new PointF(lineP1.x + t * (lineP2.x - lineP1.x), lineP1.y + t * (lineP2.y - lineP1.y));

        return (float) Math.sqrt(distanceBetweenPoint(point, intersectPoint));
    }
}

