package com.jibestream.mapuikit;

import android.app.Activity;
import android.app.SearchManager;
import android.content.Context;
import android.database.Cursor;
import android.database.MatrixCursor;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.Typeface;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import android.graphics.drawable.ShapeDrawable;
import android.os.AsyncTask;
import android.os.Build;
import android.os.CountDownTimer;
import android.provider.BaseColumns;
import android.support.annotation.NonNull;
import android.support.v4.widget.SimpleCursorAdapter;
import android.support.v7.widget.SearchView;
import android.util.Log;
import android.view.KeyEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.inputmethod.EditorInfo;
import android.view.inputmethod.InputMethodManager;
import android.widget.AutoCompleteTextView;
import android.widget.EditText;
import android.widget.FrameLayout;
import android.widget.HorizontalScrollView;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.ScrollView;
import android.widget.TextView;

import com.daimajia.androidanimations.library.Techniques;
import com.daimajia.androidanimations.library.YoYo;
import com.jibestream.jmapandroidsdk.analytics_kit.AnalyticsEvent;
import com.jibestream.jmapandroidsdk.astar.PathPerFloor;
import com.jibestream.jmapandroidsdk.components.ActiveVenue;
import com.jibestream.jmapandroidsdk.components.Amenity;
import com.jibestream.jmapandroidsdk.components.BaseModel;
import com.jibestream.jmapandroidsdk.components.Destination;
import com.jibestream.jmapandroidsdk.components.Floor;
import com.jibestream.jmapandroidsdk.components.PathType;
import com.jibestream.jmapandroidsdk.components.Map;
import com.jibestream.jmapandroidsdk.components.Waypoint;
import com.jibestream.jmapandroidsdk.jcontroller.JController;
import com.jibestream.jmapandroidsdk.jcore.JCore;
import com.jibestream.jmapandroidsdk.rendering_engine.MapLayer;
import com.jibestream.jmapandroidsdk.rendering_engine.MapView;
import com.jibestream.jmapandroidsdk.rendering_engine.Stage;
import com.jibestream.jmapandroidsdk.rendering_engine.Transform;
import com.jibestream.jmapandroidsdk.rendering_engine.drawables.MapDrawable;
import com.jibestream.jmapandroidsdk.styles.JStyle;
import com.jibestream.jmapandroidsdk.styles.JStyleConfig;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.EventBus;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;

import static android.widget.CursorAdapter.FLAG_REGISTER_CONTENT_OBSERVER;

/**
 * Created by Ken Pangilinan on 2018-03-28.
 */

public class MapUiKit {
    private static final int SEARCH_QUERY_THRESHOLD = 1;
    private static final int TYPE_DESTINATION = 0;
    private static final int TYPE_AMENITY = 1;
    private static final int TYPE_PATH_TYPE = 2;

    private final JCore core;
    private final JController controller;
    private ActiveVenue activeVenue;
    private final Stage stage;
    private final Context context;
    private RelativeLayout uiContainer;

    private FloorSelector floorSelector;
    private Compass compass;
    private PopupDrawable popupDrawable;
    private Search search;
    private CompassOptions compassOptions;
    private JStyleConfig.UiKitStyle.Default defaultStyle;

    final private int DEFAULT_MARGIN = 20;
    private Techniques animationIn, animationOut;

    private boolean allowConcurrentPopups;
    private JStyleConfig mapTemplate;
    private static ArrayList<Typeface> typefaces;
    private HashSet<FloorButton> floorButtons;
    private LinearLayout floorButtonContainer;


    /**
     * Kit that aids in adding common UI components to Jibestream's SDK map.
     *
     * @param core       {@link JCore} reference from JMap Android SDK.
     * @param controller {@link JController} reference from JMap Android SDK.
     */
    public MapUiKit(@NonNull final JCore core, @NonNull final JController controller) {
        this.core = core;
        this.controller = controller;
        this.stage = controller.getStage();
        this.activeVenue = controller.getActiveVenue();
        this.context = controller.getContext();

        //setup the initial layout that holds all the components
        setupMapUIContainer();

        //set the Map Template coming from CMS
        setupMapTemplate();

        //set default styling
        defaultStyle = new JStyleConfig().new UiKitStyle().new Default();

        if (mapTemplate != null && mapTemplate.uiKitStyle != null
                && mapTemplate.uiKitStyle.defaultStyle != null) {
            defaultStyle = mapTemplate.uiKitStyle.defaultStyle;
        } else {
            defaultStyle = new JStyleConfig().new UiKitStyle().new Default();
        }
        // Kit default

        if (defaultStyle.activeStyle == null) {
            defaultStyle.activeStyle = new JStyleConfig().new UiKitStyle().new ActiveStyle();
            defaultStyle.activeStyle.fill = "#3e4eb8";
            defaultStyle.activeStyle.opacity = 1f;
        } else {
            if (defaultStyle.activeStyle.fill == null) {
                defaultStyle.activeStyle.fill = "#3e4eb8";
            }
            if (defaultStyle.activeStyle.opacity == null) {
                defaultStyle.activeStyle.opacity = 1f;
            }
        }

        if (defaultStyle.inactiveStyle == null) {
            defaultStyle.inactiveStyle = new JStyleConfig().new UiKitStyle().new InactiveStyle();
            defaultStyle.inactiveStyle.fill = "#ffffff";
            defaultStyle.inactiveStyle.opacity = 1f;
        } else {
            if (defaultStyle.inactiveStyle.fill == null) {
                defaultStyle.inactiveStyle.fill = "#ffffff";
            }
            if (defaultStyle.inactiveStyle.opacity == null) {
                defaultStyle.inactiveStyle.opacity = 1f;
            }
        }

        if (defaultStyle.activeFont == null) {
            defaultStyle.activeFont = new JStyleConfig().new UiKitStyle().new ActiveFont();
            defaultStyle.activeFont.fill = "#ffffff";
            defaultStyle.activeFont.fontFamily = "default";
        } else {
            if (defaultStyle.activeFont.fill == null) {
                defaultStyle.activeFont.fill = "#ffffff";
            }
            if (defaultStyle.activeFont.fontFamily == null) {
                defaultStyle.activeFont.fontFamily = "default";
            }
        }

        if (defaultStyle.inactiveFont == null) {
            defaultStyle.inactiveFont = new JStyleConfig().new UiKitStyle().new InactiveFont();
            defaultStyle.inactiveFont.fill = "#000000";
            defaultStyle.inactiveFont.fontFamily = "default";
        } else {
            if (defaultStyle.inactiveFont.fill == null) {
                defaultStyle.inactiveFont.fill = "#000000";
            }
            if (defaultStyle.inactiveFont.fontFamily == null) {
                defaultStyle.inactiveFont.fontFamily = "default";
            }
        }

        defaultStyle.padding = new float[]{10, 10, 10, 10};
        //set up a list of allowed typefaces
        setupTypefaces();
    }

    private void setupTypefaces() {
        typefaces = new ArrayList<Typeface>() {{
            add(Typeface.MONOSPACE);
            add(Typeface.DEFAULT);
            add(Typeface.DEFAULT_BOLD);
            add(Typeface.SANS_SERIF);
            add(Typeface.SERIF);
        }};
    }

    /**
     * Renders a {@link Compass} onto the map.
     *
     * @param compassOptions Pass in options to set properties such as compass position.
     * @return Compass view, used to customize natively. Returns {@code null} if compassOptions is {@code null}.
     */
    public Compass renderCompass(CompassOptions compassOptions) {

        boolean addView = false;
        if (compass == null) {
            compass = new Compass(context);
            compass.setId(R.id.map_ui_compass);
            //set the image resource
            compass.setImageResource(R.drawable.ic_compass);
            addView = true;
        }

        //set the layout of the view
        final RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(
                ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);

        //Boolean to check if the compass properties are present in the map template
        boolean isCompassPresentMapTemplate = mapTemplate != null && mapTemplate.uiKitStyle != null
                && mapTemplate.uiKitStyle.compass != null;
        //Boolean to check if the default property is present in the map template
        boolean isDefaultPresent = mapTemplate != null && mapTemplate.uiKitStyle != null
                && mapTemplate.uiKitStyle.defaultStyle != null;


        if (compassOptions == null) {
            //initialize compass options
            compassOptions = new CompassOptions();
        }

        //Priority of compass options: 1) if user has set 2) if they are present in map template 3) default values are set
        if (compassOptions.position == null || compassOptions.position.length != 2) {
            int[] defaultPosition = new int[]{1, 1};
            compassOptions.position = (isCompassPresentMapTemplate && mapTemplate.uiKitStyle.compass.position != null &&
                    mapTemplate.uiKitStyle.compass.position.length == 2) ? mapTemplate.uiKitStyle.compass.position : defaultPosition;
        }

        if (compassOptions.position[0] == 0 && compassOptions.position[1] == 0) {
            params.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
            params.addRule(RelativeLayout.ALIGN_PARENT_TOP);
        } else if (compassOptions.position[0] == 0 && compassOptions.position[1] == 1) {
            params.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
            params.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
        } else if (compassOptions.position[0] == 1 && compassOptions.position[1] == 0) {
            params.addRule(RelativeLayout.ALIGN_PARENT_TOP);
            params.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
        } else if (compassOptions.position[0] == 1 && compassOptions.position[1] == 1) {
            params.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
            params.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
        }

        if (isCompassPresentMapTemplate && mapTemplate.uiKitStyle.compass.padding != null &&
                mapTemplate.uiKitStyle.compass.padding.length == 4) {
            Float[] padding = mapTemplate.uiKitStyle.compass.padding;
            params.setMargins(Math.round(padding[3]), Math.round(padding[0]),
                    Math.round(padding[1]), Math.round(padding[2]));
        } else if (isDefaultPresent && mapTemplate.uiKitStyle.defaultStyle.padding != null &&
                mapTemplate.uiKitStyle.defaultStyle.padding.length == 4) {
            float[] padding = mapTemplate.uiKitStyle.defaultStyle.padding;
            params.setMargins(Math.round(padding[3]), Math.round(padding[0]),
                    Math.round(padding[1]), Math.round(padding[2]));
        } else {
            params.setMargins(DEFAULT_MARGIN, DEFAULT_MARGIN, DEFAULT_MARGIN, DEFAULT_MARGIN);
        }

        if (compass.getHeight() == 0 && compass.getWidth() == 0) {
            if (isCompassPresentMapTemplate) {
                float height = mapTemplate.uiKitStyle.compass.height;
                float width = mapTemplate.uiKitStyle.compass.width;
                compass.setAdjustViewBounds(true);
                //check if map template from cms has height and width
                if (height != 0)
                    compass.setMaxHeight(Math.round(height));
                if (width != 0)
                    compass.setMaxWidth(Math.round(width));
            }
        }
        compass.setOptions(compassOptions);

        //if only one icon is set, set icon/rotateIcon respectively
        if (compassOptions.icon == R.drawable.ic_compass && compassOptions.rotatedIcon != R.drawable.ic_rotated_compass) {
            compassOptions.icon = compassOptions.rotatedIcon;
        } else if (compassOptions.icon != R.drawable.ic_compass && compassOptions.rotatedIcon == R.drawable.ic_rotated_compass) {
            compassOptions.rotatedIcon = compassOptions.icon;
        }

        //set controller's rotation listener so compass can rotate along w/ map
        final CompassOptions finalCompassOptions = compassOptions;
        this.compassOptions = compassOptions;
        controller.setOnRotateListener(new MapView.OnRotateListener() {
            @Override
            public void onRotateChanged(final float v) {
                final float delta = v - compass.getRotation();

                compass.post(new Runnable() {
                    @Override
                    public void run() {
                        if (compass.isCompassNorth) {
                            compass.setImageResource(finalCompassOptions.rotatedIcon);
                            compass.isCompassNorth = false;
                        }
                        compass.setRotation(compass.getRotation() + delta);
                    }
                });

                /* Needed to add logic for onRotateListener for popupDrawable so that the onRotateListener
                logic for compass does not override popupDrawable's logic */
                if (popupDrawable != null) {
                    float deltaPopup = -v;
                    popupDrawable.setRotation(deltaPopup);
                }
            }
        });

        //if compass tapped on, reset to point map/compass north
        if (compassOptions.resetRotationOnTap) {
            compass.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    Transform transform = new Transform();
                    transform.setRotate(0);
                    controller.setMapTransform(transform, 500, new MapView.OnAnimationCallback() {
                        @Override
                        public void onAnimationComplete() {
                            compass.postDelayed(new Runnable() {
                                @Override
                                public void run() {
                                    compass.setImageResource(finalCompassOptions.icon);
                                    compass.isCompassNorth = true;
                                }
                            }, 300);
                        }

                        @Override
                        public void onAnimationInterrupted() {

                        }
                    });
                }
            });
        }

        if (addView) {
            //add compass to ui container
            uiContainer.post(new Runnable() {
                @Override
                public void run() {
                    uiContainer.addView(compass, params);
                }
            });
        }
        return compass;
    }

    /**
     * Renders a {@link Compass} onto the map.
     *
     * @return Compass view, used to customize natively. Returns {@code null} if compassOptions is {@code null}.
     */
    public Compass renderCompass() {
        return renderCompass(new CompassOptions());
    }

    /**
     * Renders a {@link FloorSelector} onto the map.
     *
     * @param floorSelectorOptions Pass in options to set properties such as floorSelector position.
     * @return FloorSelector view, used to customize natively. Returns {@code null} if floorSelectorOptions is {@code null}.
     */
    public FloorSelector renderFloorSelector(FloorSelectorOptions floorSelectorOptions) {

        EventBus.getDefault().register(this);

        boolean addView = false;
        if (floorSelector == null) {
            floorSelector = new FloorSelector(context);
            floorSelector.setId(R.id.map_ui_floor_selector);
            addView = true;
        }

        final RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.WRAP_CONTENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        //Boolean to check of floor selector property is present in map template
        boolean isFloorSelectorPresentInMapTemplate = mapTemplate != null && mapTemplate.uiKitStyle != null &&
                mapTemplate.uiKitStyle.floorSelector != null;

        if (floorSelectorOptions == null) {
            floorSelectorOptions = new FloorSelectorOptions();
            if (isFloorSelectorPresentInMapTemplate && mapTemplate.uiKitStyle.floorSelector.vertical) {
                floorSelectorOptions.setVertical(true);
            }
        }

        JStyle style;
        if (floorSelectorOptions.activeStyle == null) {

            JStyleConfig.UiKitStyle.ActiveStyle mapTemplateActiveStyle = isFloorSelectorPresentInMapTemplate &&
                    (mapTemplate.uiKitStyle.floorSelector.activeStyle != null) ?
                    mapTemplate.uiKitStyle.floorSelector.activeStyle : defaultStyle.activeStyle;

            style = new JStyle();
            if (mapTemplateActiveStyle != null) {
                if (mapTemplateActiveStyle.fill != null) {
                    style.setColor(Color.parseColor(mapTemplateActiveStyle.fill));
                }
                if (mapTemplateActiveStyle.opacity != null) {
                    style.setStrokeWidth(mapTemplateActiveStyle.opacity);
                }
            }
            floorSelectorOptions.activeStyle = style;
        }

        //Priority of floor selector options: 1) if user has set 2) if they are present in map template 3) default values are set
        if (floorSelectorOptions.position == null || floorSelectorOptions.position.length != 2) {
            int[] defaultPosition = new int[2];
            defaultPosition[0] = 1;
            defaultPosition[1] = 1;
            floorSelectorOptions.position = (isFloorSelectorPresentInMapTemplate && mapTemplate.uiKitStyle.floorSelector.position != null &&
                    mapTemplate.uiKitStyle.floorSelector.position.length == 2) ? mapTemplate.uiKitStyle.floorSelector.position : defaultPosition;
            params.setMargins(DEFAULT_MARGIN, DEFAULT_MARGIN, DEFAULT_MARGIN, DEFAULT_MARGIN);
        }


        if (floorSelectorOptions.position[0] == 0 && floorSelectorOptions.position[1] == 0) {
            floorSelectorOptions.position[0] = RelativeLayout.ALIGN_PARENT_LEFT;
            floorSelectorOptions.position[1] = RelativeLayout.ALIGN_PARENT_TOP;
            params.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
            params.addRule(RelativeLayout.ALIGN_PARENT_TOP);
        } else if (floorSelectorOptions.position[0] == 0 && floorSelectorOptions.position[1] == 1) {
            floorSelectorOptions.position[0] = RelativeLayout.ALIGN_PARENT_LEFT;
            floorSelectorOptions.position[1] = RelativeLayout.ALIGN_PARENT_BOTTOM;
            params.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
            params.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
        } else if (floorSelectorOptions.position[0] == 1 && floorSelectorOptions.position[1] == 0) {
            floorSelectorOptions.position[0] = RelativeLayout.ALIGN_PARENT_RIGHT;
            floorSelectorOptions.position[1] = RelativeLayout.ALIGN_PARENT_TOP;
            params.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
            params.addRule(RelativeLayout.ALIGN_PARENT_TOP);
        } else if (floorSelectorOptions.position[0] == 1 && floorSelectorOptions.position[1] == 1) {
            floorSelectorOptions.position[0] = RelativeLayout.ALIGN_PARENT_RIGHT;
            floorSelectorOptions.position[1] = RelativeLayout.ALIGN_PARENT_BOTTOM;
            params.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
            params.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
        }

        if (floorSelectorOptions.maxHeight == 0) {
            if (isFloorSelectorPresentInMapTemplate && mapTemplate.uiKitStyle.floorSelector.maxHeight != 0) {
                floorSelectorOptions.maxHeight = Math.round(mapTemplate.uiKitStyle.floorSelector.maxHeight);
            }
        }
        if (floorSelectorOptions.inactiveStyle == null) {

            JStyleConfig.UiKitStyle.InactiveStyle mapTemplateInactiveStyle = isFloorSelectorPresentInMapTemplate &&
                    (mapTemplate.uiKitStyle.floorSelector.inactiveStyle != null) ?
                    mapTemplate.uiKitStyle.floorSelector.inactiveStyle : defaultStyle.inactiveStyle;

            style = new JStyle();
            if (mapTemplateInactiveStyle != null) {
                if (mapTemplateInactiveStyle.fill != null) {
                    style.setColor(Color.parseColor(mapTemplateInactiveStyle.fill));
                }
                if (mapTemplateInactiveStyle.opacity != null) {
                    style.setStrokeWidth(mapTemplateInactiveStyle.opacity);
                }
            } else {
                style.setColor(Color.parseColor("#ffffff"));
            }
            floorSelectorOptions.inactiveStyle = style;
        }


        if (floorSelectorOptions.activeFontStyle == null) {

            JStyleConfig.UiKitStyle.ActiveFont mapTemplateActiveFont = isFloorSelectorPresentInMapTemplate &&
                    (mapTemplate.uiKitStyle.floorSelector.activeFont != null) ?
                    mapTemplate.uiKitStyle.floorSelector.activeFont : defaultStyle.activeFont;

            style = new JStyle();
            if (mapTemplateActiveFont != null) {
                if (mapTemplateActiveFont.fill != null) {
                    style.setColor(Color.parseColor(mapTemplateActiveFont.fill));
                }
                if (mapTemplateActiveFont.fontFamily != null) {
                    String fontFamily = mapTemplateActiveFont.fontFamily;
                    Typeface typeface = Typeface.create(fontFamily, Typeface.NORMAL);
                    if (typefaces.contains(typeface)) {
                        style.setTypeFace(typeface);
                    }
                }
            }
            floorSelectorOptions.activeFontStyle = style;
        }

        if (floorSelectorOptions.inactiveFontStyle == null) {

            JStyleConfig.UiKitStyle.InactiveFont mapTemplateInActiveFont = isFloorSelectorPresentInMapTemplate &&
                    (mapTemplate.uiKitStyle.floorSelector.inactiveFont != null) ?
                    mapTemplate.uiKitStyle.floorSelector.inactiveFont : defaultStyle.inactiveFont;

            style = new JStyle();
            if (mapTemplateInActiveFont != null) {
                if (mapTemplateInActiveFont.fill != null) {
                    style.setColor(Color.parseColor(mapTemplateInActiveFont.fill));
                }
                if (mapTemplateInActiveFont.fontFamily != null) {
                    String fontFamily = mapTemplateInActiveFont.fontFamily;
                    Typeface typeface = Typeface.create(fontFamily, Typeface.NORMAL);
                    if (typefaces.contains(typeface)) {
                        style.setTypeFace(typeface);
                    }
                }
            }
            floorSelectorOptions.inactiveFontStyle = style;
        }

        //set the options
        floorSelector.setOptions(floorSelectorOptions);

        float[] padding = isFloorSelectorPresentInMapTemplate &&
                (mapTemplate.uiKitStyle.floorSelector.padding != null) ?
                mapTemplate.uiKitStyle.floorSelector.padding : defaultStyle.padding;

        params.setMargins(Math.round(padding[3]), Math.round(padding[0]),
                Math.round(padding[1]), Math.round(padding[2]));

        floorSelector.setLayoutParams(params); //causes layout update

        final LinearLayout floorButtonContainer = new LinearLayout(context);
        floorButtonContainer.setId(R.id.floor_container);

        final FrameLayout scrollView;

        boolean isVerticalBottom = false;
        boolean isHorizontalRight = false;

        //setup floor selector based on options configured
        if (floorSelectorOptions.vertical) {
            scrollView = new ScrollView(context);

            animationIn = Techniques.FadeInDown;
            animationOut = Techniques.FadeOutUp;

            floorSelector.setOrientation(LinearLayout.VERTICAL);
            floorButtonContainer.setOrientation(LinearLayout.VERTICAL);

            isVerticalBottom = floorSelectorOptions.position[1] == RelativeLayout.ALIGN_PARENT_BOTTOM;
        } else {
            scrollView = new HorizontalScrollView(context);

            animationIn = Techniques.FadeInLeft;
            animationOut = Techniques.FadeOutLeft;

            floorSelector.setOrientation(LinearLayout.HORIZONTAL);
            floorButtonContainer.setOrientation(LinearLayout.HORIZONTAL);

            isHorizontalRight = floorSelectorOptions.position[0] == RelativeLayout.ALIGN_PARENT_RIGHT;
        }

        //scroll view by default is invisible
        scrollView.setId(R.id.scroll_view);
        scrollView.setVisibility(View.INVISIBLE);

        //set the layout of the scroll view
        LinearLayout.LayoutParams scrollViewParams = new LinearLayout.LayoutParams(
                LinearLayout.LayoutParams.MATCH_PARENT, LinearLayout.LayoutParams.WRAP_CONTENT);
        scrollViewParams.weight = 1.0f;
        scrollView.setLayoutParams(scrollViewParams);
        scrollView.setPadding(0, 0, 0, 0);
        if (addView) {
            scrollView.post(new Runnable() {
                @Override
                public void run() {
                    scrollView.addView(floorButtonContainer);
                }
            });
        }

        //adjust how to animate button based on positioning on map
        if (isVerticalBottom || isHorizontalRight) {
            if (addView) {
                floorSelector.addView(scrollView, 0);
            }
            if (isVerticalBottom) {
                animationIn = Techniques.FadeInUp;
                animationOut = Techniques.FadeOutDown;
            } else {
                animationIn = Techniques.FadeInRight;
                animationOut = Techniques.FadeOutRight;
            }
        } else {
            if (addView) {
                floorSelector.addView(scrollView);
            }
        }

        //setup main floor button
        setupSelectedFloorButton(floorSelectorOptions, floorButtonContainer);
        this.floorButtonContainer = floorButtonContainer;
        /*if (addView) {
            //add and setup floor buttons
            setupFloorButtons(floorButtonContainer, floorSelectorOptions);
        }*/

        if (addView) {
            //add floor selector to ui container
            uiContainer.post(new Runnable() {
                @Override
                public void run() {
                    uiContainer.addView(floorSelector, params);
                }
            });
        }


        return floorSelector;
    }

    /**
     * Renders a {@link FloorSelector} onto the map.
     *
     * @return FloorSelector view, used to customize natively. Returns {@code null} if floorSelectorOptions is {@code null}.
     */
    public FloorSelector renderFloorSelector() {
        return renderFloorSelector(new FloorSelectorOptions());
    }

    /**
     * Renders a {@link Search} onto the map.
     *
     * @param searchOptions Pass in options to set properties such as search position.
     * @return Search view, used to customize natively. Returns {@code null} if searchOptions is {@code null}.
     */
    public Search renderSearch(SearchOptions searchOptions, @NonNull final Search.OnItemSelectListener onItemSelectListener) {

        if (onItemSelectListener == null) {
            return null;
        }
        boolean addView = false;
        if (search == null) {
            search = new Search(context);
            search.setId(R.id.map_ui_search);
            search.setIconified(false);
            search.clearFocus();

            addView = true;
        }
        if (searchOptions == null) {
            searchOptions = new SearchOptions(null);
        }
        //set the options
        search.setOptions(searchOptions);

        search.setQueryHint(searchOptions.hint);

        // populate with destinations, amenities and path types if searchArray is null
        if (searchOptions.searchArray == null) {
            Destination[] destinations = controller.getActiveVenue().getDestinations().getAll();
            Amenity[] amenities = controller.getActiveVenue().getAmenities().getAll();
            PathType[] pathTypes = controller.getActiveVenue().getPathTypes().getAll();
            ArrayList<BaseModel> baseModelArrayList = new ArrayList<>();

            baseModelArrayList.addAll(Arrays.asList(destinations));
            baseModelArrayList.addAll(Arrays.asList(amenities));
            baseModelArrayList.addAll(Arrays.asList(pathTypes));

            searchOptions.searchArray = new BaseModel[baseModelArrayList.size()];
            searchOptions.searchArray = baseModelArrayList.toArray(searchOptions.searchArray);
        }

        //add rounded corners to search, only if greater than or equal to API level 21
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            search.setBackground(context.getDrawable(R.drawable.rounded_corner));
        }

        //set the layout of the view
        final RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.WRAP_CONTENT);

        if (mapTemplate != null && mapTemplate.uiKitStyle != null &&
                mapTemplate.uiKitStyle.search != null && mapTemplate.uiKitStyle.search.padding != null
                && mapTemplate.uiKitStyle.search.padding.length == 4) {

            Float[] margins = mapTemplate.uiKitStyle.search.padding;
            params.setMargins(Math.round(margins[3]), Math.round(margins[0]),
                    Math.round(margins[1]), Math.round(margins[2]));
        } else {
            params.setMargins(DEFAULT_MARGIN, DEFAULT_MARGIN, DEFAULT_MARGIN, DEFAULT_MARGIN);
        }

        if (mapTemplate != null && mapTemplate.uiKitStyle != null &&
                mapTemplate.uiKitStyle.search != null && mapTemplate.uiKitStyle.search.position != null
                && mapTemplate.uiKitStyle.search.position.length == 2) {
            Float[] position = mapTemplate.uiKitStyle.search.position;
            params.addRule(Math.round(position[0]), Math.round(position[1]));
        } else {
            params.addRule(searchOptions.position);
        }
        search.setLayoutParams(params); //causes layout update

        //Set the threshold explicitly to 1 because default threshold for SearchView is 2
        final AutoCompleteTextView autoCompleteTextView = search.findViewById(R.id.search_src_text);
        autoCompleteTextView.setThreshold(SEARCH_QUERY_THRESHOLD);
        autoCompleteTextView.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);

        //Close button on the SearchView needs to clear the query and hide the keyboard
        ImageView closeButton = search.findViewById(R.id.search_close_btn);
        closeButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                //Find EditText view
                EditText editText = search.findViewById(R.id.search_src_text);
                //Clear the text from EditText view
                editText.setText("");
                //Clear query
                search.setQuery("", false);
                //Collapse the action view
                search.onActionViewCollapsed();
                //Set the iconified back to false
                search.setIconified(false);
                //Remove the focus from search
                search.clearFocus();

                hideKeyboard((Activity) context);
                //clear any previous maps
                if (controller != null) {
                    controller.clearWayfindingPath();
                }
            }
        });

        search.post(new Runnable() {
            @Override
            public void run() {

                search.setSuggestionsAdapter(new SimpleCursorAdapter(
                        context, android.R.layout.simple_list_item_1, null,
                        new String[]{SearchManager.SUGGEST_COLUMN_TEXT_1},
                        new int[]{android.R.id.text1}, FLAG_REGISTER_CONTENT_OBSERVER));

                search.setOnSuggestionListener(new SearchView.OnSuggestionListener() {
                    @Override
                    public boolean onSuggestionSelect(int position) {

                        return false;
                    }

                    @Override
                    public boolean onSuggestionClick(int position) {
                        Cursor cursor = search.getSuggestionsAdapter().getCursor();
                        cursor.moveToPosition(position);

                        int id = cursor.getInt(0);
                        int type = cursor.getInt(2);

                        BaseModel baseModel = null;
                        if (type == TYPE_DESTINATION) {
                            baseModel = controller.getActiveVenue().getDestinations().getById(id);
                        } else if (type == TYPE_AMENITY) {
                            baseModel = controller.getActiveVenue().getAmenities().getById(id);
                        } else if (type == TYPE_PATH_TYPE) {
                            baseModel = controller.getActiveVenue().getPathTypes().getById(id);
                        }

                        onItemSelectListener.onItemClick(baseModel);

                        return false;
                    }
                });
            }
        });

        search.setOnQueryTextListener(new SearchView.OnQueryTextListener() {

            @Override
            public boolean onQueryTextSubmit(String query) {
                // if user presses enter, do default search, ex:
                return query.length() >= SEARCH_QUERY_THRESHOLD;
            }

            @Override
            public boolean onQueryTextChange(String newText) {
                if (newText.length() >= SEARCH_QUERY_THRESHOLD) {
                    new FetchAutoCompleteSuggestionsTask().execute(newText);
                }
                return true;
            }
        });

        if (addView) {
            //add search to ui container
            uiContainer.post(new Runnable() {
                @Override
                public void run() {
                    uiContainer.addView(search);
                }
            });
        }
        return search;
    }


    /**
     * Renders a {@link PopupDrawable} onto the map.
     *
     * @param popupOptions Pass in options to set properties such as currentPopupDrawable coordinates.
     * @return Popup drawable, used to customize natively. Returns {@code null} if popupOptions is {@code null}.
     */
    public PopupDrawable renderPopup(PopupDrawableOptions popupOptions, @NonNull PointF coordinates) {

        //Boolean to check if Popup property is present in map template
        boolean isPopupPresentInMapTemplate = mapTemplate != null &&
                mapTemplate.uiKitStyle != null && mapTemplate.uiKitStyle.popup != null;

        if (popupOptions == null) {
            popupOptions = new PopupDrawableOptions("Default popup text");
            if (isPopupPresentInMapTemplate && mapTemplate.uiKitStyle.popup.showActionButton) {
                popupOptions.setShowActionButton(true);
            }
        }

        JStyle style;

        //Priority of popup options: 1) if user has set 2) if they are present in map template 3) default values are set
        if (popupOptions.titleTextStyle == null) {
            popupOptions.titleTextStyle = new JStyle();
            if (isPopupPresentInMapTemplate && mapTemplate.uiKitStyle.popup.titleTextFont != null) {
                String titleTextFontFamily = mapTemplate.uiKitStyle.popup.titleTextFont.fontFamily;
                String titleTextColor = mapTemplate.uiKitStyle.popup.titleTextFont.fill;
                if (titleTextFontFamily != null) {
                    Typeface typeface = Typeface.create(titleTextFontFamily, Typeface.NORMAL);
                    if (typefaces.contains(typeface)) {
                        popupOptions.titleTextStyle.setTypeFace(typeface);
                    }
                }
                if (titleTextColor != null) {
                    popupOptions.titleTextStyle.setColor(Color.parseColor(titleTextColor));
                }
            } else {
                String subTextFontFamily = defaultStyle.activeFont.fontFamily;
                popupOptions.titleTextStyle.setAntiAlias(true);
                popupOptions.titleTextStyle.setColor(Color.parseColor("#000000")); //defaultStyle.activeStyle.fill
                Typeface typeface = Typeface.create(subTextFontFamily, Typeface.NORMAL);
                if (typefaces.contains(typeface)) {
                    popupOptions.titleTextStyle.setTypeFace(typeface);
                }
            }
            popupOptions.setTitleText(popupOptions.titleText != null ? popupOptions.titleText : context.getString(R.string.popup_default_text));
        }
        if (popupOptions.subTextStyle == null && isPopupPresentInMapTemplate) {

            popupOptions.subTextStyle = new JStyle();
            if (mapTemplate.uiKitStyle.popup.subTextFont != null) {
                String subTextFontFamily = mapTemplate.uiKitStyle.popup.subTextFont.fontFamily;
                String subTextColor = mapTemplate.uiKitStyle.popup.subTextFont.fill;
                if (subTextFontFamily != null) {
                    Typeface typeface = Typeface.create(subTextFontFamily, Typeface.NORMAL);
                    if (typefaces.contains(typeface)) {
                        popupOptions.subTextStyle.setTypeFace(typeface);
                    }
                }
                if (subTextColor != null) {
                    popupOptions.subTextStyle.setColor(Color.parseColor(subTextColor));
                }
            } else {
                String subTextFontFamily = defaultStyle.activeFont.fontFamily;
                String subTextColor = "#000000";//defaultStyle.activeFont.fill;
                Typeface typeface = Typeface.create(subTextFontFamily, Typeface.NORMAL);
                if (typefaces.contains(typeface)) {
                    popupOptions.subTextStyle.setTypeFace(typeface);
                }
                popupOptions.subTextStyle.setColor(Color.parseColor(subTextColor));
            }
            popupOptions.setSubText(popupOptions.subText != null ? popupOptions.subText : context.getString(R.string.set_subtext));
        }

        if (popupOptions.actionButtonStyle == null) {
            popupOptions.actionButtonStyle = new JStyle();
            if (isPopupPresentInMapTemplate && mapTemplate.uiKitStyle.popup.actionButtonStyle != null) {
                String actionButtonColor = mapTemplate.uiKitStyle.popup.actionButtonStyle.fill;
                if (actionButtonColor != null) {
                    popupOptions.actionButtonStyle.setColor(Color.parseColor(actionButtonColor));
                }
            } else {
                String actionButtonColor = "#1464FA";//defaultStyle.activeStyle.fill;
                popupOptions.actionButtonStyle.setColor(Color.parseColor(actionButtonColor));
            }
            popupOptions.setShowActionButton(true);
            popupOptions.setActionButtonText(popupOptions.actionButtonText != null ? popupOptions.actionButtonText : context.getString(R.string.popup_action_button_text));
        }

        if (popupOptions.actionButtonTextStyle == null) {
            popupOptions.actionButtonTextStyle = new JStyle();
            if (isPopupPresentInMapTemplate && mapTemplate.uiKitStyle.popup.actionButtonTextFont != null) {
                String actionButtonTextFontFamily = mapTemplate.uiKitStyle.popup.actionButtonTextFont.fontFamily;
                String actionButtonTextColor = mapTemplate.uiKitStyle.popup.actionButtonTextFont.fill;
                if (actionButtonTextColor != null) {
                    popupOptions.actionButtonTextStyle.setColor(Color.parseColor(actionButtonTextColor));
                }
                Typeface typeface = Typeface.create(actionButtonTextFontFamily, Typeface.NORMAL);
                if (typefaces.contains(typeface)) {
                    popupOptions.actionButtonTextStyle.setTypeFace(typeface);
                }
            } else {
                String actionButtonTextFontFamily = defaultStyle.activeFont.fontFamily;
                String actionButtonTextColor = "#ffffff";//defaultStyle.inactiveStyle.fill;
                popupOptions.actionButtonTextStyle.setColor(Color.parseColor(actionButtonTextColor));
                Typeface typeface = Typeface.create(actionButtonTextFontFamily, Typeface.NORMAL);
                if (typefaces.contains(typeface)) {
                    popupOptions.actionButtonTextStyle.setTypeFace(typeface);
                }
                popupOptions.setActionButtonText("Default text");
            }
        }

        if (popupOptions.popupStyle == null) {
            popupOptions.popupStyle = new JStyle();
            if (isPopupPresentInMapTemplate && mapTemplate.uiKitStyle.popup.popupStyle != null) {
                String popupColor = mapTemplate.uiKitStyle.popup.popupStyle.fill;
                if (popupColor != null) {
                    popupOptions.popupStyle.setColor(Color.parseColor(popupColor));
                }

                Float opacity = mapTemplate.uiKitStyle.popup.popupStyle.opacity;
                if (opacity != null) {
                    popupOptions.popupStyle.setAlpha((int) (opacity * 255));
                }

            } else {
                String popupStyleFill = "#ffffff";
                popupOptions.popupStyle.setColor(Color.parseColor(popupStyleFill));
            }
        }

        popupOptions.setCoordinates(coordinates);

        if (!allowConcurrentPopups) {
            //remove previously added popups from map
            for (MapDrawable mapDrawable : controller.getMapDrawables()) {
                MapLayer mapLayer = mapDrawable.getMapLayer("Popups");
                if (mapLayer != null) {
                    mapLayer.clearCustomDrawables();
                }
            }
        }

        popupDrawable = new PopupDrawable(popupOptions);
        //set the options
        popupDrawable.setOptions(popupOptions);

        //add popup drawable to map
        controller.addComponent(popupDrawable, popupOptions.map != null ?
                popupOptions.map : controller.getCurrentMap(), popupOptions.coordinates, "Popups");

        popupDrawable.setRotation(-controller.getMapView().getCurrentRotation());

        //set controller's rotation listener so popupDrawable can rotate along w/ map
        controller.setOnRotateListener(new MapView.OnRotateListener() {
            @Override
            public void onRotateChanged(final float v) {
                //Delta for setting rotation parameter on popupDrawable
                final float delta = -v;
                popupDrawable.setRotation(delta);

                /* Needed to add logic for onRotateListener for compass again so that the onRotateListener
                   logic for popup does not override compass's logic */
                //Delta for setting rotation parameter on compass
                if (compass != null) {
                    final float deltaCompass = v - compass.getRotation();

                    compass.post(new Runnable() {
                        @Override
                        public void run() {
                            if (compass.isCompassNorth) {
                                compass.setImageResource(compassOptions.rotatedIcon);
                                compass.isCompassNorth = false;
                            }
                            compass.setRotation(compass.getRotation() + deltaCompass);
                        }
                    });
                }
            }
        });

        return popupDrawable;
    }

    public void setAllowConcurrentPopups(final boolean allowConcurrentPopups) {
        this.allowConcurrentPopups = allowConcurrentPopups;
    }

    /**
     * @return layout that holds all the map UI components.
     */
    public RelativeLayout getUiContainer() {
        return uiContainer;
    }

    private void setupSelectedFloorButton(final FloorSelectorOptions floorSelectorOptions, final LinearLayout floorButtonContainer) {
        Floor currentFloor = controller.getCurrentFloor();
        final FloorButton selectedFloorButton = floorSelector.findViewById(R.id.main_floor_button);

        selectedFloorButton.setMain(true);
        styleButton(selectedFloorButton, currentFloor, floorSelectorOptions.activeStyle, floorSelectorOptions.activeFontStyle);

        selectedFloorButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (!selectedFloorButton.isSelected()) {
                    //expand menu
                    expandFloorSelector(floorButtonContainer, floorSelectorOptions);
                } else {
                    //collapse menu
                    collapseFloorSelector();
                }
            }
        });
    }

    private void collapseFloorSelector() {
        final FloorButton mainButton = floorSelector.findViewById(R.id.main_floor_button);
        mainButton.setSelected(false);

        FrameLayout scrollView = floorSelector.findViewById(R.id.scroll_view);

        YoYo.with(animationOut)
                .duration(700)
                .playOn(scrollView);

        removeFloorButtons();
    }

    private void removeFloorButtons() {
        for (final FloorButton floorButton : floorButtons) {
            floorButtonContainer.removeView(floorButton);
        }
        floorButtons.clear();
    }

    private void expandFloorSelector(LinearLayout floorButtonContainer, FloorSelectorOptions floorSelectorOptions) {

        //setup floor buttons
        setupFloorButtons(floorButtonContainer, floorSelectorOptions);

        final FloorButton mainButton = floorSelector.findViewById(R.id.main_floor_button);

        //wait for some milliseconds before expanding the floor button and for the animation to clearly show
        new CountDownTimer(200, 200) {
            public void onTick(long millisUntilFinished) {
                mainButton.setEnabled(false);
            }

            public void onFinish() {
                //final FloorButton mainButton = floorSelector.findViewById(R.id.main_floor_button);
                mainButton.setSelected(true);

                FrameLayout scrollView = floorSelector.findViewById(R.id.scroll_view);

                scrollView.setVisibility(View.VISIBLE);
                YoYo.with(animationIn)
                        .duration(700)
                        .playOn(scrollView);
                mainButton.setEnabled(true);

            }
        }.start();
    }

    private void setupFloorButtons(final LinearLayout floorButtonContainer, final FloorSelectorOptions floorSelectorOptions) {
        final Floor[] floors = controller.getCurrentBuilding().getFloors().getAll();
        //Arraylist used to remove floor buttons on collapse animation
        floorButtons = new HashSet<>();
        for (final Floor floor : floors) {
            final FloorButton floorButton = new FloorButton(context);
            styleButton(floorButton, floor, floorSelectorOptions.inactiveStyle, floorSelectorOptions.inactiveFontStyle);
            floorButtons.add(floorButton);
            floorButtonContainer.post(new Runnable() {
                @Override
                public void run() {
                    if (floorSelectorOptions.position[1] == RelativeLayout.ALIGN_PARENT_BOTTOM ||
                            floorSelectorOptions.position[0] == RelativeLayout.ALIGN_PARENT_RIGHT) {
                        floorButtonContainer.addView(floorButton, 0);
                    } else {
                        floorButtonContainer.addView(floorButton);
                    }
                }
            });

            floorButton.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    final FloorButton mainButton = floorSelector.findViewById(R.id.main_floor_button);
                    if (mainButton.isSelected()) {
                        mainButton.setText(floorButton.getText());
                        collapseFloorSelector();
                        controller.showMap(floor.getMap(), new JController.OnMapShownCallback() {
                            @Override
                            public void onBeforeMapShown(MapDrawable mapDrawable) {
                            }

                            @Override
                            public void onMapShown() {

                            }

                            @Override
                            public void onError(String s) {

                            }
                        });
                    }
                }

            });
        }
    }

    private void styleButton(FloorButton floorButton, Floor floor, JStyle style, JStyle fontStyle) {
        floorButton.setMapId(floor.getMap().getId());
        Drawable background = floorButton.getBackground();

        //button is styled this way to preserve the rounded corners
        if (background instanceof ShapeDrawable) {
            ShapeDrawable shapeDrawable = ((ShapeDrawable) background);
            shapeDrawable.getPaint().setColor(style.getColor());
        } else if (background instanceof GradientDrawable) {
            GradientDrawable gradientDrawable = ((GradientDrawable) background);
            gradientDrawable.setColor(style.getColor());
            gradientDrawable.setStroke(style.getStrokeColor(), (int) style.getStrokeWidth());
        } else if (background instanceof ColorDrawable) {
            ColorDrawable colorDrawable = ((ColorDrawable) background);
            colorDrawable.setColor(style.getColor());
        }
        floorButton.setText(floor.getShortName() != null ? floor.getShortName() : "");
        floorButton.setTextColor(fontStyle.getColor());
        floorButton.setTextSize(fontStyle.getTextSize());
        floorButton.setTypeface(fontStyle.getTypeface());
    }

    private void setupMapUIContainer() {
        uiContainer = new RelativeLayout(context);
        uiContainer.setId(R.id.map_ui_container);

        uiContainer.setLayoutParams(new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.MATCH_PARENT));

        //add ui container to stage view
        stage.post(new Runnable() {
            @Override
            public void run() {
                stage.addView(uiContainer);
            }
        });
    }

    private void setupMapTemplate() {
        mapTemplate = activeVenue.getDefaultMapTemplate();

    }

    private class FetchAutoCompleteSuggestionsTask extends AsyncTask<String, Void, Cursor> {

        private final String[] sAutocompleteColNames = new String[]{
                BaseColumns._ID,                        // necessary for adapter
                SearchManager.SUGGEST_COLUMN_TEXT_1,    // the full search term
                "type"                                  // base model type
        };

        @Override
        protected Cursor doInBackground(String... params) {
            Object[] baseModels = JController.getObjectsInArrayByString
                    (search.getOptions().searchArray, params[0].toLowerCase(), null, search.getOptions().maxResults);

            MatrixCursor cursor = new MatrixCursor(sAutocompleteColNames);

            for (Object object : baseModels) {
                BaseModel baseModel = (BaseModel) object;
                String name = null;
                Integer type = null;
                if (baseModel instanceof Destination) {
                    name = ((Destination) baseModel).getName();
                    type = TYPE_DESTINATION;
                } else if (baseModel instanceof Amenity) {
                    name = ((Amenity) baseModel).getName();
                    type = TYPE_AMENITY;
                } else if (baseModel instanceof PathType) {
                    name = ((PathType) baseModel).getName();
                    type = TYPE_PATH_TYPE;
                }

                Object[] row = new Object[]{baseModel.getId(), name, type};
                cursor.addRow(row);
            }
            return cursor;
        }

        @Override
        protected void onPostExecute(Cursor result) {
            search.getSuggestionsAdapter().changeCursor(result);
        }
    }

    @Subscribe
    public void onEvent(AnalyticsEvent event) {
        if (event.getKey() == "JMAP_CONTROLLER_MAP_SHOW") {
            // Get floor changed id
            Map map = controller.getActiveVenue().getMaps().getById(Integer.valueOf(event.getData("mapId")));

            // Validate map exists
            if (map != null) {
                final Floor currentFloor = controller.getCurrentBuilding().getFloors().getByMap(map);

                // Validate floor exists
                if (currentFloor != null) {
                    final FloorButton mainButton = floorSelector.findViewById(R.id.main_floor_button);
                    // Run this on main thread so it would update properly
                    ((Activity) context).runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            mainButton.setText(currentFloor.getShortName());
                        }
                    });
                }
            }
        }
    }

    private void hideKeyboard(Activity activity) {
        InputMethodManager imm = (InputMethodManager) activity.getSystemService(Activity.INPUT_METHOD_SERVICE);
        //Find the currently focused view, so we can grab the correct window token from it.
        View view = activity.getCurrentFocus();
        //If no view currently has focus, create a new one, just so we can grab a window token from it
        if (view == null) {
            view = new View(activity);
        }
        if (imm != null) {
            imm.hideSoftInputFromWindow(view.getWindowToken(), 0);
        }
    }
}


