package com.jibestream.assetkit;

import android.graphics.PointF;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.util.SparseArray;

import com.jibestream.jmapandroidsdk.astar.ASNode;
import com.jibestream.jmapandroidsdk.astar.PathPerFloor;
import com.jibestream.jmapandroidsdk.astar.Point;
import com.jibestream.jmapandroidsdk.components.Map;
import com.jibestream.jmapandroidsdk.components.Waypoint;
import com.jibestream.jmapandroidsdk.jcontroller.JController;
import com.jibestream.jmapandroidsdk.rendering_engine.MapLayer;
import com.jibestream.jmapandroidsdk.rendering_engine.moving_objects.UserLocation;

import java.util.ArrayList;

/**
 * Created by louieyuen on 2017-03-27.
 */

public class AssetKit {
    private SparseArray<Asset> assets;
    private JController controller;
    private Asset wayfindingAsset;
    private int rerouteInterval;
    private Runnable autoRerouteRunnable;
    private Handler handler;

    /**
     * Constructor for initializing an AssetKit with a {@link JController}.
     *
     * @param controller The controller initialized in the android SDK.
     */
    public AssetKit(JController controller) {
        assets = new SparseArray<>();
        this.controller = controller;
        this.rerouteInterval = 0;
        handler = new Handler(Looper.getMainLooper());
    }

    /**
     * Update {@link Asset} to {@link PointF} on the {@link Map} with animation.
     *
     * @param assetToUpdate The asset passed in to be updated.
     * @param position      The new position of the asset.
     * @param map           The destination map to update the asset to.
     */
    public void updateAssetWithPosition(Asset assetToUpdate, PointF position, Map map) {
        Asset asset = getAssetById(assetToUpdate.getId());

        PointF newPosition = new PointF(position.x, position.y);
        if (asset == null) {
            assetToUpdate.setPosition(newPosition);
            //Doesn't exist, create it
            controller.addMovingObject(assetToUpdate, map, newPosition, MapLayer.Layer_Moving_Objects);
            assets.append(assetToUpdate.getId(), assetToUpdate);
        } else {
            //Check if new asset with same ID is trying to be added
            if (asset != assetToUpdate) {
                //Duplicate Id, do not add
                Log.i("AssetKit", "Attempted to add asset with duplicate ID: " + asset.getId());
                return;
            }

            controller.updateMovingObject(assetToUpdate, map, newPosition, true);
        }
    }

    /**
     * Gets the nearest {@link Asset} of specified type to {@link Point}
     *
     * @param type  The type string to search for.
     * @param point The reference point to search from.
     * @return {@link Asset} closest to the point with matching type.
     */
    public Asset getNearestAssetOfTypeToPoint(String type, Point point) {
        Asset closestAsset = null;
        float closestDistance = Float.MAX_VALUE;

        for (Asset asset : this.getAssetsByType(type)) {
            //Check for matching floor
            if (asset.getMapId() == (int) point.z && asset.getType().equals(type)) {
                //Compare distance
                float distance = this.distanceBetweenPoints(new PointF((float) point.x, (float) point.y), asset.getPosition());
                if (distance < closestDistance) {
                    closestDistance = distance;
                    closestAsset = asset;
                }
            }
        }

        return closestAsset;
    }

    /**
     * Shows all {@link Asset}s of given type.
     *
     * @param type The type string for displaying assets.
     */
    public void showAssetOfType(String type) {
        for (int i = 0; i < assets.size(); i++) {
            if (assets.valueAt(i).getType().equalsIgnoreCase(type)) {
                assets.valueAt(i).setVisible(true);
            }
        }
    }

    /**
     * Hides all {@link Asset}s of given type.
     *
     * @param type The type string for displaying assets.
     */
    public void hideAssetOfType(String type) {
        for (int i = 0; i < assets.size(); i++) {
            if (assets.valueAt(i).getType().equalsIgnoreCase(type)) {
                assets.valueAt(i).setVisible(false);
            }
        }
    }

    /**
     * Removes an {@link Asset}.
     *
     * @param asset The asset to be removed.
     */
    public void removeAsset(Asset asset) {
        assets.remove(asset.getId());
    }

    /**
     * Gets an {@link Asset} by ID.
     *
     * @param id The unique int ID associated to the asset.
     * @return {@link Asset} object with matching ID or null if none is found.
     */
    public Asset getAssetById(int id) {
        return assets.get(id);
    }

    /**
     * Gets all {@link Asset}s of given type.
     *
     * @param type The type string for getting all assets.
     * @return An array of {@link Asset}s with matching type.
     */
    public Asset[] getAssetsByType(String type) {
        ArrayList<Asset> assetArrayList = new ArrayList<>();
        for (int i = 0; i < assets.size(); i++) {
            if (assets.valueAt(i).getType().equalsIgnoreCase(type)) {
                assetArrayList.add(assets.valueAt(i));
            }
        }

        return assetArrayList.toArray(new Asset[assetArrayList.size()]);
    }

    /**
     * Clears the wayfinding path for asset wayfinding.
     */
    public void clearWayfindingPath() {
        this.rerouteInterval = 0;
        this.wayfindingAsset = null;
        controller.clearWayfindingPath();
    }

    /**
     * Wayfind and draw a path to an {@link Asset} from {@link Point}.
     *
     * @param asset The asset to wayfind to.
     * @param point The point to start wayfinding from.
     */
    public void wayfindToAssetFromPoint(Asset asset, Point point) {
        this.wayfindingAsset = asset;

        //Extract asset's nearest WP
        Point assetPoint = new Point(asset.getX(), asset.getY(), asset.getMapId());

        Waypoint fromWp = controller.getNearestWaypointToPoint(point);
        Waypoint toWp = controller.getNearestWaypointToPoint(assetPoint);

        //Clear wayfind path
        controller.clearWayfindingPath();

        if (fromWp != null && toWp != null) {
            PathPerFloor[] pathsPerFloor = controller.wayfindBetweenWaypoints(fromWp, toWp);

            //Draw wayfinding path
            controller.drawWayfindingPath(pathsPerFloor);
        } else {
            //From or to is null
            if (fromWp == null) {
                Log.d("assetkit", "Unable to get nearest waypoint to point.");
            } else {
                Log.d("assetkit", "Unable to get nearest waypoint to asset.");
            }
        }
    }

    /**
     * Wayfind and draw a path from the {@link com.jibestream.jmapandroidsdk.rendering_engine.moving_objects.UserLocation} to an {@link Asset} with auto rerouting.
     *
     * @param asset           The asset to wayfind to.
     * @param rerouteInterval An interval to determine whether to re-draw the path when the asset has been updated.
     */
    public void wayfindFromUserLocationToAssetWithAutoReroute(final Asset asset, final int rerouteInterval) {
        if (rerouteInterval > 0) {
            this.rerouteInterval = rerouteInterval;
        } else {
            this.rerouteInterval = 0;
        }

        controller.clearWayfindingPath();
        Point userPoint = new Point(UserLocation.getInstance().getPosition().x, UserLocation.getInstance().getPosition().y, UserLocation.getInstance().getMapId());
        wayfindToAssetFromPoint(asset, userPoint);

        if (this.rerouteInterval >= 0) {
            autoRerouteRunnable = new Runnable() {
                @Override
                public void run() {
                    try {
                        controller.clearWayfindingPath();
                        Point userPoint = new Point(UserLocation.getInstance().getPosition().x, UserLocation.getInstance().getPosition().y, UserLocation.getInstance().getMapId());
                        wayfindToAssetFromPoint(asset, userPoint);
                    } finally {
                        handler.postDelayed(autoRerouteRunnable, rerouteInterval);
                    }
                }
            };
            autoRerouteRunnable.run();
        }
    }

    public void stopAutoReroute() {
        if (autoRerouteRunnable != null) {
            handler.removeCallbacks(autoRerouteRunnable);
        }
    }

    /**
     * Wayfind and draw a path to an {@link Asset} of given type from {@link Point}.
     *
     * @param type  The type string for wayfinding to the asset.
     * @param point The starting point to wayfind from.
     */
    public Asset wayfindToNearestAssetOfType(String type, Point point) {
        Asset[] assetsArray = this.getAssetsByType(type);
        Asset wayfindToAsset = null;

        if (assetsArray.length > 0) {
            ArrayList<Asset> assetsOnFloor = new ArrayList<Asset>();
            ArrayList<Asset> assetsNotOnFloor = new ArrayList<Asset>();

            //Look for assets on current floor
            for (Asset asset : assetsArray) {
                if (asset.getMapId() == (int) point.z) {
                    //Same floor
                    assetsOnFloor.add(asset);
                } else {
                    assetsNotOnFloor.add(asset);
                }
            }

            //If exists on current floor
            if (assetsOnFloor.size() > 0) {
                //Wayfind to closest one
                if (assetsOnFloor.size() == 1) {
                    this.wayfindToAssetFromPoint(assetsOnFloor.get(0), point);
                    return assetsOnFloor.get(0);
                }

                //More than 1
                float closestDistance = Float.MAX_VALUE;
                for (Asset asset : assetsOnFloor) {
                    PointF pointFrom = new PointF((float) point.x, (float) point.y);
                    float distance = this.distanceBetweenPoints(pointFrom, asset.getPosition());
                    if (distance < closestDistance) {
                        closestDistance = distance;
                        wayfindToAsset = asset;
                    }
                }

                //We have closest asset here
                if (wayfindToAsset != null) {
                    //Draw wayfind path
                    this.wayfindToAssetFromPoint(wayfindToAsset, point);
                }
                return wayfindToAsset;
            } else {
                //Array has 1 asset
                if (assetsNotOnFloor.size() == 1) {
                    this.wayfindToAssetFromPoint(assetsNotOnFloor.get(0), point);
                    return assetsNotOnFloor.get(0);
                }

                //Multiple, calculate distance via wayfind algo
                float closestDistance = Float.MAX_VALUE;
                PathPerFloor[] lowestPathPerFloorArray = null;

                for (Asset asset : assetsNotOnFloor) {
                    Point pointTo = new Point(asset.getPosition().x, asset.getPosition().y, asset.getMapId());

                    Waypoint waypointFrom = controller.getNearestWaypointToPoint(point);
                    Waypoint waypointTo = controller.getNearestWaypointToPoint(pointTo);

                    if (waypointFrom != null && waypointTo != null) {
                        PathPerFloor[] pathPerFloorArray = controller.wayfindBetweenWaypoints(waypointFrom, waypointTo);
                        float totalDistance = 0;
                        for (PathPerFloor ppf : pathPerFloorArray) {
                            totalDistance += this.totalWayfindPathDistance(ppf.points);
                        }

                        if (totalDistance < closestDistance) {
                            closestDistance = totalDistance;
                            lowestPathPerFloorArray = pathPerFloorArray;
                            wayfindToAsset = asset;
                        }
                    }
                }

                //Has lowestPathPerFloorArray here
                if (lowestPathPerFloorArray.length > 0) {
                    controller.drawWayfindingPath(lowestPathPerFloorArray);
                }
                return wayfindToAsset;
            }
        }
        System.out.println("Could not find asset of given type: " + type);
        return wayfindToAsset;
    }

    private float totalWayfindPathDistance(ASNode[] points) {
        float totalDistance = 0;
        //No points
        if (points.length == 0) {
            return totalDistance;
        }

        //Check if it has more than 1 element
        if (points.length > 1) {
            for (int i = 1; i < points.length; i++) {
                PointF prevPoint = new PointF((float) points[i - 1].x, (float) points[i - 1].y);
                PointF currentPoint = new PointF((float) points[i].x, (float) points[i].y);
                totalDistance += this.distanceBetweenPoints(prevPoint, currentPoint);
            }
        }

        return totalDistance;
    }

    private float distanceBetweenPoints(PointF point1, PointF point2) {
        float xDistance = (point2.x - point1.x);
        float yDistance = (point2.y - point1.y);
        return (float) Math.sqrt(Math.pow(xDistance, 2) + Math.pow(yDistance, 2));
    }
}
