package com.instabug.survey.network.util;

import android.content.Context;

import androidx.annotation.VisibleForTesting;

import com.instabug.library.BuildConfig;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.user.UserManagerWrapper;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.survey.common.models.ActionEvent;
import com.instabug.survey.models.Question;
import com.instabug.survey.models.State;
import com.instabug.survey.models.Survey;
import com.instabug.survey.models.SurveyTypes;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

/**
 * @author mesbah
 */

public class SubmittingSurveysUtil {
    public static final String RESPONSE_PARAM = "responses";
    public static final String VALUE_PARAM = "value";
    public static final String QUESTION_ID_PARAM = "question_id";
    public static final String RESPONDED_AT_PARAM = "responded_at";
    public static final String NAME_PARAM = "name";
    public static final String EMAIL_PARAM = "email";
    public static final String PUSH_TOKEN_PARAM = "push_token";
    public static final String SURVEY_EVENTS = "events";
    public static final String SURVEY_CURRENT_LOCALE_PARAM = "locale";
    public static final String SDK_VERSION = "sdk_version";
    public static final String APP_VERSION = "app_version";
    public static final String USER_ATTRIBUTES = "user_attributes";
    public static final String OS = "os";
    public static final String DEVICE = "device";
    public static final String SESSION_ID = "session_id";

    public static void addParamsToSubmittingSurveyRequest(Context context, Request.Builder request, Survey survey) throws JSONException {
        if (!survey.isLastEventDismiss()) {
            JSONArray responsesArray = getResponsesFromQuestions(survey.getQuestions());
            if (responsesArray.length() > 0) {
                request.addParameter(new RequestParameter<>(RESPONSE_PARAM, responsesArray));
            }
        }
        request.addParameter(new RequestParameter<>(RESPONDED_AT_PARAM, survey.getRespondedAt()));
        String userName = InstabugCore.getIdentifiedUsername();
        if (userName != null) {
            request.addParameter(new RequestParameter<>(NAME_PARAM, userName));
        }
        request.addParameter(new RequestParameter<>(EMAIL_PARAM, UserManagerWrapper.getUserEmail()));
        request.addParameter(new RequestParameter<>(SURVEY_EVENTS, getSurveyEvents(survey.getSurveyEvents())));
        if (survey.getLocalization() != null && survey.getLocalization().getCurrentLocale() != null)
            request.addParameter(new RequestParameter<>(SURVEY_CURRENT_LOCALE_PARAM, survey.getLocalization().getCurrentLocale()));
        request.addParameter(new RequestParameter<>(SDK_VERSION, BuildConfig.SDK_VERSION));
        request.addParameter(new RequestParameter<>(APP_VERSION, DeviceStateProvider.getAppVersion(context)));
        request.addParameter(new RequestParameter<>(PUSH_TOKEN_PARAM, InstabugCore.getPushNotificationToken()));

        HashMap<String, String> userAttributes = InstabugCore.retrieveAllSDKAttributes();
        if (userAttributes != null && userAttributes.size() > 0)
            request.addParameter(new RequestParameter<>(USER_ATTRIBUTES, getUserAttributes(userAttributes)));

        request.addParameter(new RequestParameter<>(OS, DeviceStateProvider.getOS()));
        request.addParameter(new RequestParameter<>(DEVICE, DeviceStateProvider.getDevice()));
        if (survey.getSessionID() != null)
            request.addParameter(new RequestParameter<>(SESSION_ID, survey.getSessionID()));
    }

    private static JSONArray getSurveyEvents(ArrayList<ActionEvent> actionEvents) throws
            JSONException {
        JSONArray surveyEventsArray = new JSONArray();
        if (actionEvents == null || actionEvents.isEmpty()) {
            return surveyEventsArray;
        }

        for (ActionEvent event : actionEvents) {
            if (event.isSynced()) continue;
            JSONObject eventObj = new JSONObject();
            eventObj.put(ActionEvent.KEY_EVENT_TYPE, event.getEventType());
            eventObj.put(ActionEvent.KEY_TIMESTAMP, event.getTimestamp());
            eventObj.put(ActionEvent.KEY_INDEX, event.getIndex());
            surveyEventsArray.put(eventObj);
        }
        return surveyEventsArray;
    }

    public static JSONArray getResponsesFromQuestions(ArrayList<Question> questions) throws
            JSONException {
        JSONArray responsesArray = new JSONArray();
        for (Question question : questions) {
            if (question.getAnswer() != null && !question.getAnswer().equals("")) {
                JSONObject questionJsonObject = new JSONObject();
                questionJsonObject.put(VALUE_PARAM, question.getAnswer());
                questionJsonObject.put(QUESTION_ID_PARAM, question.getId());
                responsesArray.put(questionJsonObject);
            }
        }
        return responsesArray;
    }


    private static JSONObject getUserAttributes(HashMap<String, String> userAttributes) throws
            JSONException {
        JSONObject userAttributesJson = new JSONObject();
        for (Map.Entry<String, String> entry : userAttributes.entrySet()) {
            userAttributesJson.put(entry.getKey(), entry.getValue());
        }
        return userAttributesJson;
    }

    public static JSONObject getSurveyAsResponse(Survey survey, @State String surveyState) throws JSONException {
        JSONObject surveyJson = new JSONObject();
        surveyJson.put("id", survey.getId());
        surveyJson.put("type", survey.getTypeAsString());
        surveyJson.put("title", survey.getTitle());
        surveyJson.put("is_announcement", false);
        surveyJson.put("responses", getQuestionsAnswersAsJson(survey.getQuestions(), survey.getType(), surveyState));
        return surveyJson;
    }

    @VisibleForTesting
    public static JSONArray getQuestionsAnswersAsJson(ArrayList<Question> questions, @SurveyTypes int surveyType, @State String surveyState) throws JSONException {
        JSONArray questionsJson = new JSONArray();
        if (surveyType == SurveyTypes.CUSTOM || surveyType == SurveyTypes.NPS) {
            if (surveyState.equals(State.DISMISSED)) {
                // Return an empty response if the custom survey was dismissed or ended
                return questionsJson;
            }
        }
        for (Question question : questions) {
            if (question.getAnswer() != null && !question.getAnswer().equals("")) {
                JSONObject questionJson = new JSONObject();
                questionJson.put("question_id", question.getId());
                questionJson.put("question_title", question.getTitle());
                questionJson.put("question_type", question.getTypeAsString());
                questionJson.put("response_timestamp", question.getAnsweredAt());
                questionJson.put("response_value", question.getAnswer());
                questionsJson.put(questionJson);
            }
        }
        return questionsJson;
    }

}