package com.instabug.survey.announcements.ui.fragment.whatsnew;

import android.content.Context;
import android.os.Build;
import android.os.Bundle;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewTreeObserver;
import android.widget.Button;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.ui.BackPressHandler;
import com.instabug.survey.R;
import com.instabug.survey.announcements.models.AnnouncementItem;
import com.instabug.survey.announcements.ui.activity.AnnouncementActivity;
import com.instabug.survey.announcements.ui.custom.DynamicRelativeLayout;
import com.instabug.survey.announcements.ui.fragment.AnnouncementItemFragment;

import org.jetbrains.annotations.NotNull;

public class WhatsNewFragment extends AnnouncementItemFragment<WhatsNewPresenter> implements
        View.OnTouchListener, View.OnClickListener, WhatsNewContract.View, BackPressHandler {
    @VisibleForTesting
    @Nullable
    protected WhatsNewFeaturesAdapter whatsNewFeaturesAdapter;
    @VisibleForTesting
    @Nullable
    protected RecyclerView newFeaturesRecyclerView;
    @Nullable
    private Button submitButton;
    @Nullable
    private TextView title;
    @Nullable
    private WhatsNewPresenter presenter;
    @Nullable
    private AnnouncementActivity activityCallback;

    public static WhatsNewFragment newInstance(AnnouncementItem announcementItem) {
        Bundle args = new Bundle();
        args.putSerializable(KEY_ANNOUNCE_ARGUMENT, announcementItem);
        WhatsNewFragment fragment = new WhatsNewFragment();
        fragment.setArguments(args);
        return fragment;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setRetainInstance(true);
    }

    @Override
    protected void initViews(View rootView, @Nullable Bundle savedInstanceState) {
        super.initViews(rootView, savedInstanceState);

        title = rootView.findViewById(R.id.instabug_announcement_title);
        newFeaturesRecyclerView = rootView.findViewById(R.id.instabug_announcement_features_grid_view);
        submitButton = rootView.findViewById(R.id.instabug_btn_submit);
        container = rootView.findViewById(R.id.instabug_announcement_dialog_container);
        if (container instanceof DynamicRelativeLayout) {
            container.setOnTouchListener(this);
            container.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
                @Override
                public void onGlobalLayout() {
                    if (container == null || submitButton == null || title == null || newFeaturesRecyclerView == null) {
                        return;
                    }
                    if (((DynamicRelativeLayout) container).isMaxHeightReached()) {
                        // Change the layout config to match the screen
                        // 1. Submit button (Align it to the bottom)
                        RelativeLayout.LayoutParams submitButtonLayoutParams = ((RelativeLayout.LayoutParams) submitButton.getLayoutParams());
                        submitButtonLayoutParams.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
                        // To avoid its cyclic reference between it and the recycler view
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                            submitButtonLayoutParams.removeRule(RelativeLayout.BELOW);
                        } else {
                            // A workaround for removing the rule below API 17
                            submitButtonLayoutParams.addRule(RelativeLayout.BELOW, 0);
                        }
                        submitButton.setLayoutParams(submitButtonLayoutParams);

                        // 2. Title textView (Align it to the top)
                        RelativeLayout.LayoutParams titleTextViewLayoutParams = ((RelativeLayout.LayoutParams) title.getLayoutParams());
                        titleTextViewLayoutParams.addRule(RelativeLayout.ALIGN_PARENT_TOP);
                        title.setLayoutParams(titleTextViewLayoutParams);

                        // 3. Announcement Items RV (Make it above the submit button)
                        RelativeLayout.LayoutParams itemsRecyclerViewLayoutParams = ((RelativeLayout.LayoutParams) newFeaturesRecyclerView.getLayoutParams());
                        itemsRecyclerViewLayoutParams.addRule(RelativeLayout.ABOVE, R.id.instabug_btn_submit);
                        newFeaturesRecyclerView.setLayoutParams(itemsRecyclerViewLayoutParams);
                    }

                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                        container.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                    } else {
                        container.getViewTreeObserver().removeGlobalOnLayoutListener(this);
                    }
                }
            });
        }

        if (getArguments() != null) {
            announcementItem = (AnnouncementItem) getArguments().getSerializable(KEY_ANNOUNCE_ARGUMENT);
        }
        presenter = new WhatsNewPresenter(this);
        if (announcementItem != null) {
            presenter.showWhatsNewAnnouncement(announcementItem);
        }
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);
        try {
            activityCallback = (AnnouncementActivity) context;
        } catch (ClassCastException e) {
            throw new ClassCastException(context.toString()
                    + " must implement AnnouncementActivity");
        }
    }

    @Override
    public void onDetach() {
        activityCallback = null;
        super.onDetach();
    }

    @Override
    public void onViewCreated(@NotNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        view.setFocusableInTouchMode(true);
    }

    @Override
    protected int getLayout() {
        return R.layout.instabug_dialog_whats_new_announce;
    }

    @Override
    public boolean onTouch(View view, MotionEvent event) {
        if (presenter != null) {
            presenter.registerGestureHandler(view, event);
        }
        return true;
    }

    @Override
    public void onClick(View v) {
        int id = v.getId();
        if (id == R.id.instabug_btn_submit) {
            submitAnnouncement();
        }
    }

    @Override
    public void dismissAnnouncement() {
        if (activityCallback == null || announcement == null) return;
        activityCallback.dismissAnnouncement(announcement);
    }

    @Override
    public void submitAnnouncement() {
        if (announcement == null || announcement.getAnnouncementItems() == null) return;

        for (AnnouncementItem item : announcement.getAnnouncementItems()) {
            if (item.getOptions() != null)
                item.setAnswer(item.getOptions().get(0));
        }
        if (activityCallback == null) return;

        activityCallback.submitAnnouncement(announcement);
    }

    @Override
    public void showWhatsNewAnnouncement(AnnouncementItem announcementItem) {
        if (getActivity() == null) return;

        whatsNewFeaturesAdapter = new WhatsNewFeaturesAdapter(getActivity(), announcementItem);
        RecyclerView recyclerView = newFeaturesRecyclerView;
        if (recyclerView != null) {
            recyclerView.setLayoutManager(new LinearLayoutManager(getActivity()));
            recyclerView.setAdapter(whatsNewFeaturesAdapter);
        }

        if (title != null) {
            title.setText(announcementItem.getTitle() != null ? announcementItem.getTitle() : "");
            title.setTextColor(InstabugCore.getPrimaryColor());
        }

        if (submitButton != null
                && announcementItem.getOptions() != null
                && announcementItem.getOptions().size() > 0) {
            String actionText = announcementItem.getOptions().get(0);
            submitButton.setAllCaps(false);
            submitButton.setText(actionText);
            submitButton.setContentDescription(actionText);
            submitButton.setBackgroundColor(InstabugCore.getPrimaryColor());
            submitButton.setOnClickListener(this);
        }
    }

    @Override
    public boolean onBackPress() {
        return true;
    }
}
