package com.instabug.survey;

import android.annotation.SuppressLint;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.OnUsageExceededReady;
import com.instabug.library.apichecker.APIChecker;
import com.instabug.library.apichecker.ReturnableRunnable;
import com.instabug.library.apichecker.VoidRunnable;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.survey.callbacks.OnDismissCallback;
import com.instabug.survey.callbacks.OnFinishCallback;
import com.instabug.survey.callbacks.OnShowCallback;
import com.instabug.survey.di.ServiceLocator;
import com.instabug.survey.settings.SurveysSettings;
import com.instabug.survey.utils.SurveysUtils;

import java.util.List;

/**
 * Created by tarek on 3/25/17.
 */
public class Surveys {

    /**
     * Show any valid survey if exist
     *
     * @return return true if a valid survey was shown otherwise false
     * @since 8.0.0
     */
    @WorkerThread
    public static boolean showSurveyIfAvailable() {
        Boolean isAvailable = APIChecker.checkAndGet("Surveys.showSurveyIfAvailable", new ReturnableRunnable<Boolean>() {
            @Nullable
            @Override
            public Boolean run() {
                InstabugSDKLogger.d(Constants.LOG_TAG, "showSurveyIfAvailable");
                return SurveysManager.getInstance() != null && SurveysManager.getInstance().showValidSurvey();
            }
        }, false);
        return isAvailable != null && isAvailable;
    }

    /**
     * Get available surveys
     *
     * @return an array of Survey objects
     * Objects has a title property and a show() method
     * @since 8.0.0
     */
    @WorkerThread
    @Nullable
    public static List<com.instabug.survey.Survey> getAvailableSurveys() {
        return APIChecker.checkAndGet("Surveys.getAvailableSurveys", new ReturnableRunnable<List<com.instabug.survey.Survey>>() {
            @Override
            @Nullable
            public List<com.instabug.survey.Survey> run() {
                InstabugSDKLogger.d(Constants.LOG_TAG, "getAvailableSurveys");
                if (SurveysManager.getInstance() != null &&
                        SurveysUtils.isSurveysFeatureEnabled() &&
                        ServiceLocator.getConfigurationsProvider().isSurveysAvailableAndUsageNotExceeded()) {
                    return SurveysManager.getInstance().getAvailableSurveys();
                }
                return null;
            }
        }, null);
    }

    /**
     * Set Surveys auto-showing state, default state auto-showing enabled
     *
     * @param isAutoShowingEnabled whether Surveys should be auto-showing or not
     * @since 8.0.0
     */
    public static void setAutoShowingEnabled(final boolean isAutoShowingEnabled) {
        APIChecker.checkAndRunInExecutor("Surveys.setAutoShowingEnabled", new VoidRunnable() {
            @Override
            public void run() {
                InstabugSDKLogger.d(Constants.LOG_TAG, "setAutoShowingEnabled:" + isAutoShowingEnabled);
                SurveysSettings.setSurveysAutoShowing(isAutoShowingEnabled);
            }
        });
    }

    /**
     * Sets the callback that gets executed just before showing any valid survey<br/>
     * WARNING: This runs on your application's main UI thread. Please do not include
     * any blocking operations to avoid ANRs.
     *
     * @param onShowCallback to run on the UI thread before showing any valid survey
     * @since 8.0.0
     */
    public static void setOnShowCallback(final OnShowCallback onShowCallback) {
        APIChecker.checkAndRunInExecutorThenPostOnMain("Surveys.setOnShowCallback", new VoidRunnable() {
            @Override
            public void run() {
                SurveysSettings.setOnShowCallback(onShowCallback);
            }
        });
    }

    /**
     * Sets the runnable that gets executed just after showing any valid survey<br/>
     * WARNING: This runs on your application's main UI thread. Please do not include
     * any blocking operations to avoid ANRs.
     *
     * @param onDismissCallback to run on the UI thread after dismissing any survey
     * @since 8.0.0
     */
    public static void setOnDismissCallback(final OnDismissCallback onDismissCallback) {
        APIChecker.checkAndRunInExecutorThenPostOnMain("Surveys.setOnDismissCallback", new VoidRunnable() {
            @Override
            public void run() {
                SurveysSettings.setOnDismissCallback(onDismissCallback);
            }
        });
    }

    /**
     * Show specific survey using its token
     *
     * @param token token for the survey to show
     * @return returns true if there is a survey with this token
     */
    @WorkerThread
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public static boolean showSurvey(@NonNull final String token) {
        return APIChecker.checkAndGet("Surveys.showSurvey", new ReturnableRunnable<Boolean>() {
            @Override
            public Boolean run() {
                if (token == null || token.equals("null")) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Survey token was null");
                    return false;
                } else {
                    return SurveysManager.getInstance() != null && SurveysManager.getInstance().showSurvey(token);
                }
            }
        }, false);
    }

    /**
     * A cross platform version of #showSurvey(String) to offload it's work on a background
     * thread.
     *
     * @param token token for the survey to show
     * @see #showSurvey(String)
     */
    private static void showSurveyCP(@NonNull final String token) {
        APIChecker.checkAndRunInExecutor("Surveys.showSurveyCP", () -> showSurvey(token));
    }

    /**
     * returns weather the user has respond to this survey or not
     *
     * @param token token for the survey
     * @return true if the user has respond/answered the survey
     */
    @WorkerThread
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public static boolean hasRespondToSurvey(@NonNull final String token) {
        return APIChecker.checkAndGet("Surveys.hasRespondToSurvey", new ReturnableRunnable<Boolean>() {
            @Override
            public Boolean run() {
                if (token == null || token.equals("null")) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Survey token was null");
                    return false;
                } else {
                    return SurveysManager.getInstance() != null && SurveysManager.getInstance().hasRespondToSurvey(token);
                }
            }
        }, false);
    }

    /**
     * Set Surveys welcome screen enabled, default value is false
     *
     * @param shouldShow whether should show welcome screen Surveys before surveys or not
     * @since 4.14.0
     */
    public static void setShouldShowWelcomeScreen(final boolean shouldShow) {
        APIChecker.checkAndRunInExecutor("Surveys.setShouldShowWelcomeScreen", new VoidRunnable() {
            @Override
            public void run() {
                SurveysSettings.setShouldShowSurveysWelcomeScreen(shouldShow);
            }
        });
    }

    /**
     * Enable/disable surveys.
     *
     * @param state desired state of surveys feature.
     * @see com.instabug.library.Feature.State
     * @since 3.3.1
     */
    public static void setState(@NonNull final Feature.State state) {
        APIChecker.checkAndRunInExecutor("Surveys.setState", new VoidRunnable() {
            @Override
            public void run() {
                InstabugCore.setFeatureState(IBGFeature.SURVEYS, state);
                InstabugCore.setFeatureState(IBGFeature.ANNOUNCEMENTS, state);
            }
        });
    }

    /**
     * Private api to be called by Verizon to apply app rating customized design
     */
    private static void enableCustomization() {
        APIChecker.checkAndRunInExecutor("Surveys.enableCustomization", new VoidRunnable() {
            @Override
            public void run() throws Exception {
                SurveysSettings.enableCustomization();
            }
        });
    }

    /**
     * Used to override the play-store rating question validation to show or not in NPS Surveys
     * <p>
     * This API Can be used before building Instabug
     *
     * @param isAppStoreRatingEnabled
     * @since 9.1.5
     */
    public static void setIsAppStoreRatingEnabled(final boolean isAppStoreRatingEnabled) {
        APIChecker.checkAndRunInExecutor("Surveys.setIsAppStoreRatingEnabled", new VoidRunnable() {
            @Override
            public void run() {
                SurveysSettings.setIsAppStoreRatingEnabled(isAppStoreRatingEnabled);
            }
        });
    }

    /**
     * Sets the runnable that gets executed just after finishing (Dismissing or Submitting) any valid survey
     *
     * @param onFinishCallback to run after finishing any survey
     */
    public static void setOnFinishCallback(final OnFinishCallback onFinishCallback)
            throws IllegalStateException {
        APIChecker.checkAndRunInExecutor("Surveys.setOnFinishCallback", new VoidRunnable() {
            @Override
            public void run() throws Exception {
                SurveysSettings.setOnFinishCallback(onFinishCallback);
            }
        });
    }

    /**
     * @param onUsageExceededReady will be called with
     *                             true if Surveys has exceeded the usage limit on your plan.
     *                             Otherwise,  will be called with false
     */
    public static void getUsageExceeded(@NonNull OnUsageExceededReady onUsageExceededReady) {
        try {
            APIChecker.checkAndRunOrThrow("Surveys.getUsageExceeded", () -> {
                final boolean isUsageExceeded = ServiceLocator
                        .getConfigurationsProvider()
                        .isSurveysUsageExceeded();
                PoolProvider.postMainThreadTask(() -> {
                    if (onUsageExceededReady != null) {
                        onUsageExceededReady.onReady(isUsageExceeded);
                    }
                });

            });

        } catch (Exception exception) {
            if (onUsageExceededReady != null) {
                onUsageExceededReady.onReady(false);
            }
        }
    }
}
