package com.instabug.survey.utils;


import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.BuildFieldsProvider;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.ObjectMapper;
import com.instabug.library.util.StringUtility;
import com.instabug.library.util.TimeUtils;
import com.instabug.survey.Constants;
import com.instabug.survey.cache.SurveysCacheManager;
import com.instabug.survey.cache.SurveysDbHelper;
import com.instabug.survey.common.models.Condition;
import com.instabug.survey.models.CountryInfo;
import com.instabug.survey.models.Survey;
import com.instabug.survey.models.SurveyTypes;
import com.instabug.survey.settings.PerSessionSettings;
import com.instabug.survey.settings.SurveysSettings;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


/**
 * @author mesbah
 */
public class SurveysValidator {

    private Callback callback;
    @Deprecated
    private String appVersionName; // eg. 1.0.0-beta
    private String appVersionAndCodeName; // eg. 1.0.0-beta (3)

    public SurveysValidator(Callback callback, String versionName, String versionAndCodeName) {
        this.callback = callback;
        this.appVersionName = versionName;
        this.appVersionAndCodeName = versionAndCodeName;
    }

    public void showSurveysByEventTriggerIfAvailable(String triggerEvent) throws ParseException {
        log("showSurveysByEventTriggerIfAvailable(triggerEvent: " + triggerEvent + ")");

        List<Survey> eventTriggeredSurveys = SurveysDbHelper.retrieveByTriggerEvent(triggerEvent);
        log("eventTriggeredSurveys: " + eventTriggeredSurveys.size());

        if (eventTriggeredSurveys.size() > 0) {
            List<Survey> validSurveys = getAllValidSurveys(eventTriggeredSurveys);
            log("validSurveys: " + validSurveys.size());

            Survey survey = validSurveys.size() > 0 ? validSurveys.get(0) : null;
            if (survey == null) {
                log("no valid surveys for the event" + triggerEvent + ". Returning null");
            } else {
                log("Survey with id:{ " + survey.getId() + "} is first valid survey for the event" + triggerEvent);
                callback.onUserEventTrigger(survey);
            }
        }
    }

    public void showSurveysByTimeTriggerIfAvailable() throws ParseException {
        log("showSurveysByTimeTriggerIfAvailable()");

        List<Survey> timeTriggeredSurveys = SurveysCacheManager.getTimeTriggeredSurveys();
        log("timeTriggeredSurveys: " + timeTriggeredSurveys.size());

        List<Survey> validSurveys = getAllValidSurveys(timeTriggeredSurveys);
        log("validSurveys: " + validSurveys.size());

        Survey survey = validSurveys.size() > 0 ? validSurveys.get(0) : null;
        if (survey == null) {
            log("no valid time-triggered surveys. Returning null...");
        } else {
            log("Survey with id:{ " + survey.getId() + "}  is first valid survey for time-triggered surveys");
            callback.onTimeEventTrigger(survey);
        }
    }

    @NonNull
    private List<Survey> getAllValidSurveys(List<Survey> availableSurveys) throws ParseException {
        // Do the basic validation
        int size = availableSurveys == null ? 0 : availableSurveys.size();
        log("getAllValidSurveys(availableSurveys: " + size + ")");

        List<Survey> validSurveys = new ArrayList<>();
        if (availableSurveys != null) {
            for (Survey survey : availableSurveys) {
                if (!survey.isOptInSurvey() && !survey.isPaused() && checkSurveyValidity(survey)) {
                    if (new FrequencyProcessor().isFrequentSurvey(survey)) {
                        validSurveys.add(survey);
                    }
                }
            }
            log("validSurveys: " + validSurveys.size());
            // Sort to get the most earlier survey
            Collections.sort(availableSurveys, new Comparator<Survey>() {
                @Override
                public int compare(Survey o1, Survey o2) {
                    return Long.compare(o1.getDismissedAt(), o2.getDismissedAt());
                }
            });
        }
        return validSurveys;
    }

    public boolean hasValidSurveys() throws ParseException {
        boolean hasValidSurveys = getFirstValidSurvey() != null;
        log("hasValidSurveys() ? " + hasValidSurveys);
        return hasValidSurveys;
    }

    @Nullable
    public Survey getFirstValidSurvey() throws ParseException {
        log("getFirstValidSurvey()");

        List<Survey> timeTriggeredSurveys = SurveysCacheManager.getTimeTriggeredSurveys();
        log("timeTriggeredSurveys: " + timeTriggeredSurveys.size());

        List<Survey> validSurveys = getAllValidSurveys(timeTriggeredSurveys);
        log("timeTriggeredSurveys: " + timeTriggeredSurveys.size());

        Survey survey = validSurveys.size() > 0 ? validSurveys.get(0) : null;
        if (survey == null) {
            log("no valid surveys. Returning null...");
        } else {
            log("Survey with id:{ " + survey.getId() + "}  is first valid survey");
        }
        return survey;
    }

    @VisibleForTesting
    boolean checkSurveyValidity(Survey survey) throws ParseException {
        log("checkSurveyValidity(survey: " + survey + ")");

        // avoid showing store rating survey if isLive request was not fired
        if (survey.getType() == SurveyTypes.STORE_RATING) {
            if (!survey.isGooglePlayAppRating() && !PerSessionSettings.getInstance().isLiveAppRequested())
                return false;
        }

        // avoid showing NPS survey if isLive request was not fired
        if (survey.getType() == SurveyTypes.NPS) {
            if (!PerSessionSettings.getInstance().isLiveAppRequested()) return false;
        }

        boolean primitiveTypesValidity = checkPrimitiveTypes(survey.getTargetAudiences(),
                survey.getConditionsOperator(), survey.getDismissedAt());
        log("primitiveTypesValidity: " + primitiveTypesValidity);

        boolean customAttributesValidity
                = ValidationUtils.checkCustomAttributesConditions(survey.getCustomAttributes(), survey
                .getConditionsOperator());
        log("customAttributesValidity: " + customAttributesValidity);

        boolean userEventsValidity = ValidationUtils.checkUserEvents(survey.getUserEvents(), survey
                .getConditionsOperator());
        log("userEventsValidity: " + userEventsValidity);

        if ((survey.getUserEvents() != null && survey.getUserEvents().size() > 0)
                || survey.getCustomAttributes().size() > 0
                || survey.getTargetAudiences().size() > 0) {
            if (TargetingOperator.OR.equals(survey.getConditionsOperator())) {
                return primitiveTypesValidity || customAttributesValidity || userEventsValidity;
            }
            return primitiveTypesValidity && customAttributesValidity && userEventsValidity;
        } else {
            return true;
        }
    }

    @VisibleForTesting
    boolean checkPrimitiveTypes(ArrayList<Condition> primitiveTypesConditions,
                                String conditionsOperator, long dismissedAt) throws ParseException {
        int size = primitiveTypesConditions == null ? 0 : primitiveTypesConditions.size();
        log("checkPrimitiveTypes(primitiveTypesConditions: " + size + ", "
                + "conditionsOperator: " + conditionsOperator + ", dismissedAt: " + dismissedAt + ")");

        boolean conditionsResult = conditionsOperator.equals(TargetingOperator.AND);
        for (int i = 0; i < size; i++) {
            boolean primitiveTypesCondition = checkPrimitiveType(primitiveTypesConditions.get(i),
                    dismissedAt);
            if (i == 0) {
                conditionsResult = primitiveTypesCondition;
            } else {
                if (TargetingOperator.OR.equals(conditionsOperator)) {
                    conditionsResult |= primitiveTypesCondition;
                } else {
                    conditionsResult &= primitiveTypesCondition;
                }
            }
        }
        return conditionsResult;
    }

    @VisibleForTesting
    boolean checkPrimitiveType(Condition primitiveTypeCondition, long dismissedAt) throws
            ParseException {
        log("checkPrimitiveType(primitiveTypeCondition: " + primitiveTypeCondition + ", "
                + "dismissedAt: " + dismissedAt + ")");
        if (primitiveTypeCondition.getKey() == null) {
            return false;
        }
        switch (primitiveTypeCondition.getKey()) {
            case TargetingCondition.APP_VERSION:
                return validateAppVersion(primitiveTypeCondition);
            case TargetingCondition.APP_VERSION_V2:
                return validateAppVersionV2(primitiveTypeCondition);
            case TargetingCondition.DATE:
                return validateCurrentDate(primitiveTypeCondition);
            case TargetingCondition.USER_EMAIL:
                return validateUserEmail(primitiveTypeCondition);
            case TargetingCondition.SESSION_COUNT:
                return validateSessionCount(primitiveTypeCondition, SettingsManager.getInstance()
                        .getSessionsCount());
            case TargetingCondition.DAYS_SINCE_SIGN_UP:
                return validateDaysSinceSignup(primitiveTypeCondition);
            case TargetingCondition.DAYS_SINCE_DISMISS:
                return validateDaysSinceDismiss(primitiveTypeCondition, dismissedAt);
            case TargetingCondition.COUNTRY:
                String countryInfoJson = SurveysSettings.getCountryInfo();
                CountryInfo countryInfo = ObjectMapper.fromJson(countryInfoJson, CountryInfo.class);
                return validateCountryCode(primitiveTypeCondition, countryInfo);
            case TargetingCondition.LAST_SEEN:
                return checkLastSeenTimestamp(primitiveTypeCondition);
            case TargetingCondition.OS_API_LEVEL:
                return checkOSApiLevel(primitiveTypeCondition);
            default:
                return false;
        }
    }

    public boolean checkOSApiLevel(Condition primitiveTypeCondition) {
        return validateOSApiLevel(primitiveTypeCondition);
    }

    private boolean validateOSApiLevel(Condition condition) {
        int currentAndroidSDKLevel = BuildFieldsProvider.INSTANCE.provideBuildVersion();
        log("validateOSApiLevel(condition: " + condition + ")");
        log("Build.VERSION.SDK_INT: " + currentAndroidSDKLevel);
        try {
            if (condition.getValue() != null
                    && condition.getOperator() != null) {
                int targetOSApiLevel = Integer.parseInt(condition.getValue());
                switch (condition.getOperator()) {
                    case TargetingOperator.EQUAL:
                        return currentAndroidSDKLevel == targetOSApiLevel;
                    case TargetingOperator.NOT_EQUAL:
                        return currentAndroidSDKLevel != targetOSApiLevel;
                    case TargetingOperator.GREATER_THAN:
                        return currentAndroidSDKLevel > targetOSApiLevel;
                    case TargetingOperator.LESS_THAN:
                        return currentAndroidSDKLevel < targetOSApiLevel;
                    default:
                        return false;
                }
            }
        } catch (NumberFormatException exception) {
            return ValidationUtils.onParseException(condition, exception);
        }
        return false;
    }

    @VisibleForTesting
    public boolean validateCountryCode(Condition primitiveTypeCondition, @Nullable CountryInfo countryInfo) {
        log("validateOSApiLevel(primitiveTypeCondition: " + primitiveTypeCondition + ", "
                + "countryInfo: " + countryInfo + ")");
        if (countryInfo != null && primitiveTypeCondition != null) {
            String countyCode = countryInfo.getCountryCode();
            if (TargetingOperator.EQUAL.equals(primitiveTypeCondition.getOperator())
                    && primitiveTypeCondition.getValue() != null) {
                return primitiveTypeCondition.getValue().equalsIgnoreCase(countyCode);
            }
            return false;
        }
        return false;

    }

    @VisibleForTesting
    @Deprecated
    boolean validateAppVersion(Condition condition) {
        log("validateAppVersion(condition: " + condition + ")");
        String operator = condition.getOperator();
        if (operator == null) {
            return false;
        }
        // If operator is "equals" or "not_equal" then delegate the validation to string validator
        if (!operator.equals(TargetingOperator.EQUAL) && !operator.equals(TargetingOperator.NOT_EQUAL)) {
            String conditionVersionName = extractVersion(condition.getValue());
            String currentVersionName = extractVersion(appVersionName);
            log("appVersionName: " + appVersionName + ", currentVersionName: " + currentVersionName);
            if (conditionVersionName != null && currentVersionName != null) {
                try {
                    int result = StringUtility.compareVersion(currentVersionName, conditionVersionName);
                    switch (operator) {
                        case TargetingOperator.EQUAL:
                            return result == 0;
                        case TargetingOperator.NOT_EQUAL:
                            return result != 0;
                        case TargetingOperator.GREATER_THAN:
                            return result == 1;
                        case TargetingOperator.LESS_THAN:
                            return result == -1;
                        default:
                            return false;
                    }
                } catch (NumberFormatException e) {
                    return false;
                }
            }
        }
        return ValidationUtils.checkStringCondition(condition, appVersionName);
    }

    @Nullable
    private String extractVersion(@Nullable String s) {
        if (s == null) {
            return null;
        }
        String regex = "\\d+(\\.\\d+)*";
        final Pattern pattern = Pattern.compile(regex);
        final Matcher matcher = pattern.matcher(s);
        if (matcher.find()) {
            return matcher.group(0);
        }
        return null;
    }

    boolean validateAppVersionV2(Condition condition) {
        log("validateAppVersion(condition: " + condition + ")");
        String operator = condition.getOperator();
        // If operator is "equals" or "not_equal" then delegate the validation to string validator
        if (operator == null || condition.getValue() == null) {
            return false;
        }
        if (operator.equals(TargetingOperator.GREATER_THAN) || operator.equals(TargetingOperator.LESS_THAN)) {
            try {
                long conditionFirstSeen = Long.parseLong(condition.getValue());
                long currentFirstSeen = SurveysSettings.getFirstSeen();
                if (conditionFirstSeen == -1 || currentFirstSeen == -1) return false;
                switch (operator) {
                    case TargetingOperator.GREATER_THAN:
                        return currentFirstSeen > conditionFirstSeen;
                    case TargetingOperator.LESS_THAN:
                        return currentFirstSeen < conditionFirstSeen;
                    default:
                        return false;
                }
            } catch (NumberFormatException e) {
                return false;
            }
        }
        return ValidationUtils.checkStringCondition(condition, appVersionAndCodeName);
    }

    @VisibleForTesting
    boolean validateCurrentDate(Condition condition) {
        return checkDateCondition(condition, new Date());
    }

    @VisibleForTesting
    boolean checkDateCondition(Condition condition, Date actualDate) {
        log("checkDateCondition(condition: " + condition + ", actualDate: " + actualDate + ")");
        if (condition.getValue() == null || actualDate == null || condition.getOperator() == null) {
            return false;
        }

        Date date = InstabugDateFormatter.getDate(condition.getValue());
        if (date != null) {
            Date conditionDate = InstabugDateFormatter.getStandardizedDate(date);
            actualDate = InstabugDateFormatter.getStandardizedDate(actualDate);
            switch (condition.getOperator()) {
                case TargetingOperator.EQUAL:
                    return actualDate.getDate() == conditionDate.getDate();
                case TargetingOperator.NOT_EQUAL:
                    return actualDate.getDate() != conditionDate.getDate();
                case TargetingOperator.GREATER_THAN:
                    return actualDate.after(conditionDate);
                case TargetingOperator.LESS_THAN:
                    return actualDate.before(conditionDate);
                default:
                    return false;
            }
        }
        return false;
    }

    @VisibleForTesting
    boolean validateUserEmail(Condition condition) {
        String userEmail = InstabugCore.getIdentifiedUserEmail();
        return ValidationUtils.checkStringCondition(condition, userEmail);
    }

    @VisibleForTesting
    boolean validateDaysSinceDismiss(Condition condition, Long lastDismissDate) {
        return checkDaysSinceDismissCondition(condition, lastDismissDate);
    }

    private boolean checkDaysSinceDismissCondition(Condition condition, long lastDismissDate) {
        log("checkDaysSinceDismissCondition(condition: " + condition + ", lastDismissDate: " + lastDismissDate + ")");
        if (condition.getValue() == null || condition.getOperator() == null) {
            return false;
        }
        if (lastDismissDate == 0L) {
            return true;
        }
        try {
            int conditionalDaysCount = Integer.parseInt(condition.getValue());
            int actualDifferenceInDays = DateUtils.getDifferenceInDays(lastDismissDate);

            switch (condition.getOperator()) {
                case TargetingOperator.EQUAL:
                    return actualDifferenceInDays == conditionalDaysCount;
                case TargetingOperator.NOT_EQUAL:
                    return actualDifferenceInDays != conditionalDaysCount;
                case TargetingOperator.GREATER_THAN:
                    return actualDifferenceInDays > conditionalDaysCount;
                case TargetingOperator.LESS_THAN:
                    return actualDifferenceInDays < conditionalDaysCount;
                default:
                    return false;
            }
        } catch (NumberFormatException exception) {
            return ValidationUtils.onParseException(condition, exception);
        }
    }

    @VisibleForTesting
    public boolean validateSessionCount(Condition condition, int actualSessionCount) {
        return checkSessionCountCondition(condition, actualSessionCount);
    }

    private boolean checkSessionCountCondition(Condition condition, int actualSessionCount) {
        log("checkDaysSinceDismissCondition(condition: " + condition
                + ", actualSessionCount: " + actualSessionCount + ")");
        if (condition.getValue() == null || condition.getOperator() == null) {
            return false;
        }
        try {
            int conditionalSessionCount = Integer.parseInt(condition.getValue());

            switch (condition.getOperator()) {
                case TargetingOperator.EQUAL:
                    return actualSessionCount == conditionalSessionCount;
                case TargetingOperator.NOT_EQUAL:
                    return actualSessionCount != conditionalSessionCount;
                case TargetingOperator.GREATER_THAN:
                    return actualSessionCount > conditionalSessionCount;
                case TargetingOperator.LESS_THAN:
                    return actualSessionCount < conditionalSessionCount;
                default:
                    return false;
            }
        } catch (NumberFormatException exception) {
            return ValidationUtils.onParseException(condition, exception);
        }
    }

    private boolean validateDaysSinceSignup(Condition condition) {
        return checkDaysSinceSignUpCondition(condition);
    }

    private boolean checkDaysSinceSignUpCondition(Condition condition) {
        log("checkDaysSinceSignUpCondition(condition: " + condition + ")");
        if (condition.getValue() == null || condition.getOperator() == null) {
            return false;
        }
        try {
            int conditionalSignupDaysCount = Integer.parseInt(condition.getValue());

            int actualDifferenceInDays = DateUtils.getDifferenceInDays(InstabugCore.getFirstRunAt());
            log("actualDifferenceInDays: " + actualDifferenceInDays);

            switch (condition.getOperator()) {
                case TargetingOperator.EQUAL:
                    return actualDifferenceInDays == conditionalSignupDaysCount;
                case TargetingOperator.NOT_EQUAL:
                    return actualDifferenceInDays != conditionalSignupDaysCount;
                case TargetingOperator.GREATER_THAN:
                    return actualDifferenceInDays > conditionalSignupDaysCount;
                case TargetingOperator.LESS_THAN:
                    return actualDifferenceInDays < conditionalSignupDaysCount;
                default:
                    return false;
            }
        } catch (NumberFormatException exception) {
            return ValidationUtils.onParseException(condition, exception);
        }
    }

    public boolean checkLastSeenTimestamp(Condition condition) {
        if (condition.getValue() == null || condition.getOperator() == null) {
            return false;
        }
        long dayDiff = Long.parseLong(condition.getValue());
        long daysSinceLastSeen = getDifferenceDays(getLastSeenTimestamp(), TimeUtils.currentTimeMillis());
        log("checkUserEvent(condition: " + condition + ", daysSinceLastSeen: " + daysSinceLastSeen + ")");
        switch (condition.getOperator()) {
            case TargetingOperator.EQUAL:
                return daysSinceLastSeen == dayDiff;
            case TargetingOperator.NOT_EQUAL:
                return daysSinceLastSeen != dayDiff;
            case TargetingOperator.GREATER_THAN:
                return daysSinceLastSeen > dayDiff;
            case TargetingOperator.LESS_THAN:
                return daysSinceLastSeen < dayDiff;
            default:
                return false;
        }
    }

    private int getDifferenceDays(long time, long currentTime) {
        return (int) TimeUnit.DAYS.convert(currentTime - time, TimeUnit.MILLISECONDS);
    }

    @VisibleForTesting
    public long getLastSeenTimestamp() {
        return InstabugCore.getLastSeenTimestamp();
    }

    public List<com.instabug.survey.Survey> getValidSurveys() {
        List<Survey> timeTriggeredSurveys = SurveysCacheManager.getTimeTriggeredSurveys();
        List<Survey> internalValidSurveys = null;
        try {
            internalValidSurveys = getAllValidSurveys(timeTriggeredSurveys);
        } catch (ParseException e) {
            if (e.getMessage() != null) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error:" + e.getMessage() + "while getting valid surveys", e);
            }
        }

        LinkedList<com.instabug.survey.Survey> validSurveys = new LinkedList<>();

        if (internalValidSurveys != null) {
            for (Survey survey : internalValidSurveys) {
                try {
                    if (checkSurveyValidity(survey)) {
                        validSurveys.add(new com.instabug.survey.Survey(survey.getId(), survey.getTitle()));
                    }
                } catch (ParseException e) {
                    if (e.getMessage() != null) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Error:" + e.getMessage() + "while getting valid surveys", e);
                    }
                }
            }
        }
        return validSurveys;
    }

    private void log(String message) {
        InstabugSDKLogger.i(Constants.LOG_TAG, "SurveysValidator: " + message);
    }

    public interface Callback {
        void onTimeEventTrigger(@NonNull Survey survey);

        void onUserEventTrigger(@NonNull Survey survey);
    }
}
