package com.instabug.survey.ui;

import android.text.TextUtils;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentActivity;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.ui.BasePresenter;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.TimeUtils;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.survey.Constants;
import com.instabug.survey.cache.SurveysCacheManager;
import com.instabug.survey.callbacks.OnFinishCallback;
import com.instabug.survey.common.LayoutUtils;
import com.instabug.survey.models.Question;
import com.instabug.survey.models.State;
import com.instabug.survey.models.Survey;
import com.instabug.survey.models.SurveyTypes;
import com.instabug.survey.network.service.InstabugSurveysSubmitterJob;
import com.instabug.survey.network.util.SubmittingSurveysUtil;
import com.instabug.survey.settings.PersistableSettings;
import com.instabug.survey.settings.SurveysSettings;
import com.instabug.survey.ui.survey.SurveyAbstractFragment;

import org.json.JSONException;

import java.util.List;

/**
 * Created by zak on 1/4/18.
 */

public class SurveyPresenter extends BasePresenter<SurveyActivityContract.View>
        implements SurveyActivityContract.Presenter {
    @Nullable
    private ViewType state;

    public SurveyPresenter(SurveyActivityContract.View view) {
        super(view);
    }

    @Override
    public void setLayoutHeightSecondary(ViewType viewType, boolean withAnimation) {
        this.state = viewType;
        if (view != null) {
            SurveyActivityContract.View surveyFragmentView = view.get();
            if (surveyFragmentView != null) {
                if (surveyFragmentView.getViewContext() != null) {
                    FragmentActivity activity = surveyFragmentView.getViewContext();
                    if (activity != null) {
                        int heightInPx = LayoutUtils.getLayoutHeight(activity, viewType);
                        if (withAnimation) {
                            surveyFragmentView.setFrameLayoutHeightWithAnimation(heightInPx);
                        } else {
                            surveyFragmentView.setFrameLayoutHeightTo(heightInPx);
                        }
                    }
                }
            }
        }
    }

    @Nullable
    public ViewType getState() {
        return state;
    }

    @Override
    public void handleOnBackPressedLogic() {
        if (view.get() != null) {
            SurveyActivityContract.View surveyFragmentView = view.get();
            if (surveyFragmentView != null && surveyFragmentView.getViewContext() != null) {
                FragmentActivity activity = surveyFragmentView.getViewContext();
                if (activity != null) {
                    if (activity.getSupportFragmentManager().getFragments().size() > 0) {
                        for (Fragment fragment : activity.getSupportFragmentManager()
                                .getFragments()) {
                            if (fragment instanceof SurveyAbstractFragment) {
                                ((SurveyAbstractFragment) fragment).onBackPressed();
                                break;
                            }
                        }
                    }
                }
            }
        }
    }

    @Override
    public void submitSurvey(final Survey survey) {
        if (survey == null) return;
        survey.setSubmitted();
        String sessionID = InstabugCore.getLatestSessionReplayId();
        survey.setSessionID(sessionID);
        PoolProvider.postIOTask(new Runnable() {
            @Override
            public void run() {
                SurveysCacheManager.update(survey);
            }
        });
        if (PersistableSettings.getInstance() != null)
            PersistableSettings.getInstance().setLastSurveyTime(TimeUtils.currentTimeMillis());

        handleOnFinishCallback(survey, State.SUBMITTED);

        if (view.get() != null) {
            SurveyActivityContract.View activityView = view.get();
            if (activityView != null && activityView.getViewContext() != null) {
                InstabugSurveysSubmitterJob.getInstance().start();
                if (survey.isNPSSurvey()) {
                    activityView.finishNPSSurvey(survey.isAppStoreRatingEnabled() && SurveysSettings.isAppLive());
                    return;
                } else if (survey.isStoreRatingSurvey()) {
                    activityView.finishSurvey(showThanksForAppRatingSurvey(survey));
                    return;
                }
                activityView.finishSurvey(true);
            }
        }
    }

    private boolean showThanksForAppRatingSurvey(Survey survey) {
        return !survey.isGooglePlayAppRating() && !TextUtils.isEmpty(survey.getQuestions().get(2).getAnswer());
    }

    @Override
    public void dismissSurvey(final Survey survey) {
        if (survey != null) {
            survey.setDismissed();
            if (survey.isCancelled()
                    && (survey.getSessionCounter() >= SurveysSettings
                    .getSurveysReshowSessionCount())) {

                if (survey.isOptInSurvey()) {
                    survey.setShouldShowAgain(true);
                    survey.resetSessionsCounter();

                } else {
                    if (survey.getSessionCounter() != 0) {
                        survey.setShouldShowAgain(false);
                    }
                }
            }

            String surveyState = getSurveyState(survey);

            handleOnFinishCallback(survey, surveyState);
            String sessionID = InstabugCore.getLatestSessionReplayId();
            survey.setSessionID(sessionID);
            SurveysCacheManager.update(survey);

            if (PersistableSettings.getInstance() != null)
                PersistableSettings.getInstance().setLastSurveyTime(TimeUtils.currentTimeMillis());
            if (view.get() != null) {
                SurveyActivityContract.View activityView = view.get();
                if (activityView != null && activityView.getViewContext() != null) {
                    InstabugSurveysSubmitterJob.getInstance().start();
                    activityView.finishSurvey(false);
                }
            }
        }
    }

    private void handleOnFinishCallback(Survey survey, @State String state) {
        OnFinishCallback onFinishCallback = SurveysSettings.getOnFinishCallback();
        if (onFinishCallback != null) {
            try {
                onFinishCallback.onFinish(Long.toString(survey.getId()),
                        state, SubmittingSurveysUtil.getSurveyAsResponse(survey, state));
            } catch (JSONException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG,
                        "Something went wrong during parsing Survey object in onFinishCallback",
                        e);
            }
        }
    }

    @State
    @VisibleForTesting
    public String getSurveyState(Survey survey) {
        if (survey.getType() == SurveyTypes.CUSTOM || survey.getType() == SurveyTypes.NPS) {
            return State.DISMISSED;
        }
        List<Question> surveyQuestions = survey.getQuestions();
        for (int index = 0; index < surveyQuestions.size(); index++) {
            String questionAnswer = surveyQuestions.get(index).getAnswer();
            if (questionAnswer == null || questionAnswer.equals("")) {
                if (index == 0) {
                    // The first question wasn't answered, the survey was dismissed from first question
                    return State.DISMISSED;
                } else {
                    // A question other than the first wasn't answered
                    return State.ENDED;
                }
            }
        }
        return State.SUBMITTED;
    }

    public boolean shouldShowWelcomeScreen() {
        return SurveysSettings.shouldShowSurveysWelcomeScreen();
    }
}
