package com.instabug.survey.models;

import static com.instabug.survey.common.models.SyncingStatus.READY_TO_SEND;
import static com.instabug.survey.models.SurveyTypes.CUSTOM;
import static com.instabug.survey.models.SurveyTypes.NPS;
import static com.instabug.survey.models.SurveyTypes.STORE_RATING;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.internal.storage.cache.Cacheable;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.ListUtils;
import com.instabug.library.util.TimeUtils;
import com.instabug.survey.Constants;
import com.instabug.survey.common.models.ActionEvent;
import com.instabug.survey.common.models.BaseLocalization;
import com.instabug.survey.common.models.Condition;
import com.instabug.survey.common.models.Frequency;
import com.instabug.survey.common.models.IUserInteraction;
import com.instabug.survey.common.models.SyncingStatus;
import com.instabug.survey.common.models.Target;
import com.instabug.survey.common.models.UserInteraction;
import com.instabug.survey.utils.DateUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

public class Survey implements Cacheable, Serializable, IUserInteraction {

    // keys used by toJson() & fromJson() methods
    private static final String KEY_PUBLISHED = "published";
    private static final String KEY_PAUSED = "paused";
    private static final String KEY_ID = "id";
    private static final String KEY_TITLE = "title";
    private static final String KEY_TOKEN = "token";
    private static final String HAS_RESPOND = "has_respond";

    private static final String KEY_QUESTIONS = "questions";
    private static final String KEY_TARGET = "target";
    private static final String KEY_TARGET_AUDIENCE = "primitive_types";
    private static final String KEY_CUSTOM_ATTRIBUTES = "custom_attributes";
    private static final String KEY_USER_EVENTS = "user_events";
    private static final String KEY_CONDITIONS_OPERATOR = "operator";
    private static final String KEY_ANSWERED = "answered";
    private static final String KEY_DISMISSED_AT = "dismissed_at";
    private static final String KEY_IS_CANCELLED = "is_cancelled";
    private static final String KEY_SURVEY_EVENTS = "events";
    private static final String KEY_SURVEY_STATE = "survey_state";
    private static final String KEY_SHOULD_SHOW_AGAIN = "should_show_again";
    private static final String KEY_SESSION_COUNTER = "session_counter";
    private static final String KEY_WELCOME_SCREEN = "welcome_screen";
    private static final String KEY_TYPE = "type";
    private static final String KEY_THANKS_LIST = "thanks_list";
    private static final String KEY_SHOW_AT = "show_at";
    private static final String KEY_IS_GOOGLE_PLAY_APP_RATING = "app_rating";
    private static final String KEY_IS_DISMISSIBLE = "dismissible";

    private long id;
    private String title;
    private @SurveyTypes
    int type;
    @Nullable
    private String token;
    private ArrayList<Question> questions;
    private UserInteraction userInteraction;
    private boolean paused = false;
    private ArrayList<ThankYouItem> thankYouItems;
    private BaseLocalization localization;
    private boolean isGooglePlayAppRating = false;
    private boolean isDismissible = true;

    @Nullable
    private String sessionID;

    public Survey() {
        title = "";
        questions = new ArrayList<>();
        thankYouItems = new ArrayList<>();
        localization = new BaseLocalization();
        userInteraction = new UserInteraction(UserInteractionOnTypes.SURVEY);
    }

    public static List<Survey> getPausedSurveysFromJson(JSONObject response) throws JSONException {
        JSONArray jsonArray = response.getJSONArray(KEY_PAUSED);
        List<Survey> pausedSurveys = new ArrayList<>(jsonArray.length());
        for (int i = 0; i < jsonArray.length(); i++) {
            Survey survey = new Survey();
            survey.setId(jsonArray.getLong(i));
            survey.setPaused(true);
            pausedSurveys.add(survey);
        }
        return pausedSurveys;
    }

    public static List<Survey> fromJson(@NonNull JSONObject surveysJsonObject) throws JSONException {
        JSONArray jsonArray = surveysJsonObject.getJSONArray(KEY_PUBLISHED);
        List<Survey> surveys = new ArrayList<>();
        for (int i = 0; i < jsonArray.length(); i++) {
            JSONObject surveyJsonObject = jsonArray.getJSONObject(i);
            Survey survey = new Survey();
            survey.fromJson(surveyJsonObject.toString());
            surveys.add(survey);
        }
        return surveys;
    }

    public Target getTarget() {
        return userInteraction.getTarget();
    }

    public void setTarget(Target target) {
        userInteraction.setTarget(target);
    }

    public boolean isPaused() {
        return paused;
    }

    public void setPaused(boolean paused) {
        this.paused = paused;
    }

    @Override
    public long getSurveyId() {
        return id;
    }

    @Override
    public UserInteraction getUserInteraction() {
        return userInteraction;
    }

    public void setUserInteraction(UserInteraction userInteraction) {
        this.userInteraction = userInteraction;
    }

    @SurveyTypes
    public int getType() {
        return type;
    }

    public String getTypeAsString() {
        switch (type) {
            case CUSTOM:
                return "Custom";
            case NPS:
                return "NPS";
            case STORE_RATING:
                return "StoreRating";
            default:
                return "";
        }
    }

    public void setType(@SurveyTypes int type) {
        this.type = type;
    }

    public String getTitle() {
        return title;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public void setSubmitted() {
        setCancelled(false);
        setAnswered(true);

        ActionEvent submitEvent;
        //Add rate Event only if the user clicked on the rateUsOnPlayStore button
        if (isNPSSurvey() && isAnswered() && hasNPSSubmitted() && hasPositiveNpsAnswer()) {
            submitEvent = new ActionEvent(ActionEvent.EventType.RATE,
                    TimeUtils.currentTimeSeconds(), getEventIndex());
        } else {
            submitEvent = new ActionEvent(ActionEvent.EventType.SUBMIT,
                    TimeUtils.currentTimeSeconds(), getEventIndex());
            //resetting the attempts count as required for reshowing OptIn-surveys
            if (isOptInSurvey()) {
                userInteraction.setAttemptCount(0);
            }
        }

        setSurveyState(READY_TO_SEND);

        //Ignore adding submit event if the last added event was also submit
        Target target = userInteraction.getTarget();
        if (target.getActionEvents() != null && target.getActionEvents().size() > 0
                && target.getActionEvents().get(target.getActionEvents().size() - 1).getEventType()
                == ActionEvent.EventType.SUBMIT
                && submitEvent.getEventType() == ActionEvent.EventType.SUBMIT) {
            return;
        }
        if (target.getActionEvents() != null) {
            target.getActionEvents().add(submitEvent);
        }
    }

    public boolean isNPSSurvey() {
        return getType() == SurveyTypes.NPS;
    }

    public boolean isStoreRatingSurvey() {
        return getType() == SurveyTypes.STORE_RATING;
    }

    public ArrayList<Question> getQuestions() {
        return questions;
    }

    public void setQuestions(ArrayList<Question> question) {
        this.questions = question;
    }

    public ArrayList<Condition> getTargetAudiences() {
        return userInteraction.getTarget().getTargetAudiences();
    }

    public void setTargetAudiences(ArrayList<Condition> targetAudiences) {
        this.userInteraction.getTarget().setTargetAudiences(targetAudiences);
    }

    public ArrayList<Condition> getCustomAttributes() {
        return userInteraction.getTarget().getCustomAttributes();
    }

    public void setCustomAttributes(ArrayList<Condition> customAttributes) {
        this.userInteraction.getTarget().setCustomAttributes(customAttributes);
    }

    public ArrayList<Condition> getUserEvents() {
        return userInteraction.getTarget().getUserEvents();
    }

    public void setUserEvents(ArrayList<Condition> userEvents) {
        this.userInteraction.getTarget().setUserEvents(userEvents);
    }

    public ArrayList<ActionEvent> getSurveyEvents() {
        return userInteraction.getTarget().getActionEvents();
    }

    public void setSurveyEvents(ArrayList<ActionEvent> actionEvents) {
        this.userInteraction.getTarget().setActionEvents(actionEvents);
    }

    public String getConditionsOperator() {
        return userInteraction.getTarget().getConditionsOperator();
    }

    public void setConditionsOperator(String conditionOperator) {
        this.userInteraction.getTarget().setConditionsOperator(conditionOperator);
    }

    public boolean hasPositiveNpsAnswer() {
        return isNPSSurvey() && (isPromoter() || isPassive());
    }

    private boolean hasNPSSubmitted() {
        if (userInteraction.getTarget().getActionEvents() != null) {
            for (ActionEvent event : userInteraction.getTarget().getActionEvents()) {
                if (event.getEventType() == ActionEvent.EventType.SUBMIT) {
                    return true;
                }
            }
        }
        return false;
    }

    public int getEventIndex() {
        return userInteraction.getEventIndex();
    }

    public void setEventIndex(int eventIndex) {
        userInteraction.setEventIndex(eventIndex);
    }

    public boolean isOptInSurvey() {
        return getToken() != null && !String.valueOf(getToken()).equals("null");
    }

    @Nullable
    public String getToken() {
        return token;
    }

    public void setToken(@Nullable String token) {
        this.token = token;
    }

    public void setDismissed() {
        setSurveyState(READY_TO_SEND);

        //Meaning that user dismissing the RateOnPlayStore Don't add dismiss event.
        if (isNPSSurvey() && hasPositiveNpsAnswer() && hasNPSSubmitted()) {
            return;
        }

        setShouldShowAgain(isOptInSurvey() || this.userInteraction.getDismissedAt() == 0); // dismissed before this current dismiss for auto showing surveys
        userInteraction.setDismissedAt(TimeUtils.currentTimeSeconds());
        setCancelled(true);

        //Ignore adding dismissEvent if the last added event was also dismissEvent
        if (userInteraction.getTarget().getActionEvents().size() > 0
                && userInteraction.getTarget().getActionEvents().get(userInteraction.getTarget().getActionEvents().size() - 1).getEventType()
                == ActionEvent.EventType.DISMISS) {
            return;
        }
        ActionEvent dismissEvent = new ActionEvent(ActionEvent.EventType.DISMISS,
                userInteraction.getDismissedAt(), getEventIndex());
        userInteraction.getTarget().getActionEvents().add(dismissEvent);
    }

    public void addShowEvent() {
        userInteraction.setShownAt(TimeUtils.currentTimeSeconds());
        userInteraction.incrementAttemptCount();
        ActionEvent showEvent = new ActionEvent(ActionEvent.EventType.SHOW,
                userInteraction.getShownAt(), userInteraction.incrementEventIndex());
        userInteraction.getTarget().getActionEvents().add(showEvent);
    }

    public void addRateEvent() {
        ActionEvent showEvent = new ActionEvent(ActionEvent.EventType.RATE,
                TimeUtils.currentTimeSeconds(), getAttemptCount());
        userInteraction.getTarget().getActionEvents().add(showEvent);
    }

    public long getShownAt() {
        return userInteraction.getShownAt();
    }

    public void setShowAt(long shownAt) {
        this.userInteraction.setShownAt(shownAt);
    }


    public int getAttemptCount() {
        return userInteraction.getAttemptCount();
    }

    public void setAttemptCount(int attemptsCount) {
        userInteraction.setAttemptCount(attemptsCount);
    }

    public void setSessionCount(int sessionCount) {
        userInteraction.setSessionCounter(sessionCount);
    }


    private boolean isPromoter() {
        return getNPSAnswerValue() > 8;
    }

    private boolean isPassive() {
        return getNPSAnswerValue() > 6 && getNPSAnswerValue() <= 8;
    }

    private boolean isDetractor() {
        return getNPSAnswerValue() <= 6;
    }

    private int getNPSAnswerValue() {
        try {
            Question question = ListUtils.safeGet(questions, 0);
            if (question == null) return 0;
            String answer = question.getAnswer();
            if (answer == null) return 0;
            return Integer.parseInt(answer);
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "NPS score parsing failed du to: " + e.getMessage());
            return 0;
        }
    }

    @Nullable
    public String getThankYouTitle() {
        return isNPSSurvey() ? getNPSThankYouTitle() : getCustomThankYouTitle();
    }

    @Nullable
    private String getNPSThankYouTitle() {
        ArrayList<ThankYouItem> thankYouItems = getThankYouItems();
        if (thankYouItems.size() > 0) {
            ThankYouItem promoterItem = ListUtils.safeGet(thankYouItems, 0);
            ThankYouItem passiveItem = ListUtils.safeGet(thankYouItems, 1);
            ThankYouItem detractorItem = ListUtils.safeGet(thankYouItems, 2);
            if (isPromoter() && promoterItem != null) {
                return promoterItem.getThankYouTitle();
            } else if (isPassive() && passiveItem != null) {
                return passiveItem.getThankYouTitle();
            } else if (isDetractor() && detractorItem != null) {
                return detractorItem.getThankYouTitle();
            }
        }
        return null;
    }

    @Nullable
    private String getCustomThankYouTitle() {
        ArrayList<ThankYouItem> thankYouItems = getThankYouItems();
        if (thankYouItems.size() > 0) {
            ThankYouItem thankYouItem = ListUtils.safeGet(thankYouItems, 0);
            if (thankYouItem != null) {
                return thankYouItem.getThankYouTitle();
            }
        }
        return null;
    }

    @Nullable
    public String getThankYouMessage() {
        return isNPSSurvey() ? getNPSThankYouMessage() : getCustomThankYouMessage();
    }

    @Nullable
    private String getNPSThankYouMessage() {
        ArrayList<ThankYouItem> thankYouItems = getThankYouItems();
        if (thankYouItems.size() > 0) {
            ThankYouItem promoterItem = ListUtils.safeGet(thankYouItems, 0);
            ThankYouItem passiveItem = ListUtils.safeGet(thankYouItems, 1);
            ThankYouItem detractorItem = ListUtils.safeGet(thankYouItems, 2);
            if (isPromoter() && promoterItem != null) {
                return promoterItem.getThankYouMessage();
            } else if (isPassive() && passiveItem != null) {
                return passiveItem.getThankYouMessage();
            } else if (isDetractor() && detractorItem != null) {
                return detractorItem.getThankYouMessage();
            }
        }

        return null;
    }

    @Nullable
    private String getCustomThankYouMessage() {
        ArrayList<ThankYouItem> thankYouItems = getThankYouItems();
        if (thankYouItems.size() > 0) {
            ThankYouItem thankYouItem = ListUtils.safeGet(thankYouItems, 0);
            if (thankYouItem != null) {
                return thankYouItem.getThankYouMessage();
            }
        }
        return null;
    }

    public boolean isAppStoreRatingEnabled() {
        ArrayList<ThankYouItem> thankYouItems = getThankYouItems();
        if (thankYouItems.size() > 0) {
            ThankYouItem promoterItem = ListUtils.safeGet(thankYouItems, 0);
            ThankYouItem passiveItem = ListUtils.safeGet(thankYouItems, 1);
            if (isPromoter() && promoterItem != null) {
                return promoterItem.isAppStoreRatingEnabled();
            } else if (isPassive() && passiveItem != null) {
                return passiveItem.isAppStoreRatingEnabled();
            }
        }

        return false;
    }

    @Nullable
    public String getRatingCTATitle() {
        if (isAppStoreRatingEnabled()) {
            ArrayList<ThankYouItem> thankYouItems = getThankYouItems();
            if (thankYouItems.size() > 0) {
                ThankYouItem promoterItem = ListUtils.safeGet(thankYouItems, 0);
                ThankYouItem passiveItem = ListUtils.safeGet(thankYouItems, 1);
                if (isPromoter() && promoterItem != null) {
                    return promoterItem.getCallActionString();
                } else if (isPassive() && passiveItem != null) {
                    return passiveItem.getCallActionString();
                }
            }
        }
        return null;
    }

    public void incrementSessionCount() {
        userInteraction.incrementSessionCount();
    }


    public boolean isCancelled() {
        return userInteraction.isCancelled();
    }

    public void setCancelled(boolean cancelled) {
        userInteraction.setCancelled(cancelled);
    }


    public void resetSessionsCounter() {
        userInteraction.setSessionCounter(0);
    }

    /**
     * This method to rest user interaction but since it holds target that has some data other than user's action
     * so we need to hold it and only clear user's actions.
     */
    public void resetUserInteractions() {
        Target target = userInteraction.getTarget();
        target.setActionEvents(new ArrayList<ActionEvent>());
        userInteraction = new UserInteraction(UserInteractionOnTypes.SURVEY);
        userInteraction.setTarget(target);
    }

    /**
     * Reset User Answers
     */
    public void resetUserAnswers() {
        for (Question question : questions) {
            question.resetAnswer();
        }
    }


    public long getRespondedAt() {
        //surveys shouldn't be sent with answers if it has been dismissed
        //answers should be null and last-responded-at should be 0
        if (isLastEventDismiss()) {
            return 0;
        }

        if (userInteraction.getTarget().getActionEvents() != null && userInteraction.getTarget().getActionEvents().size() > 0) {
            for (ActionEvent actionEvent : userInteraction.getTarget().getActionEvents()) {
                if (actionEvent.getEventType() == ActionEvent.EventType.SUBMIT) {
                    return actionEvent.getTimestamp();
                }
            }
        }
        if (getQuestions() != null && getQuestions().size() > 0) {
            for (int i = getQuestions().size() - 1; i >= 0; i--) {
                if (getQuestions().get(i).getAnsweredAt() > 0) {
                    return getQuestions().get(i).getAnsweredAt();
                }
            }
        }
        return 0;
    }

    public boolean isLastEventDismiss() {
        return (userInteraction.getTarget().getActionEvents() != null && userInteraction.getTarget().getActionEvents().size() > 0
                && userInteraction.getTarget().getActionEvents().get(userInteraction.getTarget().getActionEvents().size() - 1).getEventType()
                == ActionEvent.EventType.DISMISS);
    }

    @Override
    public String toJson() throws JSONException {
        JSONObject surveyJsonObject = new JSONObject();
        surveyJsonObject.put(KEY_ID, id)
                .put(KEY_TYPE, type)
                .put(KEY_IS_GOOGLE_PLAY_APP_RATING, isGooglePlayAppRating)
                .put(KEY_TITLE, title)
                .put(KEY_TOKEN, token != null ? token : "")
                .put(KEY_QUESTIONS, Question.toJson(questions))
                .put(KEY_TARGET, Target.toJson(userInteraction.getTarget()))
                .put(KEY_SURVEY_EVENTS, ActionEvent.toJson(userInteraction.getTarget().getActionEvents()))
                .put(KEY_ANSWERED, userInteraction.isAnswered())
                .put(KEY_SHOW_AT, userInteraction.getShownAt())
                .put(KEY_DISMISSED_AT, getDismissedAt())
                .put(KEY_IS_CANCELLED, userInteraction.isCancelled())
                .put(KEY_SURVEY_STATE, getSurveyState().toString())
                .put(KEY_SHOULD_SHOW_AGAIN, shouldShowAgain())
                .put(KEY_THANKS_LIST, ThankYouItem.toJson(thankYouItems))
                .put(KEY_SESSION_COUNTER, getSessionCounter());

        localization.localizationToJson(surveyJsonObject);
        return surveyJsonObject.toString();
    }

    public boolean shouldShowAgain() {
        return userInteraction.isShouldShowAgain();
    }

    public long getDismissedAt() {
        return userInteraction.getDismissedAt();
    }

    public void setDismissedAt(long dismissedAt) {
        userInteraction.setDismissedAt(dismissedAt);
    }

    public SyncingStatus getSurveyState() {
        return userInteraction.getSurveyState();
    }

    public void setSurveyState(SyncingStatus surveyState) {
        userInteraction.setSurveyState(surveyState);
    }

    public int getSessionCounter() {
        return userInteraction.getSessionCounter();
    }

    private void setSessionCounter(int sessionCounter) {
        userInteraction.setSessionCounter(sessionCounter);
    }

    @Override
    public void fromJson(String modelAsJson) throws JSONException {
        JSONObject surveyJsonObject = new JSONObject(modelAsJson);
        if (surveyJsonObject.has(KEY_ID)) {
            setId(surveyJsonObject.getLong(KEY_ID));
        }
        if (surveyJsonObject.has(KEY_TYPE)) {
            @SurveyTypes int type = surveyJsonObject.getInt(KEY_TYPE);
            setType(type);
        }
        if (surveyJsonObject.has(KEY_TITLE)) {
            setTitle(surveyJsonObject.getString(KEY_TITLE));
        }
        if (surveyJsonObject.has(KEY_TOKEN) && surveyJsonObject.get(KEY_TOKEN) != JSONObject.NULL) {
            setToken(surveyJsonObject.getString(KEY_TOKEN));
        }
        if (surveyJsonObject.has(KEY_SURVEY_EVENTS)) {
            userInteraction.getTarget().setActionEvents(ActionEvent.fromJson(surveyJsonObject.getJSONArray(KEY_SURVEY_EVENTS)));
        }
        if (surveyJsonObject.has(KEY_QUESTIONS)) {
            setQuestions(Question.fromJson(surveyJsonObject.getJSONArray(KEY_QUESTIONS)));
        }
        if (surveyJsonObject.has(KEY_TARGET)) {
            JSONObject targetJsonObject = surveyJsonObject.getJSONObject(KEY_TARGET);
            userInteraction.getTarget().fromJson(targetJsonObject.toString().replace("\\", ""));
        }
        if (surveyJsonObject.has(KEY_ANSWERED)) {
            setAnswered(surveyJsonObject.getBoolean(KEY_ANSWERED));
        }
        if (surveyJsonObject.has(KEY_IS_CANCELLED)) {
            setCancelled(surveyJsonObject.getBoolean(KEY_IS_CANCELLED));
        }
        if (surveyJsonObject.has(KEY_SURVEY_STATE)) {
            setSurveyState(SyncingStatus.valueOf(surveyJsonObject.getString(KEY_SURVEY_STATE)));
        }
        if (surveyJsonObject.has(KEY_SHOULD_SHOW_AGAIN)) {
            setShouldShowAgain(surveyJsonObject.getBoolean(KEY_SHOULD_SHOW_AGAIN));
        }
        if (surveyJsonObject.has(KEY_SESSION_COUNTER)) {
            setSessionCounter(surveyJsonObject.getInt(KEY_SESSION_COUNTER));
        }
        if (surveyJsonObject.has(KEY_DISMISSED_AT)) {
            setDismissedAt(surveyJsonObject.getInt(KEY_DISMISSED_AT));
        }
        if (surveyJsonObject.has(KEY_SHOW_AT)) {
            setShowAt(surveyJsonObject.getInt(KEY_SHOW_AT));
        }
        if (surveyJsonObject.has(KEY_THANKS_LIST)) {
            JSONArray jsonArray = surveyJsonObject.getJSONArray(KEY_THANKS_LIST);
            setThankYouItems(ThankYouItem.fromJson(jsonArray));
        }

        if (surveyJsonObject.has(KEY_IS_DISMISSIBLE)) {
            setDismissible(surveyJsonObject.getBoolean(KEY_IS_DISMISSIBLE));
        }
        localization.localizationFromJson(surveyJsonObject);
        setGooglePlayAppRating(surveyJsonObject.optBoolean(KEY_IS_GOOGLE_PLAY_APP_RATING, false));

    }


    public long getId() {
        return id;
    }

    public Survey setId(long id) {
        this.id = id;
        return this;
    }

    public boolean isDismissible() {
        return isDismissible;
    }

    public void setDismissible(boolean dismissible) {
        isDismissible = dismissible;
    }

    public void setThankYouItems(ArrayList<ThankYouItem> thankYouItems) {
        this.thankYouItems = thankYouItems;
    }

    public ArrayList<ThankYouItem> getThankYouItems() {
        return thankYouItems;
    }

    public boolean isAnswered() {
        return userInteraction.isAnswered();
    }

    public void setAnswered(boolean answered) {
        this.userInteraction.setAnswered(answered);
    }

    public void setShouldShowAgain(boolean shouldShowAgain) {
        userInteraction.setShouldShowAgain(shouldShowAgain);
    }


    public BaseLocalization getLocalization() {
        return localization;
    }

    public void setLocalization(BaseLocalization localization) {
        this.localization = localization;
    }

    public void clearAnswers() {
        for (Question q : questions) {
            q.setAnswer(null);
        }
    }

    public boolean isLastEventSubmit() {
        return (userInteraction.getTarget().getActionEvents() != null && userInteraction.getTarget().getActionEvents().size() > 0
                && userInteraction.getTarget().getActionEvents().get(userInteraction.getTarget().getActionEvents().size() - 1).getEventType()
                == ActionEvent.EventType.SUBMIT);
    }

    public boolean isGooglePlayAppRating() {
        return isGooglePlayAppRating;
    }

    public void setGooglePlayAppRating(boolean googlePlayAppRating) {
        this.isGooglePlayAppRating = googlePlayAppRating;
    }

    @Override
    public int hashCode() {
        return String.valueOf(getId()).hashCode();
    }

    @Override
    public boolean equals(@Nullable Object survey) {
        if (survey == null) {
            return false;
        }

        if (survey instanceof Survey) {
            Survey comparedSurvey = (Survey) survey;
            return comparedSurvey.getId() == getId();
        } else {
            return false;
        }
    }

    @NonNull
    @Override
    public String toString() {
        try {
            return toJson();
        } catch (JSONException e) {
            if (e.getMessage() != null) {
                InstabugSDKLogger.e("Survey", e.getMessage(), e);
            }
        }
        return super.toString();
    }

    public boolean shouldShow() {
        Target target = userInteraction.getTarget();
        boolean shouldShowEveryTime = target.getFrequency().shouldShowEveryTime();
        boolean notShownBefore = !userInteraction.isAlreadyShown();
        boolean isFrequent = !target.getFrequency().shouldShowOnce();
        boolean isFrequencyPeriodPassed = DateUtils.getDifferenceInDaysFromSeconds(getShownAt()) >= target.getFrequency().getShowingInterval();
        return shouldShowEveryTime ||
                notShownBefore ||
                (isFrequent && isFrequencyPeriodPassed) ||
                shouldReshowAfterDismiss();
    }

    public boolean shouldReshowAfterDismiss() {
        Frequency frequency = getTarget().getFrequency();
        if (frequency.shouldNotReshowAfterDismiss()) return false;
        boolean isLastEventDismiss = isLastEventDismiss();
        int lastShowDifferenceInDays = DateUtils.getDifferenceInDaysFromSeconds(getShownAt());
        boolean isPeriodPassed = lastShowDifferenceInDays >= frequency.getDismissedReshowInterval();
        return isLastEventDismiss && isPeriodPassed;
    }

    /**
     * Gets the last question of App Rating Survey only
     * Store-Redirecting App Rating Survey has 3 questions, last index is 2
     * In-App GooglePlay App Rating Survey has 2 questions, last index is 1
     *
     * @return Question if survey is a {@link SurveyTypes#STORE_RATING} type, null otherwise
     */
    @Nullable
    public Question getSecondaryNegativeQuestion() {
        if (isStoreRatingSurvey()) {
            if (isGooglePlayAppRating()) {
                return getQuestions().get(1);
            }
            return getQuestions().get(2);
        }
        return null;
    }

    @Nullable
    public String getSessionID() {
        return sessionID;
    }

    public void setSessionID(@Nullable String sessionID) {
        this.sessionID = sessionID;
    }
}
