package com.instabug.survey.models;

import static com.instabug.survey.models.Question.Type.MCQ;
import static com.instabug.survey.models.Question.Type.NOT_AVAILABLE;
import static com.instabug.survey.models.Question.Type.NPS;
import static com.instabug.survey.models.Question.Type.POP_UP;
import static com.instabug.survey.models.Question.Type.STAR_RATE;
import static com.instabug.survey.models.Question.Type.STORE_RATING;
import static com.instabug.survey.models.Question.Type.TEXT;

import android.text.TextUtils;

import androidx.annotation.IntDef;
import androidx.annotation.Nullable;

import com.instabug.library.internal.storage.cache.Cacheable;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.TimeUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.ArrayList;

/**
 * @author mesbah
 */

public class Question implements Cacheable, Serializable {

    // keys used by toJson() & fromJson() methods
    private static final String KEY_ID = "id";
    private static final String KEY_TITLE = "title";
    private static final String KEY_TYPE = "type";
    private static final String KEY_OPTIONAL_ANSWERS = "options";
    private static final String KEY_ANSWER = "answer";
    private static final String KEY_ANSWERED_AT = "answered_at";
    private static final String KEY_IS_ENABLED = "enabled";


    private long id = -1;

    @Nullable
    private String title;

    private @Type
    int type;

    @Nullable
    private ArrayList<String> optionalAnswers;

    @Nullable
    private String answer;

    private long answeredAt;

    private boolean isEnabled = true;

    public Question() {
        this.type = Type.NOT_AVAILABLE;
    }

    public static ArrayList<Question> fromJson(JSONArray questionsJsonArray) throws JSONException {
        ArrayList<Question> questions = new ArrayList<>();
        if (questionsJsonArray != null) {
            for (int i = 0; i < questionsJsonArray.length(); i++) {
                Question question = new Question();
                question.fromJson(questionsJsonArray.getJSONObject(i).toString());
                questions.add(question);
            }
        }
        return questions;
    }

    public static JSONArray toJson(ArrayList<Question> questionArrayList) throws JSONException {
        JSONArray questionsJsonArray = new JSONArray();
        if (questionArrayList != null) {
            for (int i = 0; i < questionArrayList.size(); i++) {
                questionsJsonArray.put(new JSONObject(questionArrayList.get(i).toJson()));
            }
        }
        return questionsJsonArray;
    }

    public long getId() {
        return id;
    }

    public void setId(long id) {
        this.id = id;
    }

    @Nullable
    public String getTitle() {
        return title;
    }

    public void setTitle(@Nullable String title) {
        this.title = title;
    }


    public boolean isEnabled() {
        return isEnabled;
    }

    public void setEnabled(boolean enabled) {
        isEnabled = enabled;
    }

    public @Type
    int getType() {
        return type;
    }

    public String getTypeAsString() {
        switch (type) {
            case TEXT:
                return "Textfield";
            case MCQ:
                return "MCQ";
            case STAR_RATE:
                return "StarRating";
            case STORE_RATING:
                return "AppStoreRating";
            case NPS:
                return "NPS";
            case POP_UP:
                return "Alert";
            case NOT_AVAILABLE:
            default:
                return "";
        }
    }

    public void setType(@Type int type) {
        this.type = type;
    }

    @Nullable
    public ArrayList<String> getOptionalAnswers() {
        return optionalAnswers;
    }

    public void setOptionalAnswers(@Nullable ArrayList<String> optionalAnswers) {
        this.optionalAnswers = optionalAnswers;
    }

    @Nullable
    public String getAnswer() {
        return answer;
    }

    public void setAnswer(@Nullable String answer) {
        this.answer = answer;
        if (answer != null && !TextUtils.isEmpty(answer))
            setAnsweredAt(TimeUtils.currentTimeSeconds());
    }

    public void resetAnswer() {
        this.answer = null;
        setAnsweredAt(0);
    }

    public long getAnsweredAt() {
        return answeredAt;
    }

    private void setAnsweredAt(long answeredAt) {
        this.answeredAt = answeredAt;
    }

    @Override
    public String toJson() throws JSONException {
        JSONObject questionJsonObject = new JSONObject();
        questionJsonObject.put(KEY_ID, getId())
                .put(KEY_TITLE, getTitle())
                .put(KEY_TYPE, getType())
                .put(KEY_OPTIONAL_ANSWERS, optionalAnswers != null ? new JSONArray(optionalAnswers) : new JSONArray())
                .put(KEY_ANSWER, getAnswer() != null ? getAnswer() : "")
                .put(KEY_ANSWERED_AT, getAnsweredAt())
                .put(KEY_IS_ENABLED,isEnabled());
        return questionJsonObject.toString();
    }

    @Override
    public void fromJson(String modelAsJson) throws JSONException {

        JSONObject questionJsonObject = new JSONObject(modelAsJson);
        if (questionJsonObject.has(KEY_ID)) {
            setId(questionJsonObject.getLong(KEY_ID));
        }
        if (questionJsonObject.has(KEY_TITLE)) {
            setTitle(questionJsonObject.getString(KEY_TITLE));
        }
        if (questionJsonObject.has(KEY_TYPE)) {
            @Type int type = questionJsonObject.getInt(KEY_TYPE);
            setType(type);
        }

        if (questionJsonObject.has(KEY_OPTIONAL_ANSWERS)) {
            JSONArray answersJsonArray = questionJsonObject.getJSONArray(KEY_OPTIONAL_ANSWERS);
            ArrayList<String> answers = new ArrayList<>();
            for (int i = 0; i < answersJsonArray.length(); i++) {
                answers.add(String.valueOf(answersJsonArray.get(i)));
            }

            setOptionalAnswers(answers);
        }

        if (questionJsonObject.has(KEY_ANSWER)) {
            setAnswer(questionJsonObject.getString(KEY_ANSWER));
        }
        if (questionJsonObject.has(KEY_ANSWERED_AT)) {
            setAnsweredAt(questionJsonObject.getLong(KEY_ANSWERED_AT));
        }

        if(questionJsonObject.has(KEY_IS_ENABLED)) {
            setEnabled(questionJsonObject.getBoolean(KEY_IS_ENABLED));
        }
    }

    @Retention(RetentionPolicy.SOURCE)
    @IntDef({TEXT, MCQ, STAR_RATE, NPS, STORE_RATING, POP_UP, NOT_AVAILABLE})
    public @interface Type {
        int TEXT = 0;
        int MCQ = 1;
        int STAR_RATE = 2;
        int NPS = 3;
        int STORE_RATING = 4;
        int POP_UP = 5;
        int NOT_AVAILABLE = -1;
    }
}