package com.instabug.survey.common.models;

import com.instabug.library.internal.storage.cache.Cacheable;
import com.instabug.library.util.filters.Filters;
import com.instabug.survey.utils.SurveysFilterFunctions;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;

/**
 Created by Barakat on 25/11/2018
 */
public class Target implements Cacheable, Serializable {

    private static final String KEY_TARGET_AUDIENCE = "primitive_types";
    private static final String KEY_CUSTOM_ATTRIBUTES = "custom_attributes";
    private static final String KEY_USER_EVENTS = "user_events";
    private static final String KEY_ACTION_EVENTS = "events";
    private static final String KEY_SURVEY_EVENTS = "events";
    private static final String KEY_TRIGGER = "trigger";
    private static final String KEY_FREQUENCY = "frequency";
    private static final String KEY_CONDITIONS_OPERATOR = "operator";
    private static final String KEY_OPERATOR_AND = "and"; // Default

    private ArrayList<Condition> targetAudiences;
    private ArrayList<Condition> customAttributes;
    private ArrayList<Condition> userEvents;
    private ArrayList<ActionEvent> actionEvents;
    private Trigger trigger;
    private Frequency frequency;
    private String conditionsOperator;

    public Target() {
        targetAudiences = new ArrayList<>();
        customAttributes = new ArrayList<>();
        actionEvents = new ArrayList<>();
        userEvents = new ArrayList<>();
        trigger = new Trigger();
        frequency = new Frequency();
        conditionsOperator = KEY_OPERATOR_AND;
    }

    public static JSONObject toJson(Target target) throws JSONException {
        return new JSONObject(target.toJson());
    }

    public static Target fromJsonString(String targetJsonString) throws JSONException {
        Target target = new Target();
        target.fromJson(targetJsonString);
        return target;
    }

    public ArrayList<Condition> getTargetAudiences() {
        return Filters.applyOn(targetAudiences)
                .apply(SurveysFilterFunctions.knownInstabugConditions())
                .thenGet();
    }

    public void setTargetAudiences(ArrayList<Condition> targetAudiences) {
        this.targetAudiences = targetAudiences;
    }

    public ArrayList<Condition> getCustomAttributes() {
        return customAttributes;
    }

    public void setCustomAttributes(ArrayList<Condition> customAttributes) {
        this.customAttributes = customAttributes;
    }

    public ArrayList<Condition> getUserEvents() {
        return userEvents;
    }

    public void setUserEvents(ArrayList<Condition> userEvents) {
        this.userEvents = userEvents;
    }

    public ArrayList<ActionEvent> getActionEvents() {
        return actionEvents;
    }

    public void setActionEvents(ArrayList<ActionEvent> actionEvents) {
        this.actionEvents = actionEvents;
    }

    public String getConditionsOperator() {
        return conditionsOperator;
    }

    public void setConditionsOperator(String conditionsOperator) {
        this.conditionsOperator = conditionsOperator;
    }

    @Override
    public String toJson() throws JSONException {
        JSONObject targetJson = new JSONObject();
        targetJson.put(KEY_TARGET_AUDIENCE, Condition.toJson(targetAudiences))
                .put(KEY_CUSTOM_ATTRIBUTES, Condition.toJson(customAttributes))
                .put(KEY_USER_EVENTS, Condition.toJson(userEvents))
                .put(KEY_ACTION_EVENTS, ActionEvent.toJson(actionEvents))
                .put(KEY_TRIGGER, trigger.toJson())
                .put(KEY_FREQUENCY, frequency.toJson())
                .put(KEY_CONDITIONS_OPERATOR, getConditionsOperator());
        return targetJson.toString();
    }

    @Override
    public void fromJson(String jsonObject) throws JSONException {
        JSONObject targetJsonObject = new JSONObject(jsonObject);
        if (targetJsonObject.has(KEY_TARGET_AUDIENCE)) {
            setTargetAudiences(Condition.fromJson(
                    targetJsonObject.getJSONArray(KEY_TARGET_AUDIENCE)));
        }
        if (targetJsonObject.has(KEY_CUSTOM_ATTRIBUTES)) {
            setCustomAttributes(Condition.fromJson(
                    targetJsonObject.getJSONArray(KEY_CUSTOM_ATTRIBUTES)));
        }
        if (targetJsonObject.has(KEY_USER_EVENTS)) {
            setUserEvents(Condition.fromJson(
                    targetJsonObject.getJSONArray(KEY_USER_EVENTS)));
        }
        if (targetJsonObject.has(KEY_ACTION_EVENTS)) {
            setActionEvents(ActionEvent.fromJson(
                    targetJsonObject.getJSONArray(KEY_ACTION_EVENTS)));
        }
        if (targetJsonObject.has(KEY_CONDITIONS_OPERATOR)) {
            setConditionsOperator(targetJsonObject.getString(KEY_CONDITIONS_OPERATOR));
        }
        if (targetJsonObject.has(KEY_TRIGGER)) {
            setTrigger(Trigger.getFromJson(targetJsonObject.getJSONObject(KEY_TRIGGER).toString()));
        }
        if (targetJsonObject.has(KEY_FREQUENCY)) {
            setFrequency(Frequency.getFromJson(targetJsonObject.getJSONObject(KEY_FREQUENCY).toString()));
        }
    }

    public Trigger getTrigger() {
        return trigger;
    }

    public void setTrigger(Trigger trigger) {
        this.trigger = trigger;
    }

    public Frequency getFrequency() {
        return frequency;
    }

    public void setFrequency(Frequency frequency) {
        this.frequency = frequency;
    }
}
