package com.instabug.survey.common;

import android.app.Activity;
import android.content.Intent;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.library.PresentationManager;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.survey.Constants;
import com.instabug.survey.SurveysManager;
import com.instabug.survey.announcements.cache.AnnouncementCacheManager;
import com.instabug.survey.announcements.models.Announcement;
import com.instabug.survey.announcements.ui.activity.AnnouncementActivity;
import com.instabug.survey.cache.SurveysCacheManager;
import com.instabug.survey.models.Survey;
import com.instabug.survey.ui.SurveyActivity;
import com.instabug.survey.utils.SurveysUtils;

/**
 * Created by Barakat on 26/12/2018
 * <p>
 * A manager that controls all showing views runnable by SurveyManager and AnnouncementManager to PresentationManager
 */

public class AutoShowingManager {

    private static AutoShowingManager instance;
    private boolean isAnnouncementShown;
    private boolean surveyShown;
    @Nullable
    Runnable showAnnnouncementRunnable;
    @Nullable
    Runnable showSurveyRunnable;

    public synchronized static AutoShowingManager getInstance() {
        if (instance == null) {
            instance = new AutoShowingManager();
        }
        return instance;
    }

    public void showAnnouncement(final Announcement announcementToShow) {
        showAnnnouncementRunnable = new Runnable() {
            @Override
            public void run() {
                // we need to double check before showing the announcement as it might be just shown
                // and answered while this was in the queue.
                if (!announcementToShow.shouldShow()) {
                    InstabugSDKLogger.w(Constants.LOG_TAG,
                            "this announcement " + announcementToShow.getId()
                                    + " is answered and outdated");
                    return;
                }
                Activity targetActivity = InstabugInternalTrackingDelegate.getInstance()
                        .getTargetActivity();
                if (targetActivity != null && SurveysManager.getInstance() != null) {
                    SurveysManager.getInstance().unregisterSurveysTriggerEvents();
                    SurveysUtils.executeRunnableBeforeShowingSurvey();
                    announcementToShow.addShowEvent();
                    setAnnouncementShown(true);
                    Intent intent = new Intent(targetActivity, AnnouncementActivity.class);
                    intent.putExtra(AnnouncementActivity.KEY_ANNOUNCEMENT, announcementToShow);
                    targetActivity.startActivity(intent);
                }
            }
        };
        PresentationManager.getInstance().show(showAnnnouncementRunnable);
    }

    public void showSurvey(@NonNull final Survey survey) {

        showSurveyRunnable = new Runnable() {
            @Override
            public synchronized void run() {
                PoolProvider.postIOTaskWithCheck(new Runnable() {
                    @Override
                    public void run() {
                        // we need to double check before showing the survey as it might be just shown
                        // and answered while this was in the queue.
                        if (!shouldShowAgain(survey.getId())) {
                            InstabugSDKLogger.w(AutoShowingManager.class.getSimpleName(),
                                    "this survey " + survey.getId()
                                            + " is answered and outdated");
                            return;
                        }
                        Activity targetActivity = InstabugInternalTrackingDelegate.getInstance()
                                .getTargetActivity();
                        if (targetActivity != null && SurveysManager.getInstance() != null) {
                            SurveysManager.getInstance().unregisterSurveysTriggerEvents();
                            SurveysUtils.executeRunnableBeforeShowingSurvey();
                            //Remove offline surveys's answers
                            if (survey.isOptInSurvey() && survey.getSurveyEvents() != null
                                    && survey.getSurveyEvents().size() > 0
                                    && !survey.isLastEventDismiss()) {
                                survey.clearAnswers();
                            }
                            setSurveyShown(true);
                            survey.addShowEvent();
                            Intent intent = new Intent(targetActivity, SurveyActivity.class);
                            intent.putExtra(SurveyActivity.KEY_SURVEY, survey);
                            targetActivity.startActivity(intent);
                            targetActivity.overridePendingTransition(0, 0); //0 for no animation

                        }
                    }
                });
            }
        };
        PresentationManager.getInstance().show(showSurveyRunnable);
    }


    /**
     * A method to get survey and return true if survey found and answered in the cache.
     *
     * @param surveyId to fetch survey object from cache
     * @return true if found and answered, false otherwise.
     */
    private boolean isAnsweredSurvey(long surveyId) {
        Survey survey = SurveysCacheManager.getSurveyById(surveyId);
        return survey != null && !survey.isOptInSurvey() && (survey.isAnswered() || survey.getDismissedAt() != 0);
    }

    /**
     * A method to get survey and return true if survey found and shouldShowAgain.
     *
     * @param surveyId to fetch survey object from cache
     * @return true if found and dismissed, false otherwise.
     */
    private boolean shouldShowAgain(long surveyId) {
        Survey survey = SurveysCacheManager.getSurveyById(surveyId);
        return survey != null && survey.shouldShow();
    }

    @WorkerThread
    private boolean isAnsweredAnnouncement(long announcementId) {
        Announcement announcement = AnnouncementCacheManager.getAnnouncement(announcementId);
        return announcement != null && (announcement.isAnswered() || announcement.getDismissedAt() != 0);
    }

    boolean isAnnouncementShown() {
        return isAnnouncementShown;
    }

    public void setSurveyShown(boolean surveyShown) {
        this.surveyShown = surveyShown;
    }

    private boolean isSurveyShown() {
        return surveyShown;
    }

    public void setAnnouncementShown(boolean announcementShown) {
        this.isAnnouncementShown = announcementShown;
    }

    public void release() {
        showAnnnouncementRunnable = showSurveyRunnable = null;
    }
}
