package com.instabug.survey.announcements.network;

import android.content.Context;

import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Endpoints;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.survey.Constants;
import com.instabug.survey.announcements.models.Announcement;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by Barakat on 24/12/2018
 */
public class AnnouncementsService {

    public static final String CURRENT_VERSION = "2";
    public static final String KEY_LOCALE = "locale";
    private static AnnouncementsService INSTANCE;
    private final NetworkManager networkManager;

    private AnnouncementsService() {
        networkManager = new NetworkManager();
    }

    /**
     * Returns the current singleton instance of this class.
     *
     * @return singleton instance of MessagingService
     */
    public static AnnouncementsService getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new AnnouncementsService();
        }
        return INSTANCE;
    }

    public void fetchAnnouncements(String locale, final Request.Callbacks<JSONObject,
            Throwable> fetchingAnnouncementsCallbacks) throws JSONException {
        InstabugSDKLogger.d(Constants.LOG_TAG, "fetching announcements");
        // create fetchAnnouncements request
        Request fetchingAnnouncementsRequest = new Request.Builder()
                .endpoint(Endpoints.GET_ANNOUNCEMENTS)
                .method(RequestMethod.GET)
                .addParameter(new RequestParameter<>(KEY_LOCALE, locale))
                .addHeader(new RequestParameter<>("Accept",
                        "application/vnd.instabug.v" + CURRENT_VERSION))
                .addHeader(new RequestParameter<>("version", CURRENT_VERSION))
                .build();


        // do request with default connection timeout.
        networkManager.doRequest(IBGNetworkWorker.ANNOUNCEMENTS, RequestType.NORMAL, fetchingAnnouncementsRequest, new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(Constants.LOG_TAG,"fetchingAnnouncementsRequest succeeded, Response code: " + requestResponse.getResponseCode());
                InstabugSDKLogger.v(Constants.LOG_TAG, "Response: " + requestResponse);
                if (requestResponse.getResponseCode() == RequestResponse.HttpStatusCode._2xx.OK) {
                    try {
                        if (requestResponse.getResponseBody() != null) {
                            fetchingAnnouncementsCallbacks.onSucceeded(new JSONObject((String) requestResponse.getResponseBody()));
                        } else {
                            fetchingAnnouncementsCallbacks.onSucceeded(new JSONObject());
                        }
                    } catch (JSONException e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "submittingAnnouncementRequest got JSONException: " + e.getMessage(), e);
                        fetchingAnnouncementsCallbacks.onFailed(e);
                    }
                } else {
                    fetchingAnnouncementsCallbacks.onFailed(new Throwable("Fetching Announcements got error with response code:" + requestResponse.getResponseCode()));
                }
            }

            @Override
            public void onFailed(Throwable error) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "fetchingAnnouncementsRequest got error: " + error.getMessage(), error);
                fetchingAnnouncementsCallbacks.onFailed(error);
            }
        });
    }

    public void submittingAnnouncement(final Context context, final Announcement announcement, final Request
            .Callbacks<Boolean, Throwable> submittingAnnouncementCallbacks) throws JSONException {
        InstabugSDKLogger.v(Constants.LOG_TAG, "submitting announcement");

        // create submittingAnnouncement Request
        Request.Builder requestBuilder = new Request.Builder()
                .method(RequestMethod.POST)
                .endpoint(Endpoints.SUBMIT_ANNOUNCEMENT.replaceAll(":announcement_id", String.valueOf(announcement.getId())));

        AnnouncementSubmittingUtils.addParamsToSubmittingAnnouncementRequest(requestBuilder, DeviceStateProvider.getAppVersion(context), announcement);
        // do request with default connection timeout.
        networkManager.doRequestOnSameThread(RequestType.NORMAL, requestBuilder.build(), new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "submittingAnnouncementRequest succeeded, Response code: " + requestResponse.getResponseCode());
                InstabugSDKLogger.v(Constants.LOG_TAG, "submittingAnnouncementRequest Response body: " + (requestResponse.getResponseBody() != null ? requestResponse.getResponseBody() : "body is null"));
                if (requestResponse.getResponseCode() == RequestResponse.HttpStatusCode._2xx.OK) {
                    submittingAnnouncementCallbacks.onSucceeded(true);
                } else {
                    submittingAnnouncementCallbacks.onSucceeded(false);
                    submittingAnnouncementCallbacks
                            .onFailed(new Throwable("submittingAnnouncementRequest got error with response code:" + requestResponse.getResponseCode()));
                }
            }

            @Override
            public void onFailed(Throwable throwable) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "submittingAnnouncementRequest got error: " + throwable.getMessage(), throwable);
                submittingAnnouncementCallbacks.onFailed(throwable);
            }
        });
    }
}
