package com.instabug.survey.announcements.cache;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_ANNOUNCEMENT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_ANNOUNCE_CURRENT_LOCALE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_ANNOUNCE_EVENTS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_ANNOUNCE_IS_LOCALIZED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_ANNOUNCE_LOCALES;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_ANNOUNCE_STATE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_ANSWERED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_ASSETS_STATUS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_CONDITIONS_OPERATOR;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_DISMISSED_AT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_EVENT_INDEX;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_IS_CANCELLED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_IS_SHOWN;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_PAUSED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_SESSIONS_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_SHOULD_SHOW_AGAIN;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_SHOWN_AT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_TARGET;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_TITLE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.COLUMN_TYPE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AnnouncementEntry.TABLE_NAME;
import static com.instabug.survey.announcements.models.Announcement.AnnouncementAssetsStatus.NOT_AVAILABLE;

import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;

import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.survey.Constants;
import com.instabug.survey.announcements.models.Announcement;
import com.instabug.survey.announcements.models.AnnouncementItem;
import com.instabug.survey.common.models.ActionEvent;
import com.instabug.survey.common.models.SyncingStatus;
import com.instabug.survey.common.models.Target;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by Barakat on 23/12/2018
 */
public class AnnouncementsDBHelper {
    /**
     * Inserting a announcement in the database
     *
     * @param announcement is the Announcement object
     * @return the row ID of the newly inserted row, or -1 if an error occurred
     */
    @WorkerThread
    public static synchronized long insert(Announcement announcement) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();

            // Create a new map of values, where column names are the keys
            ContentValues values = getContentValues(announcement);

            // Insert the new row, returning the primary key value of the new row
            long rowId = db.insertWithOnConflict(TABLE_NAME, null, values);
            if (rowId == -1) {
                update(announcement);
            }
            db.setTransactionSuccessful();
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    "announcement id: " + announcement.getId() + " has been added to DB");
            return rowId;
        } catch (Exception | OutOfMemoryError e) {
            IBGDiagnostics.reportNonFatal(e, "announcement insertion failed due to " + e.getMessage());
            return -1;
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    private static ContentValues getContentValues(Announcement announcement) throws JSONException {
        ContentValues values = new ContentValues();
        values.put(COLUMN_ID, announcement.getId());
        if (announcement.getTitle() != null) {
            values.put(COLUMN_TITLE, announcement.getTitle());
        }
        values.put(COLUMN_TYPE, announcement.getType());
        values.put(COLUMN_CONDITIONS_OPERATOR, announcement.getConditionsOperator());
        values.put(COLUMN_ANSWERED, announcement.isAnswered() ? 1 : 0);
        values.put(COLUMN_DISMISSED_AT, announcement.getDismissedAt());
        values.put(COLUMN_SHOWN_AT, announcement.getShownAt());
        values.put(COLUMN_IS_CANCELLED, announcement.isCancelled() ? 1 : 0);
        values.put(COLUMN_EVENT_INDEX, announcement.getEventIndex());
        values.put(COLUMN_SHOULD_SHOW_AGAIN, announcement.shouldShow() ? 1 : 0);
        values.put(COLUMN_PAUSED, announcement.isPaused() ? 1 : 0);
        values.put(COLUMN_SESSIONS_COUNT, announcement.getSessionCounter());
        values.put(COLUMN_ANNOUNCEMENT, AnnouncementItem.toJson(announcement.getAnnouncementItems()).toString());
        values.put(COLUMN_TARGET, Target.toJson(announcement.getTarget()).toString());
        values.put(COLUMN_ANNOUNCE_EVENTS, ActionEvent.toJson(announcement.getAnnouncementEvents()).toString());
        values.put(COLUMN_ANNOUNCE_STATE, announcement.getAnnouncementState().toString());
        values.put(COLUMN_ASSETS_STATUS, announcement.getAssetsStatus());
        values.put(COLUMN_IS_SHOWN, announcement.isAlreadyShown() ? 1 : 0);
        values.put(COLUMN_ANNOUNCE_IS_LOCALIZED, announcement.getLocalization().isLocalized());
        values.put(COLUMN_ANNOUNCE_LOCALES, new JSONArray(announcement.getLocalization().getLocales()).toString());
        if (announcement.getLocalization() != null && announcement.getLocalization().getCurrentLocale() != null)
            values.put(COLUMN_ANNOUNCE_CURRENT_LOCALE, announcement.getLocalization().getCurrentLocale());
        return values;
    }

    @WorkerThread
    public static synchronized long update(Announcement announcement) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();
            long row = updateSingleAnnouncement(db, announcement);
            db.setTransactionSuccessful();
            return row;
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @WorkerThread
    synchronized static void updateBulk(List<Announcement> announcements) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();
            for (Announcement announcement : announcements) {
                updateSingleAnnouncement(db, announcement);
            }
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * This method to update Announcement but note that this method *DOESN'T* open database not begin/close any transactions,
     * the caller method must handle that.
     *
     * @param db           an opened SQLiteDatabaseWrapper
     * @param announcement survey to be updated
     * @return row id greater than zero if the survey is updated successfully.
     */
    @WorkerThread
    private static synchronized long updateSingleAnnouncement(@NonNull SQLiteDatabaseWrapper db, Announcement announcement) {

        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{String.valueOf(announcement.getId())};
        try {
            // Create a new map of values, where column names are the keys
            ContentValues values = getContentValues(announcement);

            // Insert the new row, returning the primary key value of the new row
            long rowId = db.update(TABLE_NAME, values, whereClause, whereArgs);
            InstabugSDKLogger.d(Constants.LOG_TAG, "announcement id: " + announcement.getId() + " has been updated");
            return rowId;
        } catch (Exception | OutOfMemoryError e) {
            IBGDiagnostics.reportNonFatal(e, "announcement updating failed due to " + e.getMessage());
            return -1;
        }
    }

    /**
     * Retrieve all Announcements
     *
     * @return a list of Announcements {@link Announcement}
     */
    @WorkerThread
    public static List<Announcement> retrieve() {

        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        Cursor cursor = null;
        try {
            cursor = database.query(
                    // Name of the table to read from
                    TABLE_NAME,
                    // String array of the columns which are supposed to be read
                    null,
                    // The selection argument which specifies which row is read. // ? symbols are
                    // parameters.
                    null,
                    // The actual parameters values for the selection as a String array. // ? above
                    // take the value
                    // from here
                    null,
                    // GroupBy clause. Specify a column name to group similar values // in that
                    // column together.
                    null,
                    // Having clause. When using the GroupBy clause this allows you to // specify
                    // which groups to
                    // include.
                    null,
                    // OrderBy clause. Specify a column name here to order the results
                    // according to that column. Optionally append ASC or DESC to specify // an
                    // ascending or
                    // descending order.
                    null);
            if (cursor == null) return new ArrayList<>();

            // To increase performance first get the index of each column in the cursor
            final int idIndex = cursor.getColumnIndex(COLUMN_ID);
            final int typeIndex = cursor.getColumnIndex(COLUMN_TYPE);
            final int titleIndex = cursor.getColumnIndex(COLUMN_TITLE);
            final int conditionsOperatorIndex = cursor.getColumnIndex(COLUMN_CONDITIONS_OPERATOR);
            final int answeredIndex = cursor.getColumnIndex(COLUMN_ANSWERED);
            final int dismissedAtIndex = cursor.getColumnIndex(COLUMN_DISMISSED_AT);
            final int shownAtIndex = cursor.getColumnIndex(COLUMN_SHOWN_AT);
            final int isCanceledIndex = cursor.getColumnIndex(COLUMN_IS_CANCELLED);
            final int eventIndexIndex = cursor.getColumnIndex(COLUMN_EVENT_INDEX);
            final int isAlreadyShownIndex = cursor.getColumnIndex(COLUMN_IS_SHOWN);
            final int pausedIndex = cursor.getColumnIndex(COLUMN_PAUSED);
            final int targetIndex = cursor.getColumnIndex(COLUMN_TARGET);
            final int announcementItemsIndex = cursor.getColumnIndex(COLUMN_ANNOUNCEMENT);
            final int assetsStatusIndex = cursor.getColumnIndex(COLUMN_ASSETS_STATUS);
            final int announceLocalesIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_LOCALES);
            final int isLocalizedIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_IS_LOCALIZED);
            final int announceCurrentLocaleIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_CURRENT_LOCALE);
            final int announcementStateIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_STATE);


            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst()) {
                cursor.close();
                return new ArrayList<>();
            }

            List<Announcement> announcements = new ArrayList<>();
            do {
                // Read the values of a row in the table using the indexes acquired above
                final long id = cursor.getLong(idIndex);
                final String title = cursor.getString(titleIndex);
                final int type = cursor.getInt(typeIndex);
                final String conditionsOperator = cursor.getString(conditionsOperatorIndex);
                final int answered = cursor.getInt(answeredIndex);
                final int dismissedAt = cursor.getInt(dismissedAtIndex);
                final int shownAt = cursor.getInt(shownAtIndex);
                final int isCanceled = cursor.getInt(isCanceledIndex);
                final int eventIndex = cursor.getInt(eventIndexIndex);
                final int isShown = cursor.getInt(isAlreadyShownIndex);
                final int paused = cursor.getInt(pausedIndex);
                final String target = cursor.getString(targetIndex);
                final String announcementItems = cursor.getString(announcementItemsIndex);
                final int assetsStatus = cursor.getInt(assetsStatusIndex);
                final String announceLocales = cursor.getString(announceLocalesIndex);
                final int announceIsLocalized = cursor.getInt(isLocalizedIndex);
                final String announceCurrentLocale = cursor.getString(announceCurrentLocaleIndex);
                String announcementState = cursor.getString(announcementStateIndex);

                Announcement announcement = new Announcement();
                announcement.setId(id);
                announcement.setType(type);
                announcement.setTitle(title);
                announcement.setConditionsOperator(conditionsOperator);
                announcement.setAnswered(answered == 1);
                announcement.setDismissedAt(dismissedAt);
                announcement.setShowAt(shownAt);
                announcement.setCancelled(isCanceled == 1);
                announcement.setEventIndex(eventIndex);
                announcement.setAlreadyShown(isShown == 1);
                announcement.setPaused(paused == 1);
                announcement.setAnnouncementItems(AnnouncementItem.fromJson(new JSONArray(announcementItems)));
                announcement.setAssetsStatus(assetsStatus);
                announcement.getLocalization().setLocaleFromJson(new JSONArray(announceLocales));
                announcement.getLocalization().setCurrentLocale(announceCurrentLocale);
                announcement.getLocalization().setLocalized(announceIsLocalized == 1);
                announcement.setAnnouncementState(SyncingStatus.valueOf(announcementState));

                Target targetObj = new Target();
                targetObj.fromJson(target);
                announcement.setTarget(targetObj);
                announcements.add(announcement);
            } while (cursor.moveToNext());
            // Read the values of a row in the table using the indexes acquired above
            InstabugSDKLogger.d(Constants.LOG_TAG, announcements.size() + " announcements have been retrieved from DB");
            return announcements;
        } catch (Exception | OutOfMemoryError e) {
            IBGDiagnostics.reportNonFatal(e, " Retrieving announcements failed: " + e.getMessage());
            return new ArrayList<>();
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            if (cursor != null) {
                cursor.close();
            }
            // close the database
            database.close();
        }
    }


    /**
     * Retrieve Announcement of a certain ID
     *
     * @return an Announcement {@link Announcement}
     */
    @WorkerThread
    @Nullable
    public static Announcement retrieveById(long announcementId) {
        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();

        final Cursor cursor = database.query(
                // Name of the table to read from
                TABLE_NAME,
                // String array of the columns which are supposed to be read
                null,
                // The selection argument which specifies which row is read. // ? symbols are
                // parameters.
                COLUMN_ID + "=? ",
                // The actual parameters values for the selection as a String array. // ? above
                // take the value
                // from here
                new String[]{String.valueOf(announcementId)},
                // GroupBy clause. Specify a column name to group similar values // in that
                // column together.
                null,
                // Having clause. When using the GroupBy clause this allows you to // specify
                // which groups to
                // include.
                null,
                // OrderBy clause. Specify a column name here to order the results
                // according to that column. Optionally append ASC or DESC to specify // an
                // ascending or
                // descending order.
                null);
        if (cursor == null) return null;

        // To increase performance first get the index of each column in the cursor
        final int idIndex = cursor.getColumnIndex(COLUMN_ID);
        final int typeIndex = cursor.getColumnIndex(COLUMN_TYPE);
        final int titleIndex = cursor.getColumnIndex(COLUMN_TITLE);
        final int conditionsOperatorIndex = cursor.getColumnIndex(COLUMN_CONDITIONS_OPERATOR);
        final int answeredIndex = cursor.getColumnIndex(COLUMN_ANSWERED);
        final int dismissedAtIndex = cursor.getColumnIndex(COLUMN_DISMISSED_AT);
        final int shownAtIndex = cursor.getColumnIndex(COLUMN_SHOWN_AT);
        final int isCanceledIndex = cursor.getColumnIndex(COLUMN_IS_CANCELLED);
        final int eventIndexIndex = cursor.getColumnIndex(COLUMN_EVENT_INDEX);
        final int isAlreadyShownIndex = cursor.getColumnIndex(COLUMN_IS_SHOWN);
        final int pausedIndex = cursor.getColumnIndex(COLUMN_PAUSED);
        final int targetIndex = cursor.getColumnIndex(COLUMN_TARGET);
        final int announcementItemsIndex = cursor.getColumnIndex(COLUMN_ANNOUNCEMENT);
        final int assetsStatusIndex = cursor.getColumnIndex(COLUMN_ASSETS_STATUS);
        final int announcLocalesIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_LOCALES);
        final int isLocalizedIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_IS_LOCALIZED);
        final int announcCurrentLocaleIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_CURRENT_LOCALE);


        try {
            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst()) {
                cursor.close();
                return null;
            }

            // Read the values of a row in the table using the indexes acquired above
            final long id = cursor.getLong(idIndex);
            final String title = cursor.getString(titleIndex);
            final int type = cursor.getInt(typeIndex);
            final String conditionsOperator = cursor.getString(conditionsOperatorIndex);
            final int answered = cursor.getInt(answeredIndex);
            final int dismissedAt = cursor.getInt(dismissedAtIndex);
            final int shownAt = cursor.getInt(shownAtIndex);
            final int isCanceled = cursor.getInt(isCanceledIndex);
            final int eventIndex = cursor.getInt(eventIndexIndex);
            final int isShown = cursor.getInt(isAlreadyShownIndex);
            final int paused = cursor.getInt(pausedIndex);
            final String target = cursor.getString(targetIndex);
            final String announcementItems = cursor.getString(announcementItemsIndex);
            final int assetsStatus = cursor.getInt(assetsStatusIndex);
            final String announceLocales = cursor.getString(announcLocalesIndex);
            final int announceIsLocalized = cursor.getInt(isLocalizedIndex);
            final String announceCurrentLocale = cursor.getString(announcCurrentLocaleIndex);

            Announcement announcement = new Announcement();
            announcement.setId(id);
            announcement.setType(type);
            announcement.setTitle(title);
            announcement.setConditionsOperator(conditionsOperator);
            announcement.setAnswered(answered == 1);
            announcement.setDismissedAt(dismissedAt);
            announcement.setShowAt(shownAt);
            announcement.setCancelled(isCanceled == 1);
            announcement.setEventIndex(eventIndex);
            announcement.setAlreadyShown(isShown == 1);
            announcement.setPaused(paused == 1);
            announcement.setAnnouncementItems(AnnouncementItem.fromJson(new JSONArray(announcementItems)));
            announcement.setAssetsStatus(assetsStatus);
            announcement.getLocalization().setLocaleFromJson(new JSONArray(announceLocales));
            announcement.getLocalization().setCurrentLocale(announceCurrentLocale);
            announcement.getLocalization().setLocalized(announceIsLocalized == 1);

            Target targetObj = new Target();
            targetObj.fromJson(target);
            announcement.setTarget(targetObj);
            return announcement;
        } catch (Exception | OutOfMemoryError e) {
            IBGDiagnostics.reportNonFatal(e, "announcement conversion failed due to " + e.getMessage());
            return null;
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            cursor.close();
            // close the database
            database.close();
        }
    }


    /**
     * Retrieve ReadyToBeSendAnnouncements
     *
     * @return a list of Announcement {@link Announcement}
     */
    public static List<Announcement> retrieveReadyToBeSend() {

        SQLiteDatabaseWrapper database = null;
        Cursor cursor = null;
        try {
            // When reading data one should always just get a readable database.
            database = DatabaseManager.getInstance().openDatabase();
            cursor = database.query(
                    // Name of the table to read from
                    TABLE_NAME,
                    // String array of the columns which are supposed to be read
                    null,
                    // The selection argument which specifies which row is read. // ? symbols are
                    // parameters.
                    COLUMN_ANNOUNCE_STATE + "=? ",
                    // The actual parameters values for the selection as a String array. // ? above
                    // take the value
                    // from here
                    new String[]{SyncingStatus.READY_TO_SEND.toString()},
                    // GroupBy clause. Specify a column name to group similar values // in that
                    // column together.
                    null,
                    // Having clause. When using the GroupBy clause this allows you to // specify
                    // which groups to
                    // include.
                    null,
                    // OrderBy clause. Specify a column name here to order the results
                    // according to that column. Optionally append ASC or DESC to specify // an
                    // ascending or
                    // descending order.
                    null);
            if (cursor == null) return new ArrayList<>();

            // To increase performance first get the index of each column in the cursor
            final int idIndex = cursor.getColumnIndex(COLUMN_ID);
            final int typeIndex = cursor.getColumnIndex(COLUMN_TYPE);
            final int titleIndex = cursor.getColumnIndex(COLUMN_TITLE);
            final int conditionsOperatorIndex = cursor.getColumnIndex(COLUMN_CONDITIONS_OPERATOR);
            final int answeredIndex = cursor.getColumnIndex(COLUMN_ANSWERED);
            final int dismissedAtIndex = cursor.getColumnIndex(COLUMN_DISMISSED_AT);
            final int shownAtIndex = cursor.getColumnIndex(COLUMN_SHOWN_AT);
            final int isCanceledIndex = cursor.getColumnIndex(COLUMN_IS_CANCELLED);
            final int eventIndexIndex = cursor.getColumnIndex(COLUMN_EVENT_INDEX);
            final int isAlreadyShownIndex = cursor.getColumnIndex(COLUMN_IS_SHOWN);
            final int pausedIndex = cursor.getColumnIndex(COLUMN_PAUSED);
            final int targetIndex = cursor.getColumnIndex(COLUMN_TARGET);
            final int announcementItemsIndex = cursor.getColumnIndex(COLUMN_ANNOUNCEMENT);
            final int assetsStatusIndex = cursor.getColumnIndex(COLUMN_ASSETS_STATUS);
            final int announceLocalesIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_LOCALES);
            final int isLocalizedIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_IS_LOCALIZED);
            final int announceCurrentLocaleIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_CURRENT_LOCALE);
            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst()) {
                cursor.close();
                return new ArrayList<>();
            }

            List<Announcement> announcements = new ArrayList<>();
            do {
                // Read the values of a row in the table using the indexes acquired above
                final long id = cursor.getLong(idIndex);
                final String title = cursor.getString(titleIndex);
                final int type = cursor.getInt(typeIndex);
                final String conditionsOperator = cursor.getString(conditionsOperatorIndex);
                final int answered = cursor.getInt(answeredIndex);
                final int dismissedAt = cursor.getInt(dismissedAtIndex);
                final int shownAt = cursor.getInt(shownAtIndex);
                final int isCanceled = cursor.getInt(isCanceledIndex);
                final int eventIndex = cursor.getInt(eventIndexIndex);
                final int isShown = cursor.getInt(isAlreadyShownIndex);
                final int paused = cursor.getInt(pausedIndex);
                final String target = cursor.getString(targetIndex);
                final String announcementItems = cursor.getString(announcementItemsIndex);
                final int assetsStatus = cursor.getInt(assetsStatusIndex);
                final String announceLocales = cursor.getString(announceLocalesIndex);
                final int announceIsLocalized = cursor.getInt(isLocalizedIndex);
                final String announceCurrentLocale = cursor.getString(announceCurrentLocaleIndex);

                Announcement announcement = new Announcement();
                announcement.setId(id);
                announcement.setType(type);
                announcement.setTitle(title);
                announcement.setConditionsOperator(conditionsOperator);
                announcement.setAnswered(answered == 1);
                announcement.setDismissedAt(dismissedAt);
                announcement.setShowAt(shownAt);
                announcement.setCancelled(isCanceled == 1);
                announcement.setEventIndex(eventIndex);
                announcement.setAlreadyShown(isShown == 1);
                announcement.setPaused(paused == 1);
                announcement.setAnnouncementItems(AnnouncementItem.fromJson(new JSONArray(announcementItems)));
                announcement.setAssetsStatus(assetsStatus);
                announcement.getLocalization().setLocaleFromJson(new JSONArray(announceLocales));
                announcement.getLocalization().setCurrentLocale(announceCurrentLocale);
                announcement.getLocalization().setLocalized(announceIsLocalized == 1);

                Target targetObj = new Target();
                targetObj.fromJson(target);
                announcement.setTarget(targetObj);
                announcements.add(announcement);
            } while (cursor.moveToNext());
            // Read the values of a row in the table using the indexes acquired above
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    announcements.size() + " announcements have been retrieved from DB");
            return announcements;
        } catch (Exception | OutOfMemoryError e) {
            IBGDiagnostics.reportNonFatal(e, "announcement conversion failed due to " + e.getMessage());
            return new ArrayList<>();
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            if (cursor != null) {
                cursor.close();
            }
            // close the database
            if (database != null) {
                database.close();
            }
        }
    }

    /**
     * Retrieve ShouldShowAgainAnnouncements
     *
     * @return a list of Announcements {@link Announcement}
     */
    public static List<Announcement> retrieveShouldShowAgain() {


        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        final Cursor cursor = database.query(
                // Name of the table to read from
                TABLE_NAME,
                // String array of the columns which are supposed to be read
                null,
                // The selection argument which specifies which row is read. // ? symbols are
                // parameters.
                COLUMN_SHOULD_SHOW_AGAIN + "=? ",
                // The actual parameters values for the selection as a String array. // ? above
                // take the value
                // from here
                new String[]{String.valueOf(1)},
                // GroupBy clause. Specify a column name to group similar values // in that
                // column together.
                null,
                // Having clause. When using the GroupBy clause this allows you to // specify
                // which groups to
                // include.
                null,
                // OrderBy clause. Specify a column name here to order the results
                // according to that column. Optionally append ASC or DESC to specify // an
                // ascending or
                // descending order.
                COLUMN_TYPE + " DESC");
        if (cursor == null) return new ArrayList<>();

        // To increase performance first get the index of each column in the cursor
        final int idIndex = cursor.getColumnIndex(COLUMN_ID);
        final int typeIndex = cursor.getColumnIndex(COLUMN_TYPE);
        final int titleIndex = cursor.getColumnIndex(COLUMN_TITLE);
        final int conditionsOperatorIndex = cursor.getColumnIndex(COLUMN_CONDITIONS_OPERATOR);
        final int answeredIndex = cursor.getColumnIndex(COLUMN_ANSWERED);
        final int dismissedAtIndex = cursor.getColumnIndex(COLUMN_DISMISSED_AT);
        final int shownAtIndex = cursor.getColumnIndex(COLUMN_SHOWN_AT);
        final int isCanceledIndex = cursor.getColumnIndex(COLUMN_IS_CANCELLED);
        final int eventIndexIndex = cursor.getColumnIndex(COLUMN_EVENT_INDEX);
        final int isAlreadyShownIndex = cursor.getColumnIndex(COLUMN_IS_SHOWN);
        final int pausedIndex = cursor.getColumnIndex(COLUMN_PAUSED);
        final int targetAudiencesIndex = cursor.getColumnIndex(COLUMN_TARGET);
        final int announcementItemsIndex = cursor.getColumnIndex(COLUMN_ANNOUNCEMENT);
        final int assetsStatusIndex = cursor.getColumnIndex(COLUMN_ASSETS_STATUS);
        final int announceLocalesIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_LOCALES);
        final int isLocalizedIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_IS_LOCALIZED);
        final int announceCurrentLocaleIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_CURRENT_LOCALE);

        try {
            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst()) {
                cursor.close();
                return new ArrayList<>();
            }

            List<Announcement> announcements = new ArrayList<>();
            do {
                // Read the values of a row in the table using the indexes acquired above
                final long id = cursor.getLong(idIndex);
                final int type = cursor.getInt(typeIndex);
                final String title = cursor.getString(titleIndex);
                final String conditionsOperator = cursor.getString(conditionsOperatorIndex);
                final int answered = cursor.getInt(answeredIndex);
                final int dismissedAt = cursor.getInt(dismissedAtIndex);
                final int shownAt = cursor.getInt(shownAtIndex);
                final int isCanceled = cursor.getInt(isCanceledIndex);
                final int eventIndex = cursor.getInt(eventIndexIndex);
                final int isShown = cursor.getInt(isAlreadyShownIndex);
                final int paused = cursor.getInt(pausedIndex);
                final String target = cursor.getString(targetAudiencesIndex);
                final String announcementItems = cursor.getString(announcementItemsIndex);
                final int assetsStatus = cursor.getInt(assetsStatusIndex);
                final String announceLocales = cursor.getString(announceLocalesIndex);
                final int announceIsLocalized = cursor.getInt(isLocalizedIndex);
                final String announceCurrentLocale = cursor.getString(announceCurrentLocaleIndex);

                Announcement announcement = new Announcement();
                announcement.setId(id);
                announcement.setType(type);
                announcement.setTitle(title);
                announcement.setConditionsOperator(conditionsOperator);
                announcement.setAnswered(answered == 1);
                announcement.setDismissedAt(dismissedAt);
                announcement.setShowAt(shownAt);
                announcement.setCancelled(isCanceled == 1);
                announcement.setEventIndex(eventIndex);
                announcement.setAlreadyShown(isShown == 1);
                announcement.setPaused(paused == 1);
                announcement.setAnnouncementItems(AnnouncementItem.fromJson(new JSONArray(announcementItems)));
                announcement.setAssetsStatus(assetsStatus);
                announcement.getLocalization().setLocaleFromJson(new JSONArray(announceLocales));
                announcement.getLocalization().setCurrentLocale(announceCurrentLocale);
                announcement.getLocalization().setLocalized(announceIsLocalized == 1);

                Target targetObj = new Target();
                targetObj.fromJson(target);
                announcement.setTarget(targetObj);
                announcements.add(announcement);
            } while (cursor.moveToNext());
            // Read the values of a row in the table using the indexes acquired above
            InstabugSDKLogger.d(Constants.LOG_TAG, announcements.size() + " announcements have been retrieved from DB");
            return announcements;
        } catch (Exception | OutOfMemoryError e) {
            IBGDiagnostics.reportNonFatal(e, "announcement conversion failed due to " + e.getMessage());
            return new ArrayList<>();
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            cursor.close();
            // close the database
            database.close();
        }
    }

    /**
     * Delete a specific userAttribute record
     *
     * @param announceId the key of the userAttribute
     */
    @WorkerThread
    public static synchronized void delete(String announceId) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{announceId};
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }


    /**
     * Delete all announcements stored in the database
     * i.e. deleting the whole table
     */
    @WorkerThread
    public static synchronized void deleteAll() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();
            db.delete(TABLE_NAME, null, null);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @WorkerThread
    public static List<Announcement> retrieveByType(int announcementType) {
        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        final Cursor cursor = database.query(
                // Name of the table to read from
                TABLE_NAME,
                // String array of the columns which are supposed to be read
                null,
                // The selection argument which specifies which row is read. // ? symbols are
                // parameters.
                COLUMN_TYPE + "=? ",
                // The actual parameters values for the selection as a String array. // ? above
                // take the value
                // from here
                new String[]{String.valueOf(announcementType)},
                // GroupBy clause. Specify a column name to group similar values // in that
                // column together.
                null,
                // Having clause. When using the GroupBy clause this allows you to // specify
                // which groups to
                // include.
                null,
                // OrderBy clause. Specify a column name here to order the results
                // according to that column. Optionally append ASC or DESC to specify // an
                // ascending or
                // descending order.
                null);
        if (cursor == null) return new ArrayList<>();

        // To increase performance first get the index of each column in the cursor
        final int idIndex = cursor.getColumnIndex(COLUMN_ID);
        final int typeIndex = cursor.getColumnIndex(COLUMN_TYPE);
        final int titleIndex = cursor.getColumnIndex(COLUMN_TITLE);
        final int conditionsOperatorIndex = cursor.getColumnIndex(COLUMN_CONDITIONS_OPERATOR);
        final int answeredIndex = cursor.getColumnIndex(COLUMN_ANSWERED);
        final int dismissedAtIndex = cursor.getColumnIndex(COLUMN_DISMISSED_AT);
        final int shownAtIndex = cursor.getColumnIndex(COLUMN_SHOWN_AT);
        final int isCanceledIndex = cursor.getColumnIndex(COLUMN_IS_CANCELLED);
        final int eventIndexIndex = cursor.getColumnIndex(COLUMN_EVENT_INDEX);
        final int isAlreadyShownIndex = cursor.getColumnIndex(COLUMN_IS_SHOWN);
        final int pausedIndex = cursor.getColumnIndex(COLUMN_PAUSED);
        final int targetAudiencesIndex = cursor.getColumnIndex(COLUMN_TARGET);
        final int announcementItemsIndex = cursor.getColumnIndex(COLUMN_ANNOUNCEMENT);
        final int assetsStatusIndex = cursor.getColumnIndex(COLUMN_ASSETS_STATUS);
        final int announceLocalesIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_LOCALES);
        final int isLocalizedIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_IS_LOCALIZED);
        final int announceCurrentLocaleIndex = cursor.getColumnIndex(COLUMN_ANNOUNCE_CURRENT_LOCALE);

        try {
            // If moveToFirst() returns false then cursor is empty
            if (!cursor.moveToFirst()) {
                cursor.close();
                return new ArrayList<>();
            }

            List<Announcement> announcements = new ArrayList<>();
            do {
                // Read the values of a row in the table using the indexes acquired above
                final long id = cursor.getLong(idIndex);
                final int type = cursor.getInt(typeIndex);
                final String title = cursor.getString(titleIndex);
                final String conditionsOperator = cursor.getString(conditionsOperatorIndex);
                final int answered = cursor.getInt(answeredIndex);
                final int dismissedAt = cursor.getInt(dismissedAtIndex);
                final int shownAt = cursor.getInt(shownAtIndex);
                final int isCanceled = cursor.getInt(isCanceledIndex);
                final int eventIndex = cursor.getInt(eventIndexIndex);
                final int isShown = cursor.getInt(isAlreadyShownIndex);
                final int paused = cursor.getInt(pausedIndex);
                final String target = cursor.getString(targetAudiencesIndex);
                final String announcementItems = cursor.getString(announcementItemsIndex);
                final int assetsStatus = cursor.getInt(assetsStatusIndex);
                final String announceLocales = cursor.getString(announceLocalesIndex);
                final int announceIsLocalized = cursor.getInt(isLocalizedIndex);
                final String announceCurrentLocale = cursor.getString(announceCurrentLocaleIndex);

                Announcement announcement = new Announcement();
                announcement.setId(id);
                announcement.setType(type);
                announcement.setTitle(title);
                announcement.setConditionsOperator(conditionsOperator);
                announcement.setAnswered(answered == 1);
                announcement.setDismissedAt(dismissedAt);
                announcement.setShowAt(shownAt);
                announcement.setCancelled(isCanceled == 1);
                announcement.setEventIndex(eventIndex);
                announcement.setAlreadyShown(isShown == 1);
                announcement.setPaused(paused == 1);
                announcement.setPaused(paused == 1);
                announcement.setAnnouncementItems(AnnouncementItem.fromJson(new JSONArray(announcementItems)));
                announcement.setAssetsStatus(assetsStatus);
                announcement.getLocalization().setLocaleFromJson(new JSONArray(announceLocales));
                announcement.getLocalization().setCurrentLocale(announceCurrentLocale);
                announcement.getLocalization().setLocalized(announceIsLocalized == 1);

                Target targetObj = new Target();
                targetObj.fromJson(target);
                announcement.setTarget(targetObj);
                announcements.add(announcement);
            } while (cursor.moveToNext());
            // Read the values of a row in the table using the indexes acquired above
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    announcements.size() + " announcements have been retrieved from DB");
            return announcements;
        } catch (Exception | OutOfMemoryError e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "announcement conversion failed due to " + e.getMessage(), Constants.LOG_TAG);
            return new ArrayList<>();
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            cursor.close();
            // close the database
            database.close();
        }
    }

    @WorkerThread
    public static boolean isExisting(long announcementId) {
        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        try (final Cursor cursor = database.query(
                // Name of the table to read from
                TABLE_NAME,
                // String array of the columns which are supposed to be read
                null,
                // The selection argument which specifies which row is read. // ? symbols are
                // parameters.
                COLUMN_ID + "=? ",
                // The actual parameters values for the selection as a String array. // ? above
                // take the value
                // from here
                new String[]{String.valueOf(announcementId)},
                // GroupBy clause. Specify a column name to group similar values // in that
                // column together.
                null,
                // Having clause. When using the GroupBy clause this allows you to // specify
                // which groups to
                // include.
                null,
                // OrderBy clause. Specify a column name here to order the results
                // according to that column. Optionally append ASC or DESC to specify // an
                // ascending or
                // descending order.
                null)) {

            if (cursor == null) return false;

            boolean isExist = cursor.moveToFirst();
            return isExist;
        } catch (Exception | OutOfMemoryError e) {
            IBGDiagnostics.reportNonFatal(e, "check announcement Existing failed due to " + e.getMessage());
            return false;
        } finally {
            database.close();

        }

    }

    /**
     * Inserting a announcement in the database
     *
     * @param announcement         is the Announcement object
     * @param isPublishStateChange to update announcement's statue
     * @param isLocaleChanged      to update announcement's locales
     * @return the row ID of the newly inserted row, or -1 if an error occurred
     */
    @WorkerThread
    public static synchronized long insertOrUpdatePausedOrLocale(Announcement announcement,
                                                                 boolean isPublishStateChange,
                                                                 boolean isLocaleChanged) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();

            // Create a new map of values, where column names are the keys
            ContentValues values = getContentValues(announcement);

            // Insert the new row, returning the primary key value of the new row
            long rowId = db.insertWithOnConflict(TABLE_NAME, null, values);
            if (rowId == -1) {
                if (isPublishStateChange) {
                    updateAnnouncementPausedField(db, announcement);
                }
                if (isLocaleChanged) {
                    updateLocales(db, announcement);
                }
            }
            db.setTransactionSuccessful();
            InstabugSDKLogger.d(Constants.LOG_TAG,
                    "Announcement with id: " + announcement.getId() + " has been added to DB");
            return rowId;
        } catch (Exception | OutOfMemoryError e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "announcement insertion failed due to " + e.getMessage(), Constants.LOG_TAG);
            return -1;
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @WorkerThread
    private static void updateAnnouncementPausedField(@NonNull SQLiteDatabaseWrapper db, Announcement announcement) {
        ContentValues publishedStatusValues = new ContentValues();
        publishedStatusValues.put(COLUMN_PAUSED, announcement.isPaused());
        updateField(db, announcement.getId(), publishedStatusValues);
    }

    @WorkerThread
    @VisibleForTesting
    private static void updateLocales(@NonNull SQLiteDatabaseWrapper db, Announcement announcement) throws JSONException {
        ContentValues localeContentValues = new ContentValues();
        if (announcement != null && announcement.getLocalization() != null && announcement.getLocalization().getCurrentLocale() != null)
            localeContentValues.put(COLUMN_ANNOUNCE_CURRENT_LOCALE, announcement.getLocalization().getCurrentLocale());
        if (announcement != null && announcement.getLocalization() != null && announcement.getAnnouncementItems() != null)
            localeContentValues.put(COLUMN_ANNOUNCEMENT, AnnouncementItem.toJson(announcement.getAnnouncementItems()).toString());
        if (announcement != null)
            updateField(db, announcement.getId(), localeContentValues);
    }


    /**
     * A method to update a given field with given value, based on survey id condition
     *
     * @param db             database object and it should be opened and ready for transaction
     * @param announcementId id
     * @param values         value to be updated with
     * @return row id > 0 if succeed
     */
    @WorkerThread
    private static synchronized long updateField(@NonNull SQLiteDatabaseWrapper db,
                                                 long announcementId, ContentValues values) {
        // Gets the data repository in write mode

        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{String.valueOf(announcementId)};


        // Insert the new row, returning the primary key value of the new row
        long rowId = db.update(TABLE_NAME, values, whereClause, whereArgs);
        InstabugSDKLogger.d(Constants.LOG_TAG,
                "announcement with id: " + announcementId + " has been updated in DB");
        return rowId;

    }


    @WorkerThread
    public static void resetAssetsStatus() {
        List<Announcement> retrievedAnnouncements = retrieve();
        for (Announcement announcement : retrievedAnnouncements) {
            announcement.setAssetsStatus(NOT_AVAILABLE);
            update(announcement);
        }
    }

    public static int updateAssetStatus(long announcementId, int downloadSucceed) {
        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{String.valueOf(announcementId)};
        ContentValues values = new ContentValues();
        values.put(COLUMN_ASSETS_STATUS, downloadSucceed);
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        return db.update(TABLE_NAME, values, whereClause, whereArgs);
    }
}
