package com.instabug.survey.announcements;

import android.os.Build;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.device.InstabugDeviceProperties;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.ObjectMapper;
import com.instabug.library.util.StringUtility;
import com.instabug.library.util.TimeUtils;
import com.instabug.survey.announcements.cache.AnnouncementsDBHelper;
import com.instabug.survey.announcements.models.Announcement;
import com.instabug.survey.announcements.models.AnnouncementType;
import com.instabug.survey.announcements.settings.AnnouncementsSettings;
import com.instabug.survey.common.models.Condition;
import com.instabug.survey.models.CountryInfo;
import com.instabug.survey.settings.SurveysSettings;
import com.instabug.survey.utils.TargetingCondition;
import com.instabug.survey.utils.TargetingOperator;
import com.instabug.survey.utils.ValidationUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


/**
 * Created by Barakat on 25/12/2018
 */
public class AnnouncementValidator {

    @Deprecated
    private String appVersionName; // eg. 1.0.0-beta
    private String appVersionAndCodeName; // eg. 1.0.0-beta (3)

    AnnouncementValidator(String versionName, String versionAndCodeName) {
        this.appVersionName = versionName;
        this.appVersionAndCodeName = versionAndCodeName;
    }

    @Nullable
    private static Condition getTargetVersionCondition(Announcement announcement) {
        if (announcement == null)
            log("getTargetVersionCondition(announcement: null)");
        else {
            log("getTargetVersionCondition(announcementId: " + announcement.getId() + ")");
            for (Condition condition : announcement.getTarget().getTargetAudiences()) {
                if (condition.getKey() != null
                        && condition.getKey().equals(TargetingCondition.APP_VERSION_V2)) {
                    log("condition: " + condition);
                    return condition;
                }
            }
        }
        return null;
    }

    public List<Announcement> getValidAnnouncements() {
        List<Announcement> validAnnouncements = getValidUpdateMessageAnnouncements();
        if (validAnnouncements.size() == 0) {
            validAnnouncements = getValidWhatsNewAnnouncements();
        }
        return validAnnouncements;
    }

    private List<Announcement> getValidUpdateMessageAnnouncements() {
        List<Announcement> announcements = AnnouncementsDBHelper.retrieveByType(AnnouncementType.UPDATE_MSG);
        ArrayList<Announcement> validAnnouncements = new ArrayList<>();
        if (announcements != null && announcements.size() > 0) {
            for (Announcement announcement : announcements) {
                if (checkAnnouncementValidity(announcement)) {
                    if (validateShowingRepetition(announcement)) {
                        validAnnouncements.add(announcement);
                    }
                }
            }
        }
        return validAnnouncements;
    }

    private boolean validateShowingRepetition(Announcement announcement) {
        boolean shouldShow = announcement.shouldShow();
        log("validateShowingRepetition(announcement: " + announcement + "). ShouldShow ? " + shouldShow);
        return shouldShow;
    }

    private List<Announcement> getValidWhatsNewAnnouncements() {
        List<Announcement> announcements = AnnouncementsDBHelper.retrieveByType(AnnouncementType.WHAT_IS_NEW);
        ArrayList<Announcement> validAnnouncements = new ArrayList<>();
        if (announcements != null && announcements.size() > 0) {
            for (Announcement announcement : announcements) {
                if (validateAnnouncementTargetVersion(announcement) &&
                        validateShowingRepetition(announcement) &&
                        validateAssetsAreReady(announcement)) {
                    validAnnouncements.add(announcement);
                }
            }
        }
        return validAnnouncements;
    }

    private boolean validateAssetsAreReady(Announcement announcement) {
        return announcement.getAssetsStatus() == Announcement.AnnouncementAssetsStatus.DOWNLOAD_SUCCEED;
    }

    @Nullable
    public Announcement getFirstValidAnnouncement() {
        List<Announcement> validAnnouncements = getValidAnnouncements();
        if (validAnnouncements != null && validAnnouncements.size() > 0) {
            Announcement firstAnnouncement = validAnnouncements.get(0);
            log("getFirstValidAnnouncement:" + validAnnouncements.size() + " available announcements");
            log("getFirstValidAnnouncement: " + firstAnnouncement);
            return firstAnnouncement;
        }
        log("getFirstValidAnnouncement: no valid announcements. Returning null...");
        return null;
    }

    private boolean validateAnnouncementTargetVersion(Announcement announcement) {
        if (Instabug.getApplicationContext() == null) return false;

        if (!InstabugDeviceProperties.isFirstInstall(Instabug.getApplicationContext())) {
            Condition condition = getTargetVersionCondition(announcement);
            if (condition != null) {
                return validateAppVersionV2(condition);
            }
        }
        return false;
    }

    @Nullable
    public String extractVersion(@Nullable String s) {
        if (s == null) {
            return null;
        }
        String regex = "\\d+(\\.\\d+)*";
        final Pattern pattern = Pattern.compile(regex);
        final Matcher matcher = pattern.matcher(s);
        if (matcher.find()) {
            return matcher.group(0);
        }
        return null;
    }

    @VisibleForTesting
    boolean checkAnnouncementValidity(Announcement announcement) {
        log("checkStringCondition(announcement: " + announcement + ")");
        boolean primitiveTypesValidity = checkPrimitiveTypes(announcement.getTarget().getTargetAudiences(),
                announcement.getConditionsOperator());
        log("primitiveTypesValidity: " + primitiveTypesValidity);

        boolean customAttributesValidity
                = ValidationUtils.checkCustomAttributesConditions(announcement.getTarget().getCustomAttributes(), announcement
                .getConditionsOperator());
        log("customAttributesValidity: " + customAttributesValidity);

        boolean userEventsValidity = ValidationUtils.checkUserEvents(announcement.getTarget().getUserEvents(), announcement
                .getConditionsOperator());
        log("userEventsValidity: " + userEventsValidity);

        if (announcement.getTarget().getUserEvents().size() > 0
                || announcement.getTarget().getCustomAttributes().size() > 0
                || announcement.getTarget().getTargetAudiences().size() > 0) {
            if (TargetingOperator.OR.equals(announcement.getConditionsOperator())) {
                return primitiveTypesValidity || customAttributesValidity || userEventsValidity;
            }
            return primitiveTypesValidity && customAttributesValidity && userEventsValidity;
        } else {
            return true;
        }
    }

    @VisibleForTesting
    boolean checkPrimitiveTypes(ArrayList<Condition> primitiveTypesConditions,
                                String conditionsOperator) {
        int size = primitiveTypesConditions == null ? 0 : primitiveTypesConditions.size();
        log("checkPrimitiveTypes(primitiveTypesConditions: " + size
                + ", conditionsOperator: " + conditionsOperator + ")");
        boolean conditionsResult = conditionsOperator.equals(TargetingOperator.AND);
        for (int i = 0; i < size; i++) {
            boolean primitiveTypesCondition = checkPrimitiveType(primitiveTypesConditions.get(i));
            if (i == 0) {
                conditionsResult = primitiveTypesCondition;
            } else {
                if (TargetingOperator.OR.equals(conditionsOperator)) {
                    conditionsResult |= primitiveTypesCondition;
                } else {
                    conditionsResult &= primitiveTypesCondition;
                }
            }
        }
        return conditionsResult;
    }

    @VisibleForTesting
    boolean checkPrimitiveType(Condition primitiveTypeCondition) {
        log("checkPrimitiveType(primitiveTypeCondition: " + primitiveTypeCondition + ")");
        if (primitiveTypeCondition.getKey() != null) {
            switch (primitiveTypeCondition.getKey()) {
                case TargetingCondition.APP_VERSION:
                    return validateAppVersion(primitiveTypeCondition);
                case TargetingCondition.APP_VERSION_V2:
                    return validateAppVersionV2(primitiveTypeCondition);
                case TargetingCondition.USER_EMAIL:
                    return validateUserEmail(primitiveTypeCondition);
                case TargetingCondition.SESSION_COUNT:
                    return validateSessionCount(primitiveTypeCondition, SettingsManager.getInstance().getSessionsCount());
                case TargetingCondition.COUNTRY:
                    String countryInfoJson = AnnouncementsSettings.getCountryInfo();
                    CountryInfo countryInfo = ObjectMapper.fromJson(countryInfoJson, CountryInfo.class);
                    return validateCountryCode(primitiveTypeCondition, countryInfo);
                case TargetingCondition.LAST_SEEN:
                    return checkLastSeenTimestamp(primitiveTypeCondition);
                case TargetingCondition.OS_API_LEVEL:
                    return checkOSApiLevel(primitiveTypeCondition);
                default:
                    return false;
            }
        } else {
            return false;
        }
    }

    public boolean checkOSApiLevel(Condition primitiveTypeCondition) {
        return validateOSApiLevel(primitiveTypeCondition);
    }

    private boolean validateOSApiLevel(Condition condition) {
        log("validateOSApiLevel(condition: " + condition + ")");
        log("Build.VERSION.SDK_INT: " + Build.VERSION.SDK_INT);
        if (condition.getValue() == null || condition.getOperator() == null) {
            return false;
        }
        int targetOSApiLevel = Integer.parseInt(condition.getValue());
        switch (condition.getOperator()) {
            case TargetingOperator.EQUAL:
                return Build.VERSION.SDK_INT == targetOSApiLevel;
            case TargetingOperator.NOT_EQUAL:
                return Build.VERSION.SDK_INT != targetOSApiLevel;
            case TargetingOperator.GREATER_THAN:
                return Build.VERSION.SDK_INT > targetOSApiLevel;
            case TargetingOperator.LESS_THAN:
                return Build.VERSION.SDK_INT < targetOSApiLevel;
            default:
                return false;
        }
    }

    @VisibleForTesting
    public boolean validateCountryCode(Condition primitiveTypeCondition, @Nullable CountryInfo countryInfo) {
        if (countryInfo != null && primitiveTypeCondition != null) {
            String countyCode = countryInfo.getCountryCode();
            if (TargetingOperator.EQUAL.equals(primitiveTypeCondition.getOperator()) &&
                    primitiveTypeCondition.getValue() != null) {
                return primitiveTypeCondition.getValue().equalsIgnoreCase(countyCode);
            }
            return false;
        }
        return false;

    }

    @Deprecated
    boolean validateAppVersion(Condition condition) {
        log("validateAppVersion(condition: " + condition + ")");
        String conditionVersionName = extractVersion(condition.getValue());
        String currentVersionName = extractVersion(appVersionAndCodeName);
        if (conditionVersionName != null) {
            int result;
            try {
                if (currentVersionName == null) return false;

                result = StringUtility.compareVersion(currentVersionName, conditionVersionName);
            } catch (NumberFormatException e) {
                return false;
            }

            if (condition.getOperator() == null) {
                return false;
            }
            switch (condition.getOperator()) {
                case TargetingOperator.EQUAL:
                    return result == 0;
                case TargetingOperator.NOT_EQUAL:
                    return result != 0;
                case TargetingOperator.GREATER_THAN:
                    return result == 1;
                case TargetingOperator.LESS_THAN:
                    return result == -1;
                default:
                    return false;
            }
        }
        return ValidationUtils.checkStringCondition(condition, appVersionName);
    }

    boolean validateAppVersionV2(Condition condition) {
        log("validateAppVersion(condition: " + condition + ")");
        String operator = condition.getOperator();
        // If operator is "equals" or "not_equal" then delegate the validation to string validator
        if (operator == null || condition.getValue() == null) {
            return false;
        }
        if (operator.equals(TargetingOperator.GREATER_THAN) || operator.equals(TargetingOperator.LESS_THAN)) {
            try {
                long conditionFirstSeen = Long.parseLong(condition.getValue());
                long currentFirstSeen = SurveysSettings.getFirstSeen();
                if (conditionFirstSeen == -1 || currentFirstSeen == -1) return false;
                switch (operator) {
                    case TargetingOperator.GREATER_THAN:
                        return currentFirstSeen > conditionFirstSeen;
                    case TargetingOperator.LESS_THAN:
                        return currentFirstSeen < conditionFirstSeen;
                    default:
                        return false;
                }
            } catch (NumberFormatException e) {
                return false;
            }
        }
        return ValidationUtils.checkStringCondition(condition, appVersionAndCodeName);
    }

    @VisibleForTesting
    boolean validateUserEmail(Condition condition) {
        String userEmail = InstabugCore.getIdentifiedUserEmail();
        log("validateUserEmail(condition: " + condition + "). userEmail: " + userEmail);
        return ValidationUtils.checkStringCondition(condition, userEmail);
    }

    public boolean checkLastSeenTimestamp(Condition condition) {
        if (condition.getOperator() == null || condition.getValue() == null) {
            return false;
        }
        long dayDiff = Long.parseLong(condition.getValue());
        long daysSinceLastSeen = getDifferenceDays(getLastSeenTimestamp(), TimeUtils.currentTimeMillis());
        log("checkLastSeenTimestamp(condition: " + condition + "). daysSinceLastSeen: " + daysSinceLastSeen);
        switch (condition.getOperator()) {
            case TargetingOperator.EQUAL:
                return daysSinceLastSeen == dayDiff;
            case TargetingOperator.NOT_EQUAL:
                return daysSinceLastSeen != dayDiff;
            case TargetingOperator.GREATER_THAN:
                return daysSinceLastSeen > dayDiff;
            case TargetingOperator.LESS_THAN:
                return daysSinceLastSeen < dayDiff;
            default:
                return false;
        }
    }

    private boolean validateSessionCount(Condition condition, int actualSessionCount) {
        return checkSessionCountCondition(condition, actualSessionCount);
    }


    private boolean checkSessionCountCondition(Condition condition, int actualSessionCount) {
        log("checkSessionCountCondition(condition: " + condition + "). actualSessionCount: " + actualSessionCount);
        if (condition.getValue() == null || condition.getOperator() == null) {
            return false;
        }
        int conditionalSessionCount = Integer.parseInt(condition.getValue());
        switch (condition.getOperator()) {
            case TargetingOperator.EQUAL:
                return actualSessionCount == conditionalSessionCount;
            case TargetingOperator.NOT_EQUAL:
                return actualSessionCount != conditionalSessionCount;
            case TargetingOperator.GREATER_THAN:
                return actualSessionCount > conditionalSessionCount;
            case TargetingOperator.LESS_THAN:
                return actualSessionCount < conditionalSessionCount;
            default:
                return false;
        }
    }

    private int getDifferenceDays(long time, long currentTime) {
        return (int) TimeUnit.DAYS.convert(currentTime - time, TimeUnit.MILLISECONDS);
    }

    @VisibleForTesting
    public long getLastSeenTimestamp() {
        return InstabugCore.getLastSeenTimestamp();
    }

    private static void log(String message) {
        InstabugSDKLogger.i("AnnouncementValidator", message);
    }
}
