package com.instabug.survey.utils;

import android.annotation.SuppressLint;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.storage.cache.db.userAttribute.UserAttributesDbHelper;
import com.instabug.library.logging.InstabugUserEventLogger;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.survey.Constants;
import com.instabug.survey.common.models.Condition;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;

/**
 * Created by Shannan on 30/12/2021.
 */
public class ValidationUtils {


    @VisibleForTesting
    public static boolean checkCustomAttributesConditions(ArrayList<Condition> customAttributesConditions,
                                                          String conditionsOperator) {
        int size = customAttributesConditions == null ? 0 : customAttributesConditions.size();
        InstabugSDKLogger.v(Constants.LOG_TAG, "checkCustomAttributesConditions(customAttributesConditions: " + size
                + ", conditionsOperator: " + conditionsOperator + ")");

        boolean conditionsResult = conditionsOperator.equals(TargetingOperator.AND);
        for (int i = 0; i < size; i++) {
            boolean customAttributeCondition = checkCustomAttributeCondition
                    (customAttributesConditions.get(i));
            if (i == 0) {
                conditionsResult = customAttributeCondition;
            } else {
                if (TargetingOperator.OR.equals(conditionsOperator)) {
                    conditionsResult |= customAttributeCondition;
                } else {
                    conditionsResult &= customAttributeCondition;
                }
            }
        }
        return conditionsResult;
    }

    @VisibleForTesting
    public static boolean checkCustomAttributeCondition(@NonNull Condition customAttributeCondition) {
        if (customAttributeCondition == null) {
            return true;
        }
        HashMap<String, String> userAttributes = UserAttributesDbHelper.getAll();
        return userAttributes.containsKey(customAttributeCondition
                .getKey()) && checkStringCondition(customAttributeCondition, userAttributes.
                get(customAttributeCondition.getKey()));
    }

    public static boolean checkUserEvents(ArrayList<Condition> userEventsConditions, String
            conditionsOperator) {
        boolean conditionsResult = conditionsOperator.equals(TargetingOperator.AND);
        if (userEventsConditions == null) {
            return conditionsResult;
        }
        for (int i = 0; i < userEventsConditions.size(); i++) {
            boolean userEventCondition = checkUserEvent(userEventsConditions.get(i));
            if (i == 0) {
                conditionsResult = userEventCondition;
            } else {
                switch (conditionsOperator) {
                    case TargetingOperator.OR:
                        conditionsResult |= userEventCondition;
                        break;
                    case TargetingOperator.AND:
                    default:
                        conditionsResult &= userEventCondition;
                        break;
                }
            }
        }
        return conditionsResult;
    }

    public static boolean checkUserEvent(Condition condition) {
        if (condition == null) {
            return true;
        }
        int actualEventLoggingCount = getLoggingEventCount(condition.getKey());
        InstabugSDKLogger.v(Constants.LOG_TAG, "checkUserEvent(condition: "
                + condition + ", actualEventLoggingCount: " + actualEventLoggingCount + ")");

        return checkNumberCondition(condition, actualEventLoggingCount);
    }

    @VisibleForTesting
    public static int getLoggingEventCount(@Nullable String key) {
        if (key == null) {
            return 0;
        }
        try {
            Method getLoggingEventCountMethod
                    = InstabugUserEventLogger.class.getDeclaredMethod("getLoggingEventCount",
                    String.class);
            getLoggingEventCountMethod.setAccessible(true);
            Object loggingEventCount =
                    getLoggingEventCountMethod.invoke(InstabugUserEventLogger.getInstance(), key);
            if (loggingEventCount instanceof Integer) {
                return ((int) loggingEventCount);
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, e.getMessage() + "");
        }
        return 0;
    }

    public static boolean checkNumberCondition(Condition condition, int actualValue) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "checkUserEvent(condition: " + condition + ", actualValue: " + actualValue + ")");
        if (condition == null) {
            return true;
        }
        if (condition.getValue() == null || condition.getOperator() == null) {
            return false;
        }
        try {
            int conditionValue = Integer.parseInt(condition.getValue());
            switch (condition.getOperator()) {
                case TargetingOperator.EQUAL:
                    return actualValue == conditionValue;
                case TargetingOperator.NOT_EQUAL:
                    return actualValue != conditionValue;
                case TargetingOperator.GREATER_THAN:
                    return actualValue > conditionValue;
                case TargetingOperator.LESS_THAN:
                    return actualValue < conditionValue;
                default:
            }
            return false;
        } catch (NumberFormatException exception) {
            return onParseException(condition, exception);
        }
    }


    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static boolean checkStringCondition(@Nullable Condition condition, @Nullable String actualValue) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "checkStringCondition(condition: " + condition + ", actualValue: " + actualValue + ")");
        if (condition == null) {
            return true;
        }
        if (condition.getValue() == null || actualValue == null) {
            return false;
        }
        String conditionValue = condition.getValue();
        if (condition.getOperator() != null) {
            switch (condition.getOperator()) {
                case TargetingOperator.EQUAL:
                    return actualValue.equals(conditionValue);
                case TargetingOperator.NOT_EQUAL:
                    return !actualValue.equals(conditionValue);
                case TargetingOperator.CONTAINS:
                    return actualValue.contains(conditionValue);
                case TargetingOperator.NOT_CONTAINS:
                    return !actualValue.contains(conditionValue);
                default:
                    return false;
            }
        } else {
            return false;
        }
    }

    public static boolean onParseException(Condition condition, Throwable exception) {
        InstabugCore.reportError(exception, "Failed to resolve condition: " + condition);
        InstabugSDKLogger.e(Constants.LOG_TAG, "Failed to resolve condition: " + condition, exception);
        return false;
    }
}
