package com.instabug.survey.ui.survey;

import android.os.Build;
import android.os.Bundle;
import android.view.View;
import android.view.accessibility.AccessibilityEvent;
import android.view.accessibility.AccessibilityNodeInfo;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.CallSuper;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.lifecycle.Lifecycle;

import com.instabug.library.core.ui.InstabugBaseFragment;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.LocaleHelper;
import com.instabug.library.util.OrientationUtils;
import com.instabug.survey.R;
import com.instabug.survey.models.Question;
import com.instabug.survey.models.Survey;
import com.instabug.survey.models.SurveyTypes;
import com.instabug.survey.ui.SurveyActivity;
import com.instabug.survey.ui.SurveyActivityCallback;
import com.instabug.survey.ui.ViewType;
import com.instabug.survey.ui.gestures.GesturesHandler;
import com.instabug.survey.ui.survey.rateus.RateUsAbstractFragment;
import com.instabug.survey.ui.survey.rateus.RateUsFragment;

/**
 * Created by mohamedzakaria on 5/23/17.
 */

public abstract class QuestionAbstractFragment extends InstabugBaseFragment implements View.OnClickListener,
        GesturesHandler.GestureCallback {
    protected static final String KEY_QUESTION_ARGUMENT = "question";
    protected static final String KEY_SHOULD_CHANGE_CONTAINER_HEIGHT_ARGUMENT = "should_change_container_height";
    @Nullable
    protected Question question;
    @Nullable
    protected SurveyFragmentCallbacks surveyFragmentCallbacks;
    @Nullable
    protected TextView questionTextView;
    @Nullable
    protected View shadow;
    @Nullable
    protected RelativeLayout container;
    @Nullable
    protected Survey survey;

    public void setSurveyFragmentCallbacks(@NonNull SurveyFragmentCallbacks surveyFragmentCallbacks) {
        this.surveyFragmentCallbacks = surveyFragmentCallbacks;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        if (getActivity() == null) return;

        if (getActivity() instanceof SurveyActivity)
            survey = ((SurveyActivity) getActivity()).getSurvey();
        super.onCreate(savedInstanceState);
    }

    @Override
    @CallSuper
    protected void initViews(View rootView, @Nullable Bundle savedInstanceState) {
        if (getActivity() == null) return;

        if (getActivity() instanceof SurveyActivity) {
            ((SurveyActivity) getActivity()).setBackgroundTransparent(false);
        }
        if (getContext() == null) {
            return;
        }
        shadow = findViewById(R.id.survey_shadow);
        questionTextView = rootView.findViewById(R.id.instabug_text_view_question);
        container = (RelativeLayout) findViewById(R.id.instabug_survey_dialog_container);

        if (AccessibilityUtils.isTalkbackEnabled() && container != null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN)
                container.setImportantForAccessibility(View.IMPORTANT_FOR_ACCESSIBILITY_NO);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P)
                container.setScreenReaderFocusable(false);
        }

        if (getContext() != null && !isPartial() && LocaleHelper.isRTL(getContext())) {
            rootView.setRotation(180);
        }

        if (this instanceof RateUsAbstractFragment)
            //RateUsAbstractFragment has a special case as it uses different textview fr question title.
            return;

        setupQuestionAnnouncementOrder(questionTextView);
    }

    public abstract boolean isPartial();

    @Nullable
    public abstract String getAnswer();

    protected void handleTextViewMaxLines() {
        if (getActivity() == null) return;

        TextView textView = questionTextView;
        if (textView != null && OrientationUtils.isInLandscape(getActivity())) {
            textView.setMaxLines(3);
        }
    }

    protected void handleOpenSurveyClicked(final Survey survey, boolean shouldShowKeyboard) {
        if (getActivity() != null && (getActivity() instanceof SurveyActivity)) {
            if (survey != null && survey.getQuestions() != null && survey.getQuestions().size() > 0) {
                if (survey.getType() == SurveyTypes.STORE_RATING ||
                        survey.getQuestions().get(0).getType() == Question.Type.NPS) {
                    ((SurveyActivity) getActivity()).setLayoutHeightSecondary(ViewType.PRIMARY, true);
                } else if (survey.getQuestions().get(0).getType() == Question.Type.STAR_RATE) {
                    ((SurveyActivity) getActivity()).setLayoutHeightSecondary(ViewType.PRIMARY, true);
                    for (Question question : survey.getQuestions()) {
                        if (question.getType() != Question.Type.STAR_RATE) {
                            ((SurveyActivity) getActivity()).setLayoutHeightSecondary(ViewType.SECONDARY,
                                    true);
                            break;
                        }
                    }
                } else {
                    ((SurveyActivity) getActivity()).setLayoutHeightSecondary(ViewType.SECONDARY, true);
                }
            }
            if (getActivity() != null && getActivity().getLifecycle().getCurrentState() == Lifecycle.State.RESUMED) {
                getActivity().getSupportFragmentManager().beginTransaction()
                        .setCustomAnimations(0, 0)
                        .replace(R.id.instabug_fragment_container, SurveyFragment.newInstance(survey, shouldShowKeyboard))
                        .commit();
            }
        }
    }

    /**
     * Setting up a custom content description for the question of the current view (page)
     * with respect of it's order in the current survey questions.
     */
    protected void setupQuestionAnnouncementOrder(@Nullable TextView questionTextView) {
        if (!AccessibilityUtils.isTalkbackEnabled())
            return;

        if (survey != null && survey.getQuestions().size() > 1 && question != null && questionTextView != null) {
            int totalCount = survey.getQuestions().size();
            int questionNumber = survey.getQuestions().indexOf(question) + 1;
            String contentDescription = getString(
                    R.string.ibg_surveys_question_order_content_description,
                    questionNumber, totalCount, question.getTitle());

            questionTextView.setContentDescription(contentDescription);
        }
    }

    public void focusAccessibilityOnView(@Nullable View viewToFocusOn) {
        if (viewToFocusOn != null && Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            Bundle emptyArguments = new Bundle();
            viewToFocusOn.performAccessibilityAction(
                    AccessibilityNodeInfo.ACTION_ACCESSIBILITY_FOCUS, emptyArguments
            );
            viewToFocusOn.sendAccessibilityEvent(AccessibilityEvent.TYPE_VIEW_SELECTED);
        }
    }

    @Override
    public void onClick() {
        if (survey == null) return;
        handleOpenSurveyClicked(survey, false);
    }

    @Override
    public void close() {
        if (survey == null) return;

        if (survey.isNPSSurvey() && this instanceof RateUsFragment) {
            if (getActivity() instanceof SurveyActivityCallback) {
                ((SurveyActivityCallback) getActivity()).submitSurvey(survey);
            }
        } else {
            if (getActivity() instanceof SurveyActivityCallback) {
                ((SurveyActivityCallback) getActivity()).dismissSurvey(survey);
            }
        }
    }

    private void dismissSurvey(Survey survey) {
        if (getActivity() instanceof SurveyActivityCallback) {
            ((SurveyActivityCallback) getActivity()).dismissSurvey(survey);
        }
    }

    private void submitSurvey(Survey survey) {
        if (getActivity() instanceof SurveyActivityCallback) {
            ((SurveyActivityCallback) getActivity()).submitSurvey(survey);
        }
    }

    @Override
    public void onDestroy() {
        GesturesHandler.destroy();
        super.onDestroy();
    }
}
