package com.instabug.survey.network.service;

import android.content.Context;

import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Endpoints;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.survey.Constants;
import com.instabug.survey.models.Survey;
import com.instabug.survey.network.util.SubmittingSurveysUtil;
import com.instabug.survey.settings.SurveysSettings;

import org.json.JSONException;
import org.json.JSONObject;

import io.reactivexport.observers.DisposableObserver;
import io.reactivexport.schedulers.Schedulers;

/**
 * @author mesbah
 */
public class SurveysService {
    private static SurveysService INSTANCE;
    private NetworkManager networkManager;

    private SurveysService() {
        networkManager = new NetworkManager();
    }

    /**
     * Returns the current singleton instance of this class.
     *
     * @return singleton instance of MessagingService
     */
    public static SurveysService getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new SurveysService();
        }
        return INSTANCE;
    }


    public void fetchSurveys(String locale, final Request.Callbacks<JSONObject,
            Throwable> fetchingSurveysCallbacks) throws JSONException {
        InstabugSDKLogger.d(Constants.LOG_TAG, "fetching surveys");

        // create fetchSurveys request
        Request fetchingSurveysRequest = new Request.Builder()
                .method(RequestMethod.GET)
                .endpoint(Endpoints.GET_SURVEYS)
                .addHeader(new RequestParameter<String>("Accept",
                        "application/vnd.instabug.v" + SurveysSettings.CURRENT_VERSION))
                .addHeader(new RequestParameter<String>("version",
                        SurveysSettings.CURRENT_VERSION))
                .addParameter(new RequestParameter<>(SurveysSettings.LOCALE, locale))
                .build();


        // do request with default connection timeout.
        networkManager.doRequest(IBGNetworkWorker.SURVEYS, RequestType.NORMAL, fetchingSurveysRequest, new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "Fetching surveys succeeded, Response: " + requestResponse);
                InstabugSDKLogger.d(Constants.LOG_TAG, "fetchingSurveysRequest succeeded, Response code: " + requestResponse.getResponseCode());
                if (requestResponse.getResponseCode() == RequestResponse.HttpStatusCode._2xx.OK) {
                    try {
                        if (requestResponse.getResponseBody() != null) {
                            fetchingSurveysCallbacks.onSucceeded(new JSONObject((String) requestResponse.getResponseBody()));
                        } else {
                            fetchingSurveysCallbacks.onSucceeded(new JSONObject());

                        }
                    } catch (JSONException e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "submittingSurveyRequest got JSONException: " + e.getMessage(), e);
                        fetchingSurveysCallbacks.onFailed(e);
                    }
                } else {
                    fetchingSurveysCallbacks.onFailed(new Throwable("Fetching Surveys got error with response code:" + requestResponse.getResponseCode()));
                }
            }

            @Override
            public void onFailed(Throwable throwable) {
                InstabugCore.reportError(throwable, "fetchingSurveysRequest got error: " + throwable.getMessage());
                InstabugSDKLogger.e(Constants.LOG_TAG, "fetchingSurveysRequest got error: " + throwable.getMessage(), throwable);
                fetchingSurveysCallbacks.onFailed(throwable);
            }
        });
    }

    public void submittingSurvey(final Context context, final Survey survey, final Request
            .Callbacks<Boolean, Throwable> submittingSurveyCallbacks) throws JSONException {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Start submitting survey");
        // create submittingSurvey Request
        Request.Builder requestBuilder = new Request.Builder()
                .method(RequestMethod.POST)
                .endpoint(Endpoints.SUBMIT_SURVEY.replaceAll(":survey_id", String.valueOf(survey.getId())));

        SubmittingSurveysUtil.addParamsToSubmittingSurveyRequest(context, requestBuilder, survey);
        // do request with default connection timeout.
        networkManager.doRequest(IBGNetworkWorker.SURVEYS, RequestType.NORMAL, requestBuilder.build(), new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "submittingSurveyRequest Succeeded, Response code: " + requestResponse.getResponseCode());
                if (requestResponse.getResponseCode() == RequestResponse.HttpStatusCode._2xx.OK) {
                    submittingSurveyCallbacks.onSucceeded(true);
                } else {
                    submittingSurveyCallbacks.onSucceeded(false);
                    submittingSurveyCallbacks.onFailed(new Throwable("submittingSurveyRequest got error with response code:" + requestResponse.getResponseCode()));
                }
            }

            @Override
            public void onFailed(Throwable error) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "submittingSurveyRequest got error: " + error.getMessage(), error);
                submittingSurveyCallbacks.onFailed(error);
            }
        });
    }

    public void resolveCountryInfo(final Request.Callbacks<JSONObject, Throwable> resolveIpCallback) throws JSONException {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Resolving the IP to get country information");
        // Create request
        Request resolverRequest = new Request.Builder()
                .method(RequestMethod.GET)
                .endpoint(Endpoints.RESOLVE_IP)
                .build();

        // Do request with default connection timeout
        networkManager.doRequest(IBGNetworkWorker.SURVEYS, RequestType.NORMAL, resolverRequest, new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Resolving the country info finished, Response code: " + requestResponse.getResponseCode());
                try {
                    if (requestResponse.getResponseCode() == RequestResponse.HttpStatusCode._2xx.OK) {
                        if (requestResponse.getResponseBody() != null) {
                            resolveIpCallback.onSucceeded(new JSONObject((String) requestResponse.getResponseBody()));
                        } else {
                            resolveIpCallback.onSucceeded(new JSONObject());
                        }
                    } else {
                        resolveIpCallback.onFailed(new Throwable("resolving the country info got error with response code:" + requestResponse.getResponseCode()));
                    }
                } catch (JSONException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Resolving the country info  failed, Response code: " + requestResponse.getResponseCode());
                    resolveIpCallback.onFailed(new Throwable("resolving the country info got error with response code:" + requestResponse.getResponseCode()));

                }
            }

            @Override
            public void onFailed(Throwable error) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "resolving the country info got eror: " + error.getMessage());
                resolveIpCallback.onFailed(error);
            }
        });
    }

}
