package com.instabug.survey.common.models;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

import androidx.annotation.IntDef;

import static com.instabug.survey.common.models.Frequency.Type.SHOW_EVERY_TIME;
import static com.instabug.survey.common.models.Frequency.Type.SHOW_ONCE;
import static com.instabug.survey.common.models.Frequency.Type.SHOW_WITH_INTERVAL;

/**
 * Created by Barakat on 25/11/2018
 */
public class Frequency implements Serializable {
    private static final String KEY_FREQUENCY_TYPE = "frequency_type";
    private static final String KEY_SHOWING_INTERVAL = "showing_surveys_interval";
    private static final String KEY_DISMISSED_RESHOWING_INTERVAL = "reshow_interval";
    private static final int DEFAULT_RESHOWING_INTERVAL = 30; //days
    private static final int DEFAULT_DISMISSED_RESHOWING_INTERVAL = -1;
    private @Type
    int type;
    private int showingInterval;
    private int dismissedReshowInterval;

    public Frequency() {
    }

    public static Frequency getFromJson(String jsonString) throws JSONException {
        Frequency frequency = new Frequency();
        frequency.fromJson(jsonString);
        return frequency;
    }

    public void fromJson(String modelAsJson) throws JSONException {
        JSONObject jsonObject = new JSONObject(modelAsJson);
        if (jsonObject.has(KEY_FREQUENCY_TYPE)) {
            setType(jsonObject.getInt(KEY_FREQUENCY_TYPE));
        }
        setShowingInterval(jsonObject.optInt(KEY_SHOWING_INTERVAL, DEFAULT_RESHOWING_INTERVAL));
        setDismissedReshowInterval(jsonObject
                .optInt(KEY_DISMISSED_RESHOWING_INTERVAL, DEFAULT_DISMISSED_RESHOWING_INTERVAL));
    }

    public boolean shouldShowEveryTime() {
        return getType() == SHOW_EVERY_TIME;
    }

    public boolean shouldShowOnce() {
        return getType() == SHOW_ONCE;

    }

    public boolean shouldNotReshowAfterDismiss() {
        return getDismissedReshowInterval() == DEFAULT_DISMISSED_RESHOWING_INTERVAL;
    }

    public boolean isFrequent() {
        return getType() == SHOW_WITH_INTERVAL;
    }

    public @Type
    int getType() {
        return type;
    }

    public void setType(@Type int type) {
        this.type = type;
    }

    public int getShowingInterval() {
        if (showingInterval <= 0) {
            showingInterval = DEFAULT_RESHOWING_INTERVAL;
        }
        return showingInterval;
    }

    public void setShowingInterval(int showingInterval) {
        this.showingInterval = showingInterval;
    }

    /**
     * @return dismissedReshowInterval or the showingInterval
     * if showingInterval is less than the re-show value
     * -1 if no dismissedReshowInterval
     */
    public int getDismissedReshowInterval() {
        if (dismissedReshowInterval <= 0) {
            dismissedReshowInterval = DEFAULT_DISMISSED_RESHOWING_INTERVAL;
        }
        return Math.min(dismissedReshowInterval, showingInterval);
    }

    public void setDismissedReshowInterval(int dismissedReshowInterval) {
        this.dismissedReshowInterval = dismissedReshowInterval;
    }

    public JSONObject toJson() throws JSONException {
        return new JSONObject()
                .put(KEY_FREQUENCY_TYPE, type)
                .put(KEY_SHOWING_INTERVAL, showingInterval)
                .put(KEY_DISMISSED_RESHOWING_INTERVAL, dismissedReshowInterval);
    }

    @IntDef({
            SHOW_ONCE,
            SHOW_WITH_INTERVAL,
            SHOW_EVERY_TIME
    })
    @Retention(RetentionPolicy.RUNTIME)
    public @interface Type {
        int SHOW_WITH_INTERVAL = 0;
        int SHOW_ONCE = 1;
        int SHOW_EVERY_TIME = 2;
    }
}
