package com.instabug.survey.cache;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.library.apichecker.ReturnableRunnable;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.survey.models.Survey;

import java.util.ArrayList;
import java.util.List;

/**
 * @author mesbah
 */
public class SurveysCacheManager {

    @Deprecated
    public static final String SURVEYS_DISK_CACHE_FILE_NAME = "/surveys.cache";
    @Deprecated
    public static final String SURVEYS_DISK_CACHE_KEY = "surveys_disk_cache";

    /**
     * inserting or updating a survey
     *
     * @param survey Survey that will be added to cash.
     */
    public static void addSurvey(final Survey survey) {
        PoolProvider.getSurveysDBExecutor().execute(new Runnable() {
            @Override
            public void run() {
                SurveysDbHelper.insert(survey);
            }
        });
    }

    /**
     * insert or update surveys
     *
     * @param surveys Surveys that will be added to the db.
     */
    public static void addSurveys(final List<Survey> surveys) {
        PoolProvider.getSurveysDBExecutor().execute(new Runnable() {
            @Override
            public void run() {
                for (Survey survey : surveys) {
                    addSurvey(survey);
                }
            }
        });
    }

    /**
     * @return All surveys in the db.
     */
    public static List<Survey> getSurveys() {
        List<Survey> surveys = PoolProvider.getSurveysDBExecutor().executeAndGet(new ReturnableRunnable<List<Survey>>() {
            @Override
            public List<Survey> run() {
                return SurveysDbHelper.retrieve();
            }
        });

        return surveys != null ? surveys : new ArrayList<Survey>();
    }

    /**
     * @return All surveys that triggered after 10s of session start in the db.
     */
    public static List<Survey> getTimeTriggeredSurveys() {
        List<Survey> timeTriggeredSurveys = PoolProvider.getSurveysDBExecutor().executeAndGet(new ReturnableRunnable<List<Survey>>() {
            @Override
            public List<Survey> run() {
                return SurveysDbHelper.retrieveTimeTriggeredSurveys();
            }
        });

        return timeTriggeredSurveys != null ? timeTriggeredSurveys : new ArrayList<Survey>();
    }


    /**
     * @return All surveys that triggered by a user event in the db.
     */
    public static List<Survey> getEventTriggeredSurveys() {
        List<Survey> eventTriggeredSurveys = PoolProvider.getSurveysDBExecutor().executeAndGet(new ReturnableRunnable<List<Survey>>() {
            @Override
            public List<Survey> run() {
                return SurveysDbHelper.retrieveEventTriggeredSurveys();
            }
        });

        return eventTriggeredSurveys != null ? eventTriggeredSurveys : new ArrayList<Survey>();
    }

    /**
     * @return surveys that are not answered.
     */
    public static List<Survey> getNotAnsweredSurveys() {
        List<Survey> notAnsweredSurveys = PoolProvider.getSurveysDBExecutor().executeAndGet(new ReturnableRunnable<List<Survey>>() {
            @Override
            public List<Survey> run() {
                return SurveysDbHelper.retrieveNotAnswered();
            }
        });

        return notAnsweredSurveys != null ? notAnsweredSurveys : new ArrayList<Survey>();
    }

    /**
     * @return surveys that are ready to be sent.
     */
    public static List<Survey> getReadyToSendSurveys() {
        List<Survey> readyToSendSurveys = PoolProvider.getSurveysDBExecutor().executeAndGet(new ReturnableRunnable<List<Survey>>() {
            @Override
            public List<Survey> run() {
                return SurveysDbHelper.retrieveReadyToBeSend();
            }
        });

        return readyToSendSurveys != null ? readyToSendSurveys : new ArrayList<Survey>();
    }

    /**
     * update a survey, update each field in the survey.
     *
     * @param survey the updated survey.
     */
    public static void update(final Survey survey) {
        PoolProvider.getSurveysDBExecutor().execute(new Runnable() {
            @Override
            public void run() {
                SurveysDbHelper.update(survey);
            }
        });
    }

    /**
     * update a survey's session-counter only.
     *
     * @param survey the updated survey.
     */
    @WorkerThread
    public static void updateSessions(final Survey survey) {
        PoolProvider.getSurveysDBExecutor().execute(new Runnable() {
            @Override
            public void run() {
                SurveysDbHelper.updateSessions(survey);
            }
        });
    }


    /**
     * inserting a survey if it doesn't exist Or Update its paused state or locales.
     *
     * @param survey               the remote survey that has the latest paused state and locales.
     * @param publishStatusChanged if true then update survey's status else skip it
     * @param localeChanged        if true then update survey's locales else skip it
     */
    public static void insertOrUpdatePausedOrLocale(final Survey survey,
                                                    final boolean publishStatusChanged, final boolean localeChanged) {
        PoolProvider.getSurveysDBExecutor().execute(new Runnable() {
            @Override
            public void run() {
                SurveysDbHelper.insertOrUpdatePausedOrLocales(survey, publishStatusChanged, localeChanged);
            }
        });
    }

    /**
     * update the survey target if changed
     */
    public static void updateSurveyTarget(final @NonNull Survey survey) {
        PoolProvider.getSurveysDBExecutor().execute(() -> SurveysDbHelper.updateSurveyTarget(survey));
    }


    /**
     * @param id the survey id that we need to look for in the db.
     * @return Survey object that associated with the id,
     * if it's existing in the db else returns null
     */
    @Nullable
    public static Survey getSurveyById(final long id) {
        return PoolProvider.getSurveysDBExecutor().executeAndGet(new ReturnableRunnable<Survey>() {
            @Nullable
            @Override
            public Survey run() {
                return SurveysDbHelper.retrieveById(id);
            }
        });
    }


    /**
     * @param id the survey id that we need to look for in the db.
     * @return true if the survey associated with the id is existing in the db
     */
    public static boolean isSurveyExisting(final long id) {
        Boolean surveyExists = PoolProvider.getSurveysDBExecutor().executeAndGet(new ReturnableRunnable<Boolean>() {
            @Nullable
            @Override
            public Boolean run() {
                return SurveysDbHelper.retrieveById(id) != null;
            }
        });

        if (surveyExists != null) {
            return surveyExists;
        } else {
            return false;
        }
    }

    /**
     * This method to reset all user interactions to the default after that it update them in the database
     *
     * @param surveys surveys to be reset and updated
     */
    public static void resetSurveyUserInteraction(List<Survey> surveys) {
        for (Survey survey : surveys) {
            survey.resetUserInteractions();
            survey.resetUserAnswers();
        }
        updateBulk(surveys);
    }


    public static void delete(final long id) {
        PoolProvider.getSurveysDBExecutor().execute(new Runnable() {
            @Override
            public void run() {
                SurveysDbHelper.delete(id);
            }
        });
    }

    @WorkerThread
    public static void updateBulk(final List<Survey> surveys) {
        PoolProvider.getSurveysDBExecutor().execute(new Runnable() {
            @Override
            public void run() {
                SurveysDbHelper.updateBulk(surveys);
            }
        });
    }

    public static void deleteAll() {
        PoolProvider.getSurveysDBExecutor().execute(new Runnable() {
            @Override
            public void run() {
                SurveysDbHelper.deleteAll();
            }
        });
    }
}