package com.instabug.featuresrequest.network.service;

import static com.instabug.library.networkv2.NetworkManager.isOnline;

import android.util.Log;

import com.instabug.featuresrequest.Constants;
import com.instabug.featuresrequest.models.NewComment;
import com.instabug.featuresrequest.network.AddCommentUtil;
import com.instabug.featuresrequest.settings.FeaturesRequestSettings;
import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Endpoints;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.HttpURLConnection;
import java.util.Calendar;
import java.util.Locale;

import io.reactivexport.android.plugins.RxAndroidPlugins;


/**
 * Created by mohamedzakaria on 9/5/17.
 */

public class FeaturesRequestService {

    private static final String KEY_PAGE = "page";
    private static final String KEY_COMPLETED = "completed";
    private static final String KEY_SORT_BY_TOP_VOTED = "sort_top_votes";
    private static final String KEY_MY_POSTS = "my_posts";
    private static FeaturesRequestService INSTANCE;
    private NetworkManager networkManager;

    private FeaturesRequestService() {
        networkManager = new NetworkManager();
    }

    public static FeaturesRequestService getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new FeaturesRequestService();
        }
        return INSTANCE;
    }

    public void fetchFeaturesRequests(final int page,
                                      final boolean completed,
                                      final boolean sortByTopVoted,
                                      final boolean myPosts,
                                      final Request.Callbacks<JSONObject, Throwable>
                                              requestCallbacks) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "fetch Features Requests started");

        // create fetchFeaturesRequests request
        Request.Builder fetchFeatureRequestsRequestBuilder = null;
        try {
            fetchFeatureRequestsRequestBuilder = new Request.Builder().endpoint(Endpoints.GET_FEATURES_REQUEST).method(RequestMethod.GET);

            fetchFeatureRequestsRequestBuilder.addParameter(new RequestParameter(KEY_PAGE, page));
            fetchFeatureRequestsRequestBuilder.addParameter(new RequestParameter(KEY_COMPLETED, completed));
            fetchFeatureRequestsRequestBuilder.addParameter(new RequestParameter(KEY_SORT_BY_TOP_VOTED, sortByTopVoted));
            fetchFeatureRequestsRequestBuilder.addParameter(new RequestParameter(KEY_MY_POSTS, myPosts));
        } catch (Exception e) {
            requestCallbacks.onFailed(e);
            return;
        }

        fetchFeatureRequestsRequestBuilder.addHeader(new RequestParameter("Accept",
                "application/vnd.instabug.v" + FeaturesRequestSettings.CURRENT_VERSION));
        fetchFeatureRequestsRequestBuilder.addHeader(new RequestParameter("version",
                FeaturesRequestSettings.CURRENT_VERSION));
        // do request with default connection timeout.
        networkManager.doRequest(IBGNetworkWorker.FEATURES_REQUEST, RequestType.NORMAL, fetchFeatureRequestsRequestBuilder.build(), new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "FeaturesRequests request succeeded, Response code: " + requestResponse.getResponseCode());
                InstabugSDKLogger.v(Constants.LOG_TAG, "FeaturesRequests request succeeded," +
                        "Response body: " + requestResponse.getResponseBody());
                if (requestResponse.getResponseCode() == HttpURLConnection.HTTP_OK) {
                    try {
                        if (requestResponse.getResponseBody() == null) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Request response is null");
                            return;
                        }

                        requestCallbacks.onSucceeded(new JSONObject(
                                (String) requestResponse.getResponseBody()));
                    } catch (JSONException e) {
                        InstabugCore.reportError(e, "FeaturesRequests request got error: " + e.getMessage());
                        InstabugSDKLogger.e(Constants.LOG_TAG, "FeaturesRequests request got " +
                                "JSONException: " +
                                "" + e.getMessage(), e);
                        requestCallbacks.onFailed(e);
                    }
                } else {
                    requestCallbacks.onFailed(new Throwable("Fetching " +
                            "FeaturesRequests request got error with response code:" +
                            requestResponse.getResponseCode()));
                }
            }

            @Override
            public void onFailed(Throwable error) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "FeaturesRequests request got error: ", error);
                requestCallbacks.onFailed(error);
            }
        });
    }

    public void vote(long featureId,
                     String method,
                     final Request.Callbacks<JSONObject, Throwable> voteRequestsCallback) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Voting request for feature with id : " + featureId);

        Request votingFeatureRequest = new Request.Builder()
                .endpoint(Endpoints.VOTE_FEATURE.replaceAll
                        (":feature_req_id", String.valueOf(featureId)))
                .method(method)
                .build();

        // do request with default connection timeout.
        try {

            networkManager.doRequest(IBGNetworkWorker.FEATURES_REQUEST, RequestType.NORMAL, votingFeatureRequest, new Request.Callbacks<RequestResponse, Throwable>() {
                @Override
                public void onSucceeded(RequestResponse requestResponse) {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "Voting request succeeded, Response code: " + requestResponse.getResponseCode());
                    InstabugSDKLogger.v(Constants.LOG_TAG, "Voting succeeded, Response body: " + requestResponse.getResponseBody());
                    if (requestResponse.getResponseCode() == RequestResponse.HttpStatusCode._2xx.OK) {
                        try {
                            long time = Calendar.getInstance(Locale.ENGLISH)
                                    .getTime()
                                    .getTime();
                            FeaturesRequestSettings.getInstance().setLastActivityTime(time);
                            if (requestResponse.getResponseBody() == null) {
                                InstabugSDKLogger.e(Constants.LOG_TAG, "Request response is null");
                                return;
                            }

                            voteRequestsCallback.onSucceeded(new JSONObject((String)
                                    requestResponse
                                            .getResponseBody()));
                        } catch (JSONException e) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "voting got JSONException: " + e
                                    .getMessage(), e);
                            voteRequestsCallback.onFailed(e);
                        }
                    } else {
                        voteRequestsCallback.onFailed(new Throwable("vote request got " +
                                "error " +
                                "with " +
                                "response code:" + requestResponse.getResponseCode()));
                    }
                }

                @Override
                public void onFailed(Throwable error) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "voting got error: " + error.getMessage(),
                            error);
                    voteRequestsCallback.onFailed(error);
                }
            });
        } catch (Exception e) {
            RxAndroidPlugins.reset();
            voteRequestsCallback.onFailed(e);
        }
    }


    public void search(String searchKeyword,
                       String method,
                       final Request.Callbacks<JSONArray, Throwable> searchRequestsCallback)
            throws JSONException {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Searching for feature with keyword " + searchKeyword);
        // create fetchFeaturesRequests request
        Request searchFeatureRequests = new Request.Builder()
                .endpoint(Endpoints.SEARCH.toString().concat("/" + searchKeyword))
                .method(method).addHeader(new RequestParameter("Accept",
                        "application/vnd.instabug.v" + FeaturesRequestSettings.CURRENT_VERSION))
                .addHeader(new RequestParameter("version",
                        FeaturesRequestSettings.CURRENT_VERSION))
                .build();
        // do request with default connection timeout.
        networkManager.doRequest(IBGNetworkWorker.FEATURES_REQUEST, RequestType.NORMAL, searchFeatureRequests, new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Voting request succeeded, Response code: " + requestResponse.getResponseCode());
                InstabugSDKLogger.v(Constants.LOG_TAG, "voting succeeded, Response body: " + requestResponse.getResponseBody());
                if (requestResponse.getResponseCode() == HttpURLConnection.HTTP_OK) {
                    try {
                        if (requestResponse.getResponseBody() == null) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Request response is null");
                            return;
                        }

                        searchRequestsCallback.onSucceeded(new JSONArray((String) requestResponse
                                .getResponseBody()));
                    } catch (JSONException e) {
                        InstabugCore.reportError(e, String.format("Searching for a feature with keyword %s got error: %s.", searchKeyword, e.getMessage()));

                        InstabugSDKLogger.e(Constants.LOG_TAG, "voting got JSONException: " + e.getMessage(), e);
                        searchRequestsCallback.onFailed(e);
                    }
                } else {
                    searchRequestsCallback.onFailed(new Throwable("vote request got error with " +
                            "response code:" + requestResponse.getResponseCode()));
                }
            }

            @Override
            public void onFailed(Throwable error) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "voting got error: " + error.getMessage(), error);
                searchRequestsCallback.onFailed(error);
                InstabugCore.reportError(error, String.format("Searching for a feature with keyword %s got error: %s.", searchKeyword, error.getMessage()));
            }
        });
    }

    public void getFeatureTimeLine(long featureId,
                                   final Request.Callbacks<JSONObject, Throwable>
                                           featureDetailsCallback) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Getting feature-request with id " + featureId);
        // create fetchFeaturesRequests request
        Request featureTimelineRequest = new Request.Builder()
                .endpoint(Endpoints.GET_FEATURE_TIMELINE.replaceAll
                        (":feature_req_id", String.valueOf(featureId)))
                .method(RequestMethod.GET)
                .addHeader(new RequestParameter("Accept",
                        "application/vnd.instabug.v" + FeaturesRequestSettings.CURRENT_VERSION))
                .addHeader(new RequestParameter("version",
                        FeaturesRequestSettings.CURRENT_VERSION))
                .addParameter(new RequestParameter("all", "true")).build();

        // do request with default connection timeout.
        networkManager.doRequest(IBGNetworkWorker.FEATURES_REQUEST, RequestType.NORMAL, featureTimelineRequest, new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Getting feature-request details Succeeded, Response code: " + requestResponse.getResponseCode());
                InstabugSDKLogger.v(Constants.LOG_TAG, "Getting feature-request details Succeeded, Response body: " + requestResponse.getResponseBody());
                if (requestResponse.getResponseCode() == HttpURLConnection.HTTP_OK) {
                    try {
                        if (requestResponse.getResponseBody() == null) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Request response is null");
                            return;
                        }

                        featureDetailsCallback.onSucceeded(new JSONObject((String) requestResponse
                                .getResponseBody()));
                    } catch (JSONException e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "getting feature-request details got " +
                                "JSONException: " + e.getMessage(), e);
                        featureDetailsCallback.onFailed(e);
                        InstabugCore.reportError(e, "getting feature-request details got error: " + e.getMessage());
                    }
                } else {
                    featureDetailsCallback.onFailed(new Throwable("getting feature-request " +
                            "details request got error with " +
                            "response code:" + requestResponse.getResponseCode()));
                }
            }

            @Override
            public void onFailed(Throwable error) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "getting feature-request details got error: " +
                        error.getMessage(), error);
                featureDetailsCallback.onFailed(error);
                InstabugCore.reportError(error, "getting feature-request details got error: " + error.getMessage());
            }
        });
    }

    public void addComment(final NewComment newComment,
                           final Request.Callbacks<JSONObject, Throwable> addCommenCallback) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Adding comment...");
        // create addingComment Request
        Request.Builder addingCommentRequest = new Request.Builder()
                .endpoint(Endpoints.ADD_COMMENT.replaceAll(":feature_req_id", String.valueOf
                        (newComment.getFeatureId())))
                .method(RequestMethod.POST);

        addingCommentRequest = AddCommentUtil.addParamsToAddCommentRequest(addingCommentRequest, newComment);

        addingCommentRequest.addHeader(new RequestParameter("Accept",
                "application/vnd.instabug.v" + FeaturesRequestSettings.CURRENT_VERSION));
        addingCommentRequest.addHeader(new RequestParameter("version",
                FeaturesRequestSettings.CURRENT_VERSION));
        addingCommentRequest.addParameter(new RequestParameter("all", "true"));

        Log.d("", addingCommentRequest.toString());

        if (isOnline())
        // do request with default connection timeout.
        networkManager.doRequest(IBGNetworkWorker.FEATURES_REQUEST, RequestType.NORMAL, addingCommentRequest.build(), new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "adding comment request succeeded, Response code: " + requestResponse.getResponseCode());
                InstabugSDKLogger.v(Constants.LOG_TAG, "adding comment request succeeded, Response body: " + requestResponse.getResponseBody());
                if (requestResponse.getResponseCode() == HttpURLConnection.HTTP_OK) {
                    try {
                        long time = Calendar.getInstance(Locale.ENGLISH)
                                .getTime()
                                .getTime();
                        FeaturesRequestSettings.getInstance().setLastActivityTime(time);
                        if (requestResponse.getResponseBody() == null) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Request response is null");
                            return;
                        }

                        addCommenCallback.onSucceeded(new JSONObject((String) requestResponse
                                .getResponseBody()));
                    } catch (JSONException e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "adding comment got JSONException: " + e
                                .getMessage(), e);
                        addCommenCallback.onFailed(e);
                    }
                } else {
                    addCommenCallback.onFailed(new Throwable("adding comment request got error " +
                            "with " +
                            "response code:" + requestResponse.getResponseCode()));
                }
            }

            @Override
            public void onFailed(Throwable error) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "adding comment got error: " + error.getMessage(),
                        error);
                InstabugCore.reportError(error, "Adding comment to feature request got error: " + error.getMessage());
                addCommenCallback.onFailed(error);
            }
        });
        else {
            addCommenCallback.onFailed(new IllegalStateException("No valid internet connection"));
        }
    }
}
