package com.instabug.featuresrequest.cache;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.FeatureRequestEntry.COLUMN_COLOR_CODE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.FeatureRequestEntry.COLUMN_COMMENTS_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.FeatureRequestEntry.COLUMN_CREATOR_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.FeatureRequestEntry.COLUMN_DATE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.FeatureRequestEntry.COLUMN_DESCRIPTION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.FeatureRequestEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.FeatureRequestEntry.COLUMN_IS_LIKED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.FeatureRequestEntry.COLUMN_LIKES_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.FeatureRequestEntry.COLUMN_STATUS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.FeatureRequestEntry.COLUMN_TITLE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.FeatureRequestEntry.COLUMN_VOTE_UPDATED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.FeatureRequestEntry.TABLE_NAME;

import android.content.ContentValues;
import android.database.Cursor;

import com.instabug.featuresrequest.Constants;
import com.instabug.featuresrequest.models.FeatureRequest;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.user.UserManagerWrapper;

import org.json.JSONException;

import java.util.ArrayList;
import java.util.List;

public class FeatureRequestsDBHelper {

    /**
     * Inserting a Feature-Request in the database
     *
     * @param featureRequest instance of {@link com.instabug.featuresrequest.models.FeatureRequest}
     * @return the row ID of the newly inserted row, or -1 if an error occurred
     * @throws JSONException
     */
    public static synchronized long insert(FeatureRequest featureRequest) throws JSONException {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();
        try {
            // Create a new map of values, where column names are the keys
            ContentValues values = new ContentValues();
            values.put(COLUMN_ID, featureRequest.getFeatureId());
            values.put(COLUMN_TITLE, featureRequest.getTitle());
            values.put(COLUMN_DESCRIPTION, featureRequest.getDescription());
            values.put(COLUMN_STATUS, featureRequest.getStatus().name());
            values.put(COLUMN_LIKES_COUNT, featureRequest.getLikesCount());
            values.put(COLUMN_COMMENTS_COUNT, featureRequest.getCommentsCount());
            values.put(COLUMN_IS_LIKED, featureRequest.isLiked());
            values.put(COLUMN_DATE, featureRequest.getDate());
            values.put(COLUMN_COLOR_CODE, featureRequest.getColorCode());
            values.put(COLUMN_CREATOR_NAME, featureRequest.getCreatorName());
            values.put(COLUMN_VOTE_UPDATED, featureRequest.getUserVoteStatus().name());

            // Insert the new row, returning the primary key value of the new row
            long rowId = db.insertWithOnConflictReplace(TABLE_NAME, null, values);
            db.setTransactionSuccessful();
            return rowId;
        } catch (OutOfMemoryError | Exception e) {
            IBGDiagnostics.reportNonFatal(e, "Error while inserting feature-request to DB: " + e.getMessage());
            return -1;
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * Retrieve all feature-request stored in the database
     *
     * @return a list of {@link FeatureRequest}
     * @throws JSONException
     */
    public synchronized static List<FeatureRequest> retrieve() throws JSONException {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        List<FeatureRequest> featureRequests = new ArrayList<>();
        Cursor c = null;
        try {
            c = db.query(
                    TABLE_NAME,
                    null,
                    null,
                    null,
                    null,
                    null,
                    null,
                    null
            );

            if (c != null && c.moveToFirst()) {
                do {
                    FeatureRequest featureRequest = new FeatureRequest(UserManagerWrapper.getUserName(), UserManagerWrapper.getUserEmail(), InstabugCore.getPushNotificationToken());
                    int idColIndex = c.getColumnIndex(COLUMN_ID);
                    featureRequest.setFeatureId(c.getInt(idColIndex));

                    int titleColIndex = c.getColumnIndex(COLUMN_TITLE);
                    featureRequest.setTitle(c.getString(titleColIndex));

                    int descriptionColIndex = c.getColumnIndex(COLUMN_DESCRIPTION);
                    featureRequest.setDescription(c.getString(descriptionColIndex));

                    int statusColIndex = c.getColumnIndex(COLUMN_STATUS);
                    featureRequest.setStatus(FeatureRequest.Status.valueOf(c.getString(statusColIndex)));

                    int likeCountColIndex = c.getColumnIndex(COLUMN_LIKES_COUNT);
                    featureRequest.setLikesCount(c.getInt(likeCountColIndex));

                    int commentsCountColIndex = c.getColumnIndex(COLUMN_COMMENTS_COUNT);
                    featureRequest.setCommentsCount(c.getInt(commentsCountColIndex));

                    int isLikedColIndex = c.getColumnIndex(COLUMN_IS_LIKED);
                    featureRequest.setLiked(c.getInt(isLikedColIndex) == 1);

                    int dateColIndex = c.getColumnIndex(COLUMN_DATE);
                    featureRequest.setDate(c.getLong(dateColIndex));

                    int colorCodeColIndex = c.getColumnIndex(COLUMN_COLOR_CODE);
                    featureRequest.setColorCode(c.getString(colorCodeColIndex));

                    int creatorNameColIndex = c.getColumnIndex(COLUMN_CREATOR_NAME);
                    featureRequest.setCreatorName(c.getString(creatorNameColIndex));

                    int voteUpdatedColIndex = c.getColumnIndex(COLUMN_VOTE_UPDATED);
                    featureRequest.setUserVoteStatus(
                            FeatureRequest.UserVoteStatus.valueOf(c.getString(voteUpdatedColIndex)));

                    featureRequests.add(featureRequest);
                } while (c.moveToNext());
            }
        } catch (OutOfMemoryError|Exception e) {
            IBGDiagnostics.reportNonFatal(e, "Error while getting feature requests from DB: " + e.getMessage());
            return featureRequests;
        } finally {
            if (c != null) {
                c.close();
            }
            db.close();
        }
        return featureRequests;
    }

    /**
     * Update a specific feature-request that is already in the database
     *
     * @param id {@link String} the id of the feature-request
     * @param cv {@link ContentValues} params of the values that needs to be updated
     */
    public static synchronized void update(String id, ContentValues cv) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{id};
        db.beginTransaction();
        try {
            db.update(TABLE_NAME, cv, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } catch (OutOfMemoryError|Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "Error while updating feature-request: " + e.getMessage(), Constants.LOG_TAG);
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * Delete a specific feature-request
     *
     * @param id {@link String} the id of the feature-request
     */
    public static synchronized void delete(String id) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{id};
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } catch (OutOfMemoryError|Exception e) {
            IBGDiagnostics.reportNonFatal(e, "Error while deleting feature-request: " + e.getMessage());
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * Delete all crash reports stored in the database
     * i.e. deleting the whole table
     */
    public static synchronized void deleteAll() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, null, null);
            db.setTransactionSuccessful();
        } catch (OutOfMemoryError|Exception e) {
            IBGDiagnostics.reportNonFatal(e, "Error while deleting all feature requests: " + e.getMessage());
        } finally {
            db.endTransaction();
            db.close();
        }
    }
}
