package com.instabug.crash.network

import android.content.ContentValues
import android.content.Context
import com.instabug.commons.di.CommonsLocator
import com.instabug.commons.utils.hasRateLimitedPrefix
import com.instabug.crash.Constants
import com.instabug.crash.cache.CrashReportsDbHelper
import com.instabug.crash.models.Crash
import com.instabug.crash.settings.CrashSettings
import com.instabug.crash.utils.deleteCrash
import com.instabug.library.internal.storage.cache.db.InstabugDbContract
import com.instabug.library.networkv2.RateLimitedException
import com.instabug.library.networkv2.request.DeferredCallBack
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.TimeUtils
import java.util.concurrent.Callable
import java.util.concurrent.ExecutorService
import java.util.concurrent.Future

fun interface CrashMetadataImmediateUploader {
    fun invoke(crash: Crash): Future<Runnable>
}

class CrashMetadataImmediateUploaderImpl(
    private val executorService: ExecutorService,
    private val context: Context,
    private val crashService: CrashesService,
    private val crashSettings: CrashSettings,

    ) : CrashMetadataImmediateUploader {

    override fun invoke(crash: Crash): Future<Runnable> =
        executorService.submit(Callable { upload(crash) })

    private fun upload(crash: Crash): Runnable {
        if (crashSettings.isRateLimited)
            return Runnable {
                handleRateLimit(crash)
            }
        crashSettings.setLastRequestStartedAt(TimeUtils.currentTimeMillis())
        InstabugSDKLogger.d(
            Constants.LOG_TAG,
            "Uploading crash: ${crash.id} is handled: ${crash.isHandled}"
        )
        val deferredCallable = object: DeferredCallBack<String?, Throwable>() {
            override fun deferredOnSucceeded(response: String?) =
                onUploadSucceeds(crash, response)

            override fun deferredOnFailed(error: Throwable) =
                onUploadFails(crash, error)

        }
        crashService.reportCrash(crash, deferredCallable, false)
        deferredCallable.completeIfNotCompleted()
        return deferredCallable.get()
    }

    private fun onUploadSucceeds(crash: Crash, temporaryServerToken: String?) {
        temporaryServerToken?.let { id ->
            when (hasRateLimitedPrefix(id)) {
                true -> {
                    onImmediateCrashSent(crash, id)
                    deleteCrash(context, crash)
                }
                false -> {
                    onImmediateCrashSent(crash, id)
                    crash.cacheChanges()
                }
            }
        } ?: run {
            InstabugSDKLogger.v(
                Constants.LOG_TAG,
                "Crash uploading response was null, aborting..."
            )
        }
    }

    private fun handleRateLimit(crash: Crash) {
        deleteCrash(context, crash)
        InstabugSDKLogger.d(
            Constants.LOG_TAG,
            String.format(RateLimitedException.RATE_LIMIT_REACHED, Constants.FEATURE_NAME)
        )
    }

    private fun onUploadFails(crash: Crash, error: Throwable) {
        if (error is RateLimitedException) {
            crashSettings.setLimitedUntil(error.period)
            handleRateLimit(crash)

        } else {
            InstabugSDKLogger.d(
                Constants.LOG_TAG,
                "Something went wrong while uploading crash"
            )
        }
    }

    private fun Crash.cacheChanges() {
        id.let { crashId ->
            temporaryServerToken?.let { token ->
                CrashReportsDbHelper.update(
                    crashId,
                    createUpdateContentValues(token)
                )
            }
        }
    }

    private fun createUpdateContentValues(token: String) = ContentValues().apply {
        put(
            InstabugDbContract.CrashEntry.COLUMN_TEMPORARY_SERVER_TOKEN,
            token
        )
        put(
            InstabugDbContract.CrashEntry.COLUMN_CRASH_STATE,
            Crash.CrashState.LOGS_READY_TO_BE_UPLOADED.name
        )
    }

    private fun notifyCrashMetadataCallback(crash: Crash) {
        CommonsLocator.crashMetadataCallback
            .onCrashSent(
                CommonsLocator.crashMetadataMapper.toMetadata(crash)
            )
    }

    private fun Crash.update(token: String) {
        setTemporaryServerToken(token)
        setCrashState(Crash.CrashState.LOGS_READY_TO_BE_UPLOADED)
    }

    private fun onImmediateCrashSent(crash: Crash, id: String) {
        crashSettings.setLastRequestStartedAt(0L)
        crashSettings.lastCrashTime = TimeUtils.currentTimeMillis()
        InstabugSDKLogger.d(Constants.LOG_TAG, "crash uploaded successfully")
        crash.update(id)
        notifyCrashMetadataCallback(crash)
    }

}