package com.instabug.crash

import android.content.Context
import com.instabug.commons.Detection
import com.instabug.commons.di.CommonsLocator.sessionLinker
import com.instabug.commons.diagnostics.di.DiagnosticsLocator.reporter
import com.instabug.commons.diagnostics.event.CalibrationDiagnosticEvent
import com.instabug.commons.logging.logVerbose
import com.instabug.commons.session.SessionIncident.ValidationStatus
import com.instabug.commons.threading.CrashDetailsParser
import com.instabug.commons.threading.CrashDetailsParser.ErrorParsingStrategy
import com.instabug.commons.threading.CrashDetailsParser.ThreadParsingStrategy
import com.instabug.crash.di.CrashesServiceLocator.crashDetectorListener
import com.instabug.crash.diagnostics.CrashIncidentType
import com.instabug.crash.models.Crash
import com.instabug.crash.screenrecording.ExternalAutoScreenRecordHelper
import com.instabug.crash.utils.CrashEmailSetter
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.AutoScreenRecordingEventBus
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.internal.storage.AttachmentsUtility
import com.instabug.library.internal.video.ScreenRecordingServiceAction.CustomeActions
import com.instabug.library.model.Report
import com.instabug.library.model.State
import com.instabug.library.model.StateBuilder
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.ReportHelper
import com.instabug.library.util.updateStateLogsAndAttachmentsFromReport
import com.instabug.library.util.updateStateMetadataFromReport

object ReportCreationHelper {
    @JvmStatic
    fun parseCrashDetails(thread: Thread, throwable: Throwable): CrashDetailsParser? =
        runCatching {
            CrashDetailsParser(
                threadParsingStrategy = ThreadParsingStrategy.Crashing(thread),
                errorParsingStrategy = ErrorParsingStrategy.Crashing(throwable),
                crashingThread = thread
            )
        }.onFailure { it.reportNonFatal("Error while parsing fatal crash details") }.getOrNull()

    @JvmStatic
    fun performPreReportActivities() {
        runCatching {
            CalibrationDiagnosticEvent(
                CrashIncidentType(),
                CalibrationDiagnosticEvent.Action.Captured
            ).also(reporter::report)
            crashDetectorListener.onDetection(Detection.Crash)
            SettingsManager.getInstance().isCrashedSession = true
            if (ExternalAutoScreenRecordHelper.getInstance().isEnabled
                && SettingsManager.getInstance().isAutoScreenRecordingEnabled
            ) {
                deleteAutoScreenRecording()
            }
        }.onFailure { it.reportNonFatal("Error while performing pre fatal crash report activities") }
    }

    private fun deleteAutoScreenRecording() {
        AutoScreenRecordingEventBus.getInstance().post(CustomeActions.STOP_DELETE)
    }

    @JvmStatic
    fun prepareReportState(context: Context): State? = runCatching {
        State.getState(context)
            .apply(State::updateSessionIdFromLatestSession)
            .let(CrashEmailSetter::updateStateEmailIfNeeded)
    }.onFailure { it.reportNonFatal("Error while preparing fatal crash report state") }.getOrNull()

    @JvmStatic
    fun prepareMetadataState(context: Context): State? = runCatching {
        StateBuilder(context)
            .withStateLogs(false)
            .build()
            .apply(State::updateSessionIdFromLatestSession)
            .let(CrashEmailSetter::updateStateEmailIfNeeded)
    }.onFailure { it.reportNonFatal("Error while preparing fatal crash report metadata state") }.getOrNull()

    @JvmStatic
    fun rebuildStateWithStateLogs(state: State?, context: Context) = runCatching {
        state?.let {
            StateBuilder(context)
                .withStateLogs(true)
                .rebuildStateLogs(it)
        }
    }.onFailure { it.reportNonFatal("Error while rebuilding fatal crash report state with state logs") }.getOrNull()

    @JvmStatic
    fun modifyReportStateWithUserInput(state: State?) {
        runCatching {
            state?.also { ReportHelper.update(it, getReport()) }
        }.onFailure { t -> t.reportNonFatal("Error while modifying fatal crash report state with user input") }
    }
    @JvmStatic
    fun modifyReportMetadataStateWithUserInputs(state: State?, report: Report) {
        runCatching {
            state?.also { state.updateStateMetadataFromReport(report) }
        }.onFailure { t -> t.reportNonFatal("Error while modifying fatal crash report metadata state with user input") }
    }
    @JvmStatic
    fun modifyReportStateLogsAndAttachmentsWithUserInputs(state: State?, report: Report) {
        runCatching {
            state?.also { state.updateStateLogsAndAttachmentsFromReport(report) }
        }.onFailure { t -> t.reportNonFatal("Error while modifying fatal crash report state logs and attachments with user input") }
    }

    @JvmStatic
    fun getReport(): Report =
        runCatching {
            Report().also {
                SettingsManager.getInstance().onReportCreatedListener?.onReportCreated(it)
            }
        }.getOrDefault(Report())

    @JvmStatic
    fun handleReportAttachments(crash: Crash, context: Context) {
        runCatching {
            InstabugCore.getExtraAttachmentFiles()?.forEach { (key, value) ->
                AttachmentsUtility.getNewFileAttachmentUri(context, key, value)
                    ?.also(crash::addAttachment)
            }
            AttachmentsUtility.encryptAttachments(crash.attachments)
        }.onFailure { t -> t.reportNonFatal("Error while handling fatal crash report attachments") }
    }

    @JvmStatic
    fun performPostReportActivities(crash: Crash) {
        runCatching { sessionLinker.link(crash, ValidationStatus.VALIDATED) }
            .onFailure { t -> t.reportNonFatal("Error while performing post fatal crash report activities") }
    }

    @JvmStatic
    fun updateCrash(crash: Crash, parser: CrashDetailsParser): Crash =
        crash.setCrashMessage(parser.crashDetails.toString())
            .setThreadsDetails(parser.threadsDetails?.toString())
            .setCrashState(Crash.CrashState.READY_TO_BE_SENT)
            .setHandled(false)
            .also {
                "Updating crash before persisting to disk".logVerbose()
            }

    private fun Throwable.reportNonFatal(message: String) {
        IBGDiagnostics.reportNonFatal(this, "${message}: ${this.message}")
    }
}