package com.instabug.commons.metadata

import com.instabug.anr.model.Anr
import com.instabug.commons.models.Incident
import com.instabug.commons.utils.hasRateLimitedPrefix
import com.instabug.commons.utils.toUserAttributesMap
import com.instabug.crash.models.Crash
import com.instabug.crash.models.CrashMetadata
import com.instabug.library.model.State.VALUE_APP_STATUS_FOREGROUND
import org.json.JSONObject

interface CrashMetadataMapper {
    fun toMetadata(anr: Anr): CrashMetadata
    fun toMetadata(crash: Crash): CrashMetadata
}

private const val NON_FATAL = "Non-Fatal"
private const val CRASH = "Crash"
private const val ANR_EXCEPTION_NAME = "ANRError"
private const val ANR_DEFAULT_MESSAGE = "Application Not Responding for at least 5000 ms."
private const val ERROR_KEY = "error"
private const val MESSAGE_KEY = "message"
private const val EXCEPTION_KEY = "exception"
private const val NAME_KEY = "name"

class CrashMetadataMapperImpl : CrashMetadataMapper {
    private val JSONObject.error
        get() = optJSONObject(ERROR_KEY)
    private val JSONObject.message
        get() = optString(MESSAGE_KEY)
    private val JSONObject.name
        get() = optString(NAME_KEY)
    private val JSONObject.exception
        get() = optString(EXCEPTION_KEY)

    override fun toMetadata(anr: Anr): CrashMetadata = with(anr) {
        val errorDescription = mainThreadData
            ?.let(::JSONObject)
            ?.error
            ?.exception
            .takeUnless { it.isNullOrBlank() }
            ?: ANR_DEFAULT_MESSAGE
        val type = if (state?.appStatus == VALUE_APP_STATUS_FOREGROUND) Incident.Type.ANR else Incident.Type.BG_ANR
        CrashMetadata(
            occurrenceId = temporaryServerToken,
            errorCode = ANR_EXCEPTION_NAME,
            errorType = type.metaDataType,
            errorDescription = errorDescription,
            userAttributes = state?.toUserAttributesMap(),
            rateLimited = hasRateLimitedPrefix(temporaryServerToken)
        )
    }

    override fun toMetadata(crash: Crash): CrashMetadata = with(crash) {
        val occurrenceId = temporaryServerToken.toString()
        val errorJsonObject = crashMessage?.let(::JSONObject)?.error
        val errorDescription = errorJsonObject?.message
        val errorCode = errorJsonObject?.name.orEmpty()
        CrashMetadata(
            occurrenceId = occurrenceId,
            errorType = type.metaDataType,
            errorCode = errorCode,
            errorDescription = errorDescription,
            userAttributes = state?.toUserAttributesMap(),
            rateLimited = hasRateLimitedPrefix(occurrenceId)
        )
    }

    private val Incident.Type.metaDataType
        get() = when (this) {
            Incident.Type.NonFatalCrash -> NON_FATAL
            Incident.Type.FatalCrash -> CRASH
            else -> name
        }

}