package com.instabug.bganr

import android.content.Context
import android.os.Build
import androidx.annotation.RequiresApi
import androidx.annotation.WorkerThread
import com.instabug.anr.cache.AnrReportsDbHelper
import com.instabug.anr.model.Anr
import com.instabug.anr.model.AnrMetadata
import com.instabug.anr.model.AnrState
import com.instabug.anr.model.AnrVersion
import com.instabug.bganr.BackgroundAnrCacheDir.Companion.MIGRATED_TRACE_FILE
import com.instabug.bganr.BackgroundAnrCacheDir.Companion.VALIDATED_TRACE_FILE
import com.instabug.bganr.BackgroundAnrCacheDir.Companion.searchForTraceBaselineFile
import com.instabug.bganr.BackgroundAnrCacheDir.Companion.searchForValidatedTraceFile
import com.instabug.commons.OSExitInfo
import com.instabug.commons.OSExitInfoExtractor
import com.instabug.commons.caching.CrashesCacheDir.Companion.CRASHES_SESSION_START_MARKER
import com.instabug.commons.caching.CrashesCacheDir.Companion.getSessionStarterFile
import com.instabug.commons.caching.SessionCacheDirectory
import com.instabug.commons.isAnr
import com.instabug.commons.isExplicitlyInForeground
import com.instabug.commons.logging.getOrReportError
import com.instabug.commons.logging.logVerbose
import com.instabug.commons.logging.runOrReportError
import com.instabug.commons.models.IncidentMetadata
import com.instabug.commons.safelyActOnTraceStream
import com.instabug.commons.snapshot.StateSnapshotCaptor
import com.instabug.commons.snapshot.rename
import com.instabug.commons.utils.dropReproStepsIfNeeded
import com.instabug.commons.utils.getScreenshotsDir
import com.instabug.commons.utils.modifyWithHubData
import com.instabug.library.IssueType
import com.instabug.library.SpansCacheDirectory
import java.io.File

data class MigrationResult(
    val incidents: List<Anr>,
    val migratedSessions: List<String>,
    val migratedTimeStamps: List<Long>
)

interface IBackgroundAnrMigrator {
    @WorkerThread
    operator fun invoke(ctx: Context): MigrationResult
}

sealed interface ValidationResult {
    val startTime: Long
    val infoTimeStamp: Long?

    data class Migratable(
        val directory: File,
        override val startTime: Long,
        val isBackground: Boolean,
        val description: String?,
        override val infoTimeStamp: Long?
    ) : ValidationResult

    data class Invalid(
        override val startTime: Long,
        override val infoTimeStamp: Long?
    ) : ValidationResult
}

@RequiresApi(Build.VERSION_CODES.R)
class BackgroundAnrMigrator(
    private val crashesCacheDir: SessionCacheDirectory,
    private val exitInfoExtractor: OSExitInfoExtractor,
    private val reproScreenshotsDir: SpansCacheDirectory,
    private val configurationsProvider: IBackgroundAnrConfigurationsProvider
) : IBackgroundAnrMigrator {

    @WorkerThread
    override fun invoke(ctx: Context): MigrationResult {
        val oldDirs = crashesCacheDir.oldSessionsDirectories
        return runCatching {

            var upperLine = Long.MAX_VALUE
            val migratedTimeStamps = mutableListOf<Long>()
            val migratedIncidents = oldDirs.asSequence()
                .map(this::toDirAndStartTime)
                .onEach(this::markAsMigratedIfNoStartTime)
                .filterNot { (_, startTime) -> startTime == null }
                .sortedByDescending { (_, startTime) -> startTime }
                .onEach { (directory, _) -> "ANRs-V2 -> Processing session ${directory.name}".logVerbose() }
                .map { dirAndStartTime -> validate(ctx, dirAndStartTime, upperLine) }
                .onEach { result -> upperLine = result.startTime }
                .onEach { it.infoTimeStamp?.let(migratedTimeStamps::add) }
                .filterIsInstance(ValidationResult.Migratable::class.java)
                .mapNotNull { result -> migrate(ctx, result) }
                .toList()
            MigrationResult(migratedIncidents, oldDirs.map { dir -> dir.name }, migratedTimeStamps)
        }.getOrReportError(
            default = MigrationResult(emptyList(), oldDirs.map { dir -> dir.name }, emptyList()),
            message = "Failed to migrate Background ANRs",
            withThrowable = false
        )
    }


    private fun toDirAndStartTime(sessionDir: File) = runCatching {
        getSessionStarterFile(sessionDir)?.name
            ?.removeSuffix(CRASHES_SESSION_START_MARKER)?.toLongOrNull()
            .let { startTime -> sessionDir to startTime }
    }.getOrReportError((sessionDir to null), "ANRs-V2 -> Couldn't extract session start time")

    private fun markAsMigratedIfNoStartTime(dirAndStartTime: Pair<File, Long?>) {
        runCatching {
            val (directory, startTime) = dirAndStartTime
            if (startTime != null) return
            searchForTraceBaselineFile(directory)?.rename(MIGRATED_TRACE_FILE)
            "ANRs-V2 -> Session ${directory.name} marked as migrated (no start time available)".logVerbose()
        }.runOrReportError("ANRs-V2 -> Couldn't mark timeless session as migrated")
    }

    private fun validate(
        ctx: Context,
        dirAndStartTime: Pair<File, Long?>,
        upperLine: Long
    ): ValidationResult {
        val (directory, startTime) = dirAndStartTime
        val traceBaseline = runCatching { searchForTraceBaselineFile(directory) }
            .getOrReportError(null, "ANRs-V2 -> Error while searching for baseline file")
            ?: return ValidationResult.Invalid(requireNotNull(startTime), null)
        // Return if bg_anr dir or baseline file doesn't exist
        var anrTimeStamp: Long? = null
        return runCatching {
            exitInfoExtractor.extract(ctx, requireNotNull(startTime)).infoList
                .also { infoList -> "ANRs-V2 -> Reasonable Info for session ${directory.name} $infoList".logVerbose() }
                .lastOrNull { info -> info.timestamp < upperLine }
                .also { info -> "ANRs-V2 -> Prominent info for session ${directory.name} $info".logVerbose() }
                ?.takeIf { info -> info.isAnr() }
                ?.takeIf { info -> copyTrace(traceBaseline, info) }
                ?.also { info -> anrTimeStamp = info.timestamp }
                ?.also { "ANRs-V2 -> An incident detected for session ${directory.name}".logVerbose() }
                ?.also { traceBaseline.rename(VALIDATED_TRACE_FILE) }
                ?.let { info -> toMigratable(dirAndStartTime, info, anrTimeStamp) }
                ?: run {
                    "ANRs-V2 -> No incidents detected for session ${directory.name}".logVerbose()
                    traceBaseline.rename(MIGRATED_TRACE_FILE)
                    ValidationResult.Invalid(startTime, anrTimeStamp)
                }
        }.runOrReportError("ANRs-V2 -> Couldn't validate session")
            .onFailure { traceBaseline.rename(MIGRATED_TRACE_FILE) }
            .getOrDefault(ValidationResult.Invalid(requireNotNull(startTime), anrTimeStamp))

    }

    private fun toMigratable(
        dirAndStartTime: Pair<File, Long?>,
        info: OSExitInfo,
        anrTimeStamp: Long?
    ) = ValidationResult.Migratable(
        dirAndStartTime.first,
        requireNotNull(dirAndStartTime.second),
        !info.isExplicitlyInForeground(),
        info.description,
        anrTimeStamp,
    )

    private fun copyTrace(baselineFile: File, info: OSExitInfo): Boolean =
        info.safelyActOnTraceStream { traceInput ->
            baselineFile.outputStream().use { traceOutput -> traceInput.copyTo(traceOutput) }
        }

    private fun migrate(ctx: Context, migratable: ValidationResult.Migratable): Anr? {
        val (sessionDir, _, isBackground) = migratable

        val traceFile = runCatching { searchForValidatedTraceFile(sessionDir) }
            .getOrReportError(null, "ANRs-V2 -> Error while searching for validated trace file")
            ?: return null
        if (isBackgroundEnabled(isBackground) || isForegroundEnabled(isBackground))
            return runCatching {
                val state = StateSnapshotCaptor.getStateSnapshot(sessionDir).apply {
                    modifyWithHubData()
                    dropReproStepsIfNeeded(IssueType.ANR)
                }
                val screenshotsDir = state?.getScreenshotsDir(reproScreenshotsDir, IssueType.ANR)
                val anrMetadata = AnrMetadata(
                    sessionId = sessionDir.name,
                    description = migratable.description,
                    reproScreenshotsDir = screenshotsDir,
                    isBackground = isBackground
                )
                Anr.Factory().createOSAnr(
                    ctx,
                    traceFile.inputStream(),
                    state,
                    IncidentMetadata.Factory.create(),
                    anrMetadata
                )?.also { incident -> incident.anrState = AnrState.READY_TO_BE_SENT }
                    ?.also { traceFile.rename(MIGRATED_TRACE_FILE) }
                    ?.also { it.anrVersion = AnrVersion.VERSION_2 }
                    ?.also { incident -> AnrReportsDbHelper.insert(incident) }
                    ?.also { "ANRs-V2 -> Session ${sessionDir.name} migrated".logVerbose() }
            }.onFailure { traceFile.rename(MIGRATED_TRACE_FILE) }
                .getOrReportError(null, "ANRs-V2 -> Error while creating Anr incident.")
        return null
    }

    private fun isForegroundEnabled(isBackground: Boolean) =
        !isBackground && configurationsProvider.isAnrV2Enabled


    private fun isBackgroundEnabled(isBackground: Boolean) =
        isBackground && configurationsProvider.isEnabled

}
