package com.instabug.fatalhangs.cache

import android.content.Context
import android.net.Uri
import com.instabug.commons.models.IncidentMetadata
import com.instabug.crash.Constants
import com.instabug.fatalhangs.di.FatalHangsServiceLocator
import com.instabug.fatalhangs.model.FatalHang
import com.instabug.library.core.InstabugCore
import com.instabug.library.internal.storage.DiskUtils
import com.instabug.library.internal.storage.cache.AttachmentsDbHelper
import com.instabug.library.internal.storage.cache.db.DatabaseManager
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract
import com.instabug.library.internal.storage.cache.dbv2.IBGDbManager
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg
import com.instabug.library.internal.storage.operation.DeleteUriDiskOperation
import com.instabug.library.model.State
import com.instabug.library.util.InstabugSDKLogger

class FatalHangsCacheManagerImpl : FatalHangsCacheManager {

    private fun getContentValues(fatalHang: FatalHang): IBGContentValues {
        val contentValues = IBGContentValues()

        fatalHang.id?.let {
            contentValues.put(
                IBGDbContract.FatalHangEntry.COLUMN_ID,
                fatalHang.id,
                true
            )
        }
        fatalHang.temporaryServerToken?.let {
            contentValues.put(
                IBGDbContract.FatalHangEntry.COLUMN_TEMPORARY_SERVER_TOKEN,
                it,
                true
            )
        }
        fatalHang.message?.let {
            contentValues.put(
                IBGDbContract.FatalHangEntry.COLUMN_MESSAGE,
                it,
                true
            )
        }
        fatalHang.fatalHangState.let {
            contentValues.put(
                IBGDbContract.FatalHangEntry.COLUMN_FATAL_HANG_STATE,
                it,
                true
            )
        }
        fatalHang.stateUri?.let {
            contentValues.put(
                IBGDbContract.FatalHangEntry.COLUMN_STATE,
                it.toString(),
                true
            )
        }
        fatalHang.mainThreadData?.let {
            contentValues.put(
                IBGDbContract.FatalHangEntry.COLUMN_MAIN_THREAD_DETAIL,
                it,
                true
            )
        }
        fatalHang.restOfThreadsData?.let {
            contentValues.put(
                IBGDbContract.FatalHangEntry.COLUMN_THREADS_DETAILS,
                it,
                true
            )
        }
        contentValues.put(
            IBGDbContract.FatalHangEntry.COLUMN_LAST_ACTIVITY,
            fatalHang.lastActivity,
            true
        )
        fatalHang.metadata.uuid?.let { uuid ->
            contentValues.put(IBGDbContract.FatalHangEntry.COLUMN_UUID, uuid, true)
        }

        return contentValues
    }

    override fun insert(fatalHang: FatalHang, context: Context?) {
        try {
            IBGDbManager.getInstance()
                .insert(IBGDbContract.FatalHangEntry.TABLE_NAME, null, getContentValues(fatalHang))
            // Insert the attachments
            for (attachment in fatalHang.attachments) {
                val rowId = AttachmentsDbHelper.insert(attachment, fatalHang.id)
                if (rowId != -1L) {
                    attachment.id = rowId
                }
            }
            trimToLimit(FatalHangsServiceLocator.getStoreLimit(), context)
        } catch (e: Exception) {
            InstabugCore.reportError(e, "Failed to insert Fatal-Hang")
        }
    }

    override fun update(fatalHang: FatalHang) {
        try {
            val whereArgs = ArrayList<IBGWhereArg>()
            val whereArg = IBGWhereArg(fatalHang.id, true)
            whereArgs.add(whereArg)
            IBGDbManager.getInstance()
                .update(
                    IBGDbContract.FatalHangEntry.TABLE_NAME,
                    getContentValues(fatalHang),
                    "${IBGDbContract.FatalHangEntry.COLUMN_ID} = ?",
                    whereArgs
                )
        } catch (e: Exception) {
            InstabugCore.reportError(e, "Failed to update Fatal-Hang")
        }
    }

    override fun delete(id: String) {
        try {
            val whereArgs = ArrayList<IBGWhereArg>()
            val whereArg = IBGWhereArg(id, true)
            whereArgs.add(whereArg)
            IBGDbManager.getInstance()
                .delete(
                    IBGDbContract.FatalHangEntry.TABLE_NAME,
                    "${IBGDbContract.FatalHangEntry.COLUMN_ID} = ?",
                    whereArgs
                )
        } catch (e: Exception) {
            InstabugCore.reportError(e, "Failed to delete Fatal-Hang")
        }
    }

    override fun retrieveFirst(context: Context): FatalHang? {
        try {
            val cursor = IBGDbManager.getInstance()
                .query(
                    IBGDbContract.FatalHangEntry.TABLE_NAME,
                    null,
                    null,
                    null,
                    null,
                    null,
                    null,
                    "1"
                )
            if (cursor != null) {
                if (cursor.moveToFirst()) {
                    val metadata =
                        cursor.getString(cursor.getColumnIndex(IBGDbContract.FatalHangEntry.COLUMN_UUID))
                            .let { uuid -> IncidentMetadata.Factory.create(uuid) }
                    val id =
                        cursor.getString(cursor.getColumnIndex(IBGDbContract.FatalHangEntry.COLUMN_ID))
                            ?: return null
                    val fatalHang = FatalHang(id, metadata).apply {
                        message =
                            cursor.getString(cursor.getColumnIndex(IBGDbContract.FatalHangEntry.COLUMN_MESSAGE))
                        mainThreadData =
                            cursor.getString(cursor.getColumnIndex(IBGDbContract.FatalHangEntry.COLUMN_MAIN_THREAD_DETAIL))
                        restOfThreadsData =
                            cursor.getString(cursor.getColumnIndex(IBGDbContract.FatalHangEntry.COLUMN_THREADS_DETAILS))
                        fatalHangState =
                            cursor.getInt(cursor.getColumnIndex(IBGDbContract.FatalHangEntry.COLUMN_FATAL_HANG_STATE))
                        val stateUri =
                            cursor.getString(cursor.getColumnIndex(IBGDbContract.FatalHangEntry.COLUMN_STATE))
                        temporaryServerToken =
                            cursor.getString(cursor.getColumnIndex(IBGDbContract.FatalHangEntry.COLUMN_TEMPORARY_SERVER_TOKEN))
                        lastActivity =
                            cursor.getString(cursor.getColumnIndex(IBGDbContract.FatalHangEntry.COLUMN_LAST_ACTIVITY))
                        AttachmentsDbHelper.retrieve(
                            id,
                            DatabaseManager.getInstance().openDatabase()
                        ).also(this::setAttachments)
                        stateUri?.let { readStateFile(context, it) }
                    }
                    cursor.close()
                    return fatalHang
                }
                cursor.close()
            }
        } catch (e: Exception) {
            InstabugCore.reportError(e, "Failed to retrieve Fatal-Hangs")
        }
        return null
    }

    override fun deleteAll(context: Context?) {
        trimToLimit(0, context)
    }

    private fun trimToLimit(limit: Int, context: Context?) {
        try {
            val cursor = IBGDbManager.getInstance().query(
                IBGDbContract.FatalHangEntry.TABLE_NAME,
                null,
                null,
                null,
                null,
                null,
                null
            )
            cursor?.let {
                var currentSize = cursor.count
                if (cursor.count <= limit) {
                    cursor.close()
                    return
                } else {
                    cursor.moveToFirst()
                    context?.let {
                        while (currentSize > limit) {
                            val uri =
                                cursor.getString(cursor.getColumnIndex(IBGDbContract.FatalHangEntry.COLUMN_STATE))
                            val id =
                                cursor.getString(cursor.getColumnIndex(IBGDbContract.FatalHangEntry.COLUMN_ID))
                            if (uri != null)
                                DiskUtils.with(context)
                                    .deleteOperation(DeleteUriDiskOperation(Uri.parse(uri)))
                                    .execute()
                            delete(id)
                            currentSize--
                            cursor.moveToNext()
                        }

                    }
                }
                cursor.close()
            }
        } catch (e: Exception) {
            InstabugCore.reportError(e, "Failed to trim Fatal-Hangs")
        }
    }

    private fun FatalHang.readStateFile(context: Context, stateUri: String?) = kotlin.runCatching {
        val uri = stateUri?.let(Uri::parse)
        this.stateUri = uri
        state = State.getState(context, uri)
    }.getOrElse { throwable ->
        InstabugCore.reportError(throwable, "Retrieving Fatal hang state throws OOM")
        InstabugSDKLogger.e(
            Constants.LOG_TAG,
            "Retrieving Fatal hang state throws OOM",
            throwable
        )
    }
}